<?php
/**
 * Project/Site Management System
 * Employee Recruitment Management System
 * RR Construction (Pvt) Ltd
 */

$page_title = 'Project Management';
$active_menu = 'projects';

require_once '../includes/header.php';
requirePermission(ROLE_SUPER_ADMIN);

$currentUser = getCurrentUser();
$errors = [];
$success = false;
$action = $_GET['action'] ?? 'list';
$project_id = $_GET['id'] ?? null;

// Get project managers for dropdown
$project_managers = [];
try {
    $sql = "SELECT id, full_name, username FROM users WHERE role = 'project_manager' AND status = 'active' ORDER BY full_name";
    $project_managers = fetchAll($sql);
} catch (Exception $e) {
    error_log("Error fetching project managers: " . $e->getMessage());
}

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!verifyCSRFToken($_POST['csrf_token'] ?? '')) {
        $errors[] = 'Invalid security token.';
    } else {
        $form_action = $_POST['form_action'] ?? '';
        
        if ($form_action === 'create' || $form_action === 'edit') {
            // Validate required fields
            $required = ['name', 'code', 'location', 'status'];
            
            $validation_errors = validateRequired($_POST, $required);
            
            // Validate code uniqueness for new projects or changed codes
            if (!empty($_POST['code'])) {
                $code_check_sql = "SELECT id FROM sites_projects WHERE code = ?";
                $params = [$_POST['code']];
                
                if ($form_action === 'edit' && $project_id) {
                    $code_check_sql .= " AND id != ?";
                    $params[] = $project_id;
                }
                
                $existing_project = fetchOne($code_check_sql, $params);
                if ($existing_project) {
                    $validation_errors['code'] = 'Project code already exists';
                }
            }
            
            // Validate dates
            if (!empty($_POST['start_date']) && !empty($_POST['expected_end_date'])) {
                if (strtotime($_POST['start_date']) >= strtotime($_POST['expected_end_date'])) {
                    $validation_errors['expected_end_date'] = 'End date must be after start date';
                }
            }
            
            // Validate budget
            if (!empty($_POST['budget']) && !is_numeric($_POST['budget'])) {
                $validation_errors['budget'] = 'Budget must be a valid number';
            }
            
            if (empty($validation_errors)) {
                try {
                    beginTransaction();
                    
                    $project_data = [
                        'name' => cleanInput($_POST['name']),
                        'code' => strtoupper(cleanInput($_POST['code'])),
                        'location' => cleanInput($_POST['location']),
                        'status' => $_POST['status'],
                        'project_manager_id' => !empty($_POST['project_manager_id']) ? (int)$_POST['project_manager_id'] : null,
                        'start_date' => !empty($_POST['start_date']) ? $_POST['start_date'] : null,
                        'expected_end_date' => !empty($_POST['expected_end_date']) ? $_POST['expected_end_date'] : null,
                        'budget' => !empty($_POST['budget']) ? (float)$_POST['budget'] : null,
                        'description' => cleanInput($_POST['description'] ?? '')
                    ];
                    
                    if ($form_action === 'create') {
                        $new_project_id = insertRecord('sites_projects', $project_data);
                        logActivity($currentUser['id'], 'create', 'sites_projects', $new_project_id);
                        setFlashMessage("Project '{$project_data['name']}' created successfully!", 'success');
                    } else {
                        updateRecord('sites_projects', $project_data, 'id = ?', [$project_id]);
                        logActivity($currentUser['id'], 'update', 'sites_projects', $project_id);
                        setFlashMessage("Project '{$project_data['name']}' updated successfully!", 'success');
                    }
                    
                    commit();
                    $success = true;
                    
                } catch (Exception $e) {
                    rollback();
                    error_log("Error saving project: " . $e->getMessage());
                    $errors[] = 'Failed to save project. Please try again.';
                }
            } else {
                $errors = array_merge($errors, array_values($validation_errors));
            }
        }
        
        elseif ($form_action === 'delete' && $project_id) {
            try {
                beginTransaction();
                
                // Check if project has any employee requests
                $request_count = fetchOne("SELECT COUNT(*) as count FROM employee_requests WHERE site_project IN (SELECT name FROM sites_projects WHERE id = ?)", [$project_id])['count'];
                
                if ($request_count > 0) {
                    $errors[] = 'Cannot delete project with existing employee requests. Archive the project instead.';
                } else {
                    // Get project info before deletion
                    $project_info = fetchOne("SELECT name FROM sites_projects WHERE id = ?", [$project_id]);
                    
                    // Delete project
                    deleteRecord('sites_projects', 'id = ?', [$project_id]);
                    logActivity($currentUser['id'], 'delete', 'sites_projects', $project_id);
                    
                    commit();
                    setFlashMessage("Project '{$project_info['name']}' deleted successfully!", 'success');
                }
                
            } catch (Exception $e) {
                rollback();
                error_log("Error deleting project: " . $e->getMessage());
                $errors[] = 'Failed to delete project. Please try again.';
            }
        }
        
        elseif ($form_action === 'change_status' && $project_id) {
            $new_status = $_POST['new_status'] ?? '';
            
            if (in_array($new_status, ['active', 'completed', 'on_hold', 'cancelled'])) {
                try {
                    beginTransaction();
                    
                    updateRecord('sites_projects', ['status' => $new_status], 'id = ?', [$project_id]);
                    logActivity($currentUser['id'], 'update', 'sites_projects', $project_id);
                    
                    commit();
                    setFlashMessage("Project status updated to " . ucwords(str_replace('_', ' ', $new_status)) . "!", 'success');
                    
                } catch (Exception $e) {
                    rollback();
                    error_log("Error updating project status: " . $e->getMessage());
                    $errors[] = 'Failed to update project status. Please try again.';
                }
            } else {
                $errors[] = 'Invalid status value.';
            }
        }
    }
    
    // Redirect to prevent form resubmission
    if ($success || ($form_action === 'delete' && empty($errors)) || ($form_action === 'change_status' && empty($errors))) {
        header('Location: projects.php');
        exit;
    }
}

// Get project data for editing
$edit_project = null;
if ($action === 'edit' && $project_id) {
    $edit_project = fetchOne("SELECT * FROM sites_projects WHERE id = ?", [$project_id]);
    if (!$edit_project) {
        setFlashMessage('Project not found.', 'error');
        header('Location: projects.php');
        exit;
    }
}

// Get all projects for listing
$projects = [];
$search = $_GET['search'] ?? '';
$status_filter = $_GET['status'] ?? '';

try {
    $sql = "SELECT sp.*, u.full_name as manager_name, u.username as manager_username,
                   (SELECT COUNT(*) FROM employee_requests er WHERE er.site_project = sp.name) as request_count,
                   (SELECT COUNT(*) FROM job_applications ja 
                    JOIN employee_requests er2 ON ja.request_id = er2.id 
                    WHERE er2.site_project = sp.name AND ja.status = 'hired') as hired_count
            FROM sites_projects sp 
            LEFT JOIN users u ON sp.project_manager_id = u.id 
            WHERE 1=1";
    $params = [];
    
    if (!empty($search)) {
        $sql .= " AND (sp.name LIKE ? OR sp.code LIKE ? OR sp.location LIKE ?)";
        $search_param = "%$search%";
        $params = array_merge($params, [$search_param, $search_param, $search_param]);
    }
    
    if (!empty($status_filter)) {
        $sql .= " AND sp.status = ?";
        $params[] = $status_filter;
    }
    
    $sql .= " ORDER BY sp.created_at DESC";
    $projects = fetchAll($sql, $params);
    
} catch (Exception $e) {
    error_log("Error fetching projects: " . $e->getMessage());
}

// Status definitions
$statuses = [
    'active' => ['label' => 'Active', 'class' => 'success'],
    'completed' => ['label' => 'Completed', 'class' => 'primary'],
    'on_hold' => ['label' => 'On Hold', 'class' => 'warning'],
    'cancelled' => ['label' => 'Cancelled', 'class' => 'danger']
];
?>

<div class="row mb-4">
    <div class="col-12">
        <h4 class="mb-0">Project Management</h4>
        <small class="text-muted">Manage construction projects and sites</small>
    </div>
</div>

<?php if (!empty($errors)): ?>
<div class="alert alert-danger">
    <ul class="mb-0">
        <?php foreach ($errors as $error): ?>
            <li><?php echo htmlspecialchars($error); ?></li>
        <?php endforeach; ?>
    </ul>
</div>
<?php endif; ?>

<?php if ($action === 'create' || $action === 'edit'): ?>
<!-- Project Form -->
<div class="card mb-4">
    <div class="card-header">
        <h5 class="mb-0">
            <i class="fas fa-<?php echo $action === 'create' ? 'plus' : 'edit'; ?>"></i>
            <?php echo $action === 'create' ? 'Create New Project' : 'Edit Project'; ?>
        </h5>
    </div>
    <div class="card-body">
        <form method="POST" id="projectForm">
            <input type="hidden" name="csrf_token" value="<?php echo generateCSRFToken(); ?>">
            <input type="hidden" name="form_action" value="<?php echo $action; ?>">
            
            <!-- Basic Information -->
            <div class="row mb-3">
                <div class="col-md-6">
                    <label class="form-label">Project Name *</label>
                    <input type="text" name="name" class="form-control" required 
                           value="<?php echo htmlspecialchars($edit_project['name'] ?? ''); ?>">
                </div>
                <div class="col-md-6">
                    <label class="form-label">Project Code *</label>
                    <input type="text" name="code" class="form-control" required 
                           value="<?php echo htmlspecialchars($edit_project['code'] ?? ''); ?>"
                           style="text-transform: uppercase;">
                    <small class="text-muted">Unique identifier for the project</small>
                </div>
            </div>
            
            <div class="row mb-3">
                <div class="col-md-8">
                    <label class="form-label">Location *</label>
                    <input type="text" name="location" class="form-control" required 
                           value="<?php echo htmlspecialchars($edit_project['location'] ?? ''); ?>">
                </div>
                <div class="col-md-4">
                    <label class="form-label">Status *</label>
                    <select name="status" class="form-control" required>
                        <option value="">Select status</option>
                        <?php foreach ($statuses as $status_key => $status_info): ?>
                            <option value="<?php echo $status_key; ?>" 
                                    <?php echo ($edit_project['status'] ?? '') === $status_key ? 'selected' : ''; ?>>
                                <?php echo htmlspecialchars($status_info['label']); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
            </div>
            
            <!-- Project Manager -->
            <div class="row mb-3">
                <div class="col-md-6">
                    <label class="form-label">Project Manager</label>
                    <select name="project_manager_id" class="form-control">
                        <option value="">Select project manager</option>
                        <?php foreach ($project_managers as $pm): ?>
                            <option value="<?php echo $pm['id']; ?>" 
                                    <?php echo ($edit_project['project_manager_id'] ?? '') == $pm['id'] ? 'selected' : ''; ?>>
                                <?php echo htmlspecialchars($pm['full_name'] . ' (@' . $pm['username'] . ')'); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                    <small class="text-muted">Optional - can be assigned later</small>
                </div>
                <div class="col-md-6">
                    <label class="form-label">Budget (LKR)</label>
                    <input type="number" name="budget" class="form-control" step="0.01" min="0"
                           value="<?php echo htmlspecialchars($edit_project['budget'] ?? ''); ?>">
                </div>
            </div>
            
            <!-- Timeline -->
            <div class="row mb-3">
                <div class="col-md-6">
                    <label class="form-label">Start Date</label>
                    <input type="date" name="start_date" class="form-control" 
                           value="<?php echo htmlspecialchars($edit_project['start_date'] ?? ''); ?>">
                </div>
                <div class="col-md-6">
                    <label class="form-label">Expected End Date</label>
                    <input type="date" name="expected_end_date" class="form-control" 
                           value="<?php echo htmlspecialchars($edit_project['expected_end_date'] ?? ''); ?>">
                </div>
            </div>
            
            <!-- Description -->
            <div class="mb-3">
                <label class="form-label">Description</label>
                <textarea name="description" class="form-control" rows="3" 
                          placeholder="Project description and details"><?php echo htmlspecialchars($edit_project['description'] ?? ''); ?></textarea>
            </div>
            
            <div class="text-end">
                <a href="projects.php" class="btn btn-outline-secondary">Cancel</a>
                <button type="submit" class="btn btn-primary">
                    <i class="fas fa-save"></i> 
                    <?php echo $action === 'create' ? 'Create Project' : 'Update Project'; ?>
                </button>
            </div>
        </form>
    </div>
</div>

<?php else: ?>
<!-- Project List -->
<div class="card">
    <div class="card-header d-flex justify-content-between align-items-center">
        <h5 class="mb-0">Projects & Sites</h5>
        <a href="projects.php?action=create" class="btn btn-primary btn-sm">
            <i class="fas fa-plus"></i> Add New Project
        </a>
    </div>
    
    <!-- Filters -->
    <div class="card-body border-bottom">
        <form method="GET" class="row g-3">
            <div class="col-md-6">
                <input type="text" name="search" class="form-control" placeholder="Search projects..." 
                       value="<?php echo htmlspecialchars($search); ?>">
            </div>
            <div class="col-md-4">
                <select name="status" class="form-control">
                    <option value="">All Status</option>
                    <?php foreach ($statuses as $status_key => $status_info): ?>
                        <option value="<?php echo $status_key; ?>" <?php echo $status_filter === $status_key ? 'selected' : ''; ?>>
                            <?php echo htmlspecialchars($status_info['label']); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
            <div class="col-md-2">
                <button type="submit" class="btn btn-outline-primary w-100">
                    <i class="fas fa-search"></i> Filter
                </button>
            </div>
        </form>
    </div>
    
    <div class="card-body p-0">
        <?php if (empty($projects)): ?>
            <div class="text-center py-5">
                <i class="fas fa-building fa-2x text-muted mb-3"></i>
                <h6>No projects found</h6>
                <p class="text-muted">Create your first project to get started.</p>
                <a href="projects.php?action=create" class="btn btn-primary">
                    <i class="fas fa-plus"></i> Add Project
                </a>
            </div>
        <?php else: ?>
            <div class="table-responsive">
                <table class="table table-hover mb-0">
                    <thead class="table-light">
                        <tr>
                            <th>Project</th>
                            <th>Location</th>
                            <th>Project Manager</th>
                            <th>Timeline</th>
                            <th>Budget</th>
                            <th>Activity</th>
                            <th>Status</th>
                            <th width="150">Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($projects as $project): ?>
                        <tr>
                            <td>
                                <div>
                                    <strong><?php echo htmlspecialchars($project['name']); ?></strong>
                                    <br>
                                    <small class="text-muted"><?php echo htmlspecialchars($project['code']); ?></small>
                                </div>
                            </td>
                            <td>
                                <small><?php echo htmlspecialchars($project['location']); ?></small>
                            </td>
                            <td>
                                <?php if (!empty($project['manager_name'])): ?>
                                    <div>
                                        <small><?php echo htmlspecialchars($project['manager_name']); ?></small>
                                        <br>
                                        <small class="text-muted">@<?php echo htmlspecialchars($project['manager_username']); ?></small>
                                    </div>
                                <?php else: ?>
                                    <span class="text-muted">Not assigned</span>
                                <?php endif; ?>
                            </td>
                            <td>
                                <?php if (!empty($project['start_date'])): ?>
                                    <div><small><strong>Start:</strong> <?php echo date('M j, Y', strtotime($project['start_date'])); ?></small></div>
                                <?php endif; ?>
                                <?php if (!empty($project['expected_end_date'])): ?>
                                    <div><small><strong>End:</strong> <?php echo date('M j, Y', strtotime($project['expected_end_date'])); ?></small></div>
                                <?php endif; ?>
                                <?php if (empty($project['start_date']) && empty($project['expected_end_date'])): ?>
                                    <span class="text-muted">-</span>
                                <?php endif; ?>
                            </td>
                            <td>
                                <?php if (!empty($project['budget'])): ?>
                                    <small>LKR <?php echo number_format($project['budget'], 2); ?></small>
                                <?php else: ?>
                                    <span class="text-muted">-</span>
                                <?php endif; ?>
                            </td>
                            <td>
                                <div>
                                    <small><i class="fas fa-file-alt"></i> <?php echo $project['request_count']; ?> requests</small>
                                </div>
                                <div>
                                    <small><i class="fas fa-users"></i> <?php echo $project['hired_count']; ?> hired</small>
                                </div>
                            </td>
                            <td>
                                <span class="badge bg-<?php echo $statuses[$project['status']]['class']; ?>">
                                    <?php echo htmlspecialchars($statuses[$project['status']]['label']); ?>
                                </span>
                            </td>
                            <td>
                                <div class="btn-group btn-group-sm">
                                    <a href="projects.php?action=edit&id=<?php echo $project['id']; ?>" 
                                       class="btn btn-outline-primary" title="Edit">
                                        <i class="fas fa-edit"></i>
                                    </a>
                                    
                                    <div class="btn-group btn-group-sm">
                                        <button type="button" class="btn btn-outline-warning dropdown-toggle" 
                                                data-bs-toggle="dropdown" title="Change Status">
                                            <i class="fas fa-exchange-alt"></i>
                                        </button>
                                        <ul class="dropdown-menu">
                                            <?php foreach ($statuses as $status_key => $status_info): ?>
                                                <?php if ($status_key !== $project['status']): ?>
                                                    <li>
                                                        <a class="dropdown-item" href="#" 
                                                           onclick="changeStatus(<?php echo $project['id']; ?>, '<?php echo $status_key; ?>')">
                                                            <span class="badge bg-<?php echo $status_info['class']; ?> me-2"></span>
                                                            <?php echo $status_info['label']; ?>
                                                        </a>
                                                    </li>
                                                <?php endif; ?>
                                            <?php endforeach; ?>
                                        </ul>
                                    </div>
                                    
                                    <?php if ($project['request_count'] == 0): ?>
                                        <button type="button" class="btn btn-outline-danger" 
                                                onclick="deleteProject(<?php echo $project['id']; ?>, '<?php echo htmlspecialchars($project['name'], ENT_QUOTES); ?>')" 
                                                title="Delete">
                                            <i class="fas fa-trash"></i>
                                        </button>
                                    <?php endif; ?>
                                </div>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        <?php endif; ?>
    </div>
</div>

<!-- Project Statistics -->
<?php if (!empty($projects)): ?>
<div class="row mt-4">
    <div class="col-md-3">
        <div class="card text-center">
            <div class="card-body">
                <h3 class="text-success"><?php echo count(array_filter($projects, fn($p) => $p['status'] === 'active')); ?></h3>
                <small class="text-muted">Active Projects</small>
            </div>
        </div>
    </div>
    <div class="col-md-3">
        <div class="card text-center">
            <div class="card-body">
                <h3 class="text-primary"><?php echo count(array_filter($projects, fn($p) => $p['status'] === 'completed')); ?></h3>
                <small class="text-muted">Completed Projects</small>
            </div>
        </div>
    </div>
    <div class="col-md-3">
        <div class="card text-center">
            <div class="card-body">
                <h3 class="text-info"><?php echo array_sum(array_column($projects, 'request_count')); ?></h3>
                <small class="text-muted">Total Requests</small>
            </div>
        </div>
    </div>
    <div class="col-md-3">
        <div class="card text-center">
            <div class="card-body">
                <h3 class="text-warning"><?php echo array_sum(array_column($projects, 'hired_count')); ?></h3>
                <small class="text-muted">Total Hired</small>
            </div>
        </div>
    </div>
</div>
<?php endif; ?>

<?php endif; ?>

<!-- Hidden forms for actions -->
<form id="statusForm" method="POST" style="display: none;">
    <input type="hidden" name="csrf_token" value="<?php echo generateCSRFToken(); ?>">
    <input type="hidden" name="form_action" value="change_status">
    <input type="hidden" name="new_status" id="newStatus">
</form>

<form id="deleteForm" method="POST" style="display: none;">
    <input type="hidden" name="csrf_token" value="<?php echo generateCSRFToken(); ?>">
    <input type="hidden" name="form_action" value="delete">
</form>

<script>
function changeStatus(projectId, newStatus) {
    const statusLabels = {
        'active': 'Active',
        'completed': 'Completed', 
        'on_hold': 'On Hold',
        'cancelled': 'Cancelled'
    };
    
    if (confirm(`Are you sure you want to change project status to "${statusLabels[newStatus]}"?`)) {
        document.getElementById('newStatus').value = newStatus;
        const form = document.getElementById('statusForm');
        form.action = `projects.php?id=${projectId}`;
        form.submit();
    }
}

function deleteProject(projectId, projectName) {
    if (confirm(`Are you sure you want to delete project "${projectName}"?\n\nThis action cannot be undone.`)) {
        const form = document.getElementById('deleteForm');
        form.action = `projects.php?id=${projectId}`;
        form.submit();
    }
}

// Auto-uppercase project code
document.querySelector('input[name="code"]')?.addEventListener('input', function(e) {
    e.target.value = e.target.value.toUpperCase();
});

// Date validation
document.getElementById('projectForm')?.addEventListener('submit', function(e) {
    const startDate = document.querySelector('input[name="start_date"]').value;
    const endDate = document.querySelector('input[name="expected_end_date"]').value;
    
    if (startDate && endDate && new Date(startDate) >= new Date(endDate)) {
        e.preventDefault();
        alert('Expected end date must be after start date.');
        document.querySelector('input[name="expected_end_date"]').focus();
        return false;
    }
});
</script>

<?php require_once '../includes/footer.php'; ?>