<?php
/**
 * User Management System
 * Employee Recruitment Management System
 * RR Construction (Pvt) Ltd
 */

$page_title = 'User Management';
$active_menu = 'users';

require_once '../includes/header.php';
requirePermission(ROLE_SUPER_ADMIN);

$currentUser = getCurrentUser();
$errors = [];
$success = false;
$action = $_GET['action'] ?? 'list';
$user_id = $_GET['id'] ?? null;

// Get sites/projects for dropdown
$sites_projects = [];
try {
    $sql = "SELECT id, name, code FROM sites_projects WHERE status = 'active' ORDER BY name";
    $sites_projects = fetchAll($sql);
} catch (Exception $e) {
    error_log("Error fetching sites: " . $e->getMessage());
}

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!verifyCSRFToken($_POST['csrf_token'] ?? '')) {
        $errors[] = 'Invalid security token.';
    } else {
        $form_action = $_POST['form_action'] ?? '';
        
        if ($form_action === 'create' || $form_action === 'edit') {
            // Validate required fields
            $required = ['username', 'full_name', 'role'];
            if ($form_action === 'create') {
                $required[] = 'password';
            }
            
            $validation_errors = validateRequired($_POST, $required);
            
            // Validate username uniqueness for new users or changed usernames
            if (!empty($_POST['username'])) {
                $username_check_sql = "SELECT id FROM users WHERE username = ?";
                $params = [$_POST['username']];
                
                if ($form_action === 'edit' && $user_id) {
                    $username_check_sql .= " AND id != ?";
                    $params[] = $user_id;
                }
                
                $existing_user = fetchOne($username_check_sql, $params);
                if ($existing_user) {
                    $validation_errors['username'] = 'Username already exists';
                }
            }
            
            // Validate email if provided
            if (!empty($_POST['email']) && !validateEmail($_POST['email'])) {
                $validation_errors['email'] = 'Invalid email address';
            }
            
            // Validate phone if provided
            if (!empty($_POST['phone']) && !validatePhone($_POST['phone'])) {
                $validation_errors['phone'] = 'Invalid phone number format';
            }
            
            // Validate password strength for new users
            if ($form_action === 'create' && !empty($_POST['password'])) {
                if (strlen($_POST['password']) < 6) {
                    $validation_errors['password'] = 'Password must be at least 6 characters long';
                }
            }
            
            if (empty($validation_errors)) {
                try {
                    beginTransaction();
                    
                    $user_data = [
                        'username' => cleanInput($_POST['username']),
                        'full_name' => cleanInput($_POST['full_name']),
                        'role' => $_POST['role'],
                        'site_project' => cleanInput($_POST['site_project'] ?? ''),
                        'email' => cleanInput($_POST['email'] ?? ''),
                        'phone' => cleanInput($_POST['phone'] ?? ''),
                        'status' => $_POST['status'] ?? 'active'
                    ];
                    
                    // Add password for new users or if password is being changed
                    if ($form_action === 'create' || !empty($_POST['password'])) {
                        $user_data['password'] = $_POST['password']; // Simple password storage as per requirements
                    }
                    
                    if ($form_action === 'create') {
                        $new_user_id = insertRecord('users', $user_data);
                        logActivity($currentUser['id'], 'create', 'users', $new_user_id);
                        setFlashMessage("User '{$user_data['username']}' created successfully!", 'success');
                    } else {
                        updateRecord('users', $user_data, 'id = ?', [$user_id]);
                        logActivity($currentUser['id'], 'update', 'users', $user_id);
                        setFlashMessage("User '{$user_data['username']}' updated successfully!", 'success');
                    }
                    
                    commit();
                    $success = true;
                    
                } catch (Exception $e) {
                    rollback();
                    error_log("Error saving user: " . $e->getMessage());
                    $errors[] = 'Failed to save user. Please try again.';
                }
            } else {
                $errors = array_merge($errors, array_values($validation_errors));
            }
        }
        
        elseif ($form_action === 'delete') {
            $user_id = $_POST['user_id'] ?? null;
            
            if (!$user_id) {
                $errors[] = 'Invalid user ID.';
            }
            // Prevent deleting own account
            elseif ($user_id == $currentUser['id']) {
                $errors[] = 'You cannot delete your own account.';
            } else {
                try {
                    beginTransaction();
                    
                    // Get user info before deletion
                    $user_info = fetchOne("SELECT username FROM users WHERE id = ?", [$user_id]);
                    
                    if (!$user_info) {
                        $errors[] = 'User not found.';
                    } else {
                        // Delete user
                        deleteRecord('users', 'id = ?', [$user_id]);
                        logActivity($currentUser['id'], 'delete', 'users', $user_id);
                        
                        commit();
                        setFlashMessage("User '{$user_info['username']}' deleted successfully!", 'success');
                        
                        // Redirect immediately after successful deletion
                        header('Location: users.php');
                        exit;
                    }
                    
                } catch (Exception $e) {
                    rollback();
                    error_log("Error deleting user: " . $e->getMessage());
                    $errors[] = 'Failed to delete user. Please try again.';
                }
            }
        }
        
        elseif ($form_action === 'toggle_status') {
            $user_id = $_POST['user_id'] ?? null;
            
            if (!$user_id) {
                $errors[] = 'Invalid user ID.';
            }
            // Prevent deactivating own account
            elseif ($user_id == $currentUser['id']) {
                $errors[] = 'You cannot deactivate your own account.';
            } else {
                try {
                    beginTransaction();
                    
                    $current_status = fetchOne("SELECT status FROM users WHERE id = ?", [$user_id])['status'];
                    $new_status = ($current_status === 'active') ? 'inactive' : 'active';
                    
                    updateRecord('users', ['status' => $new_status], 'id = ?', [$user_id]);
                    logActivity($currentUser['id'], 'update', 'users', $user_id);
                    
                    commit();
                    $status_text = ($new_status === 'active') ? 'activated' : 'deactivated';
                    setFlashMessage("User {$status_text} successfully!", 'success');
                    
                    // Redirect immediately after successful status change
                    header('Location: users.php');
                    exit;
                    
                } catch (Exception $e) {
                    rollback();
                    error_log("Error updating user status: " . $e->getMessage());
                    $errors[] = 'Failed to update user status. Please try again.';
                }
            }
        }
    }
    
    // Redirect to prevent form resubmission (only for create/edit)
    if ($success) {
        header('Location: users.php');
        exit;
    }
}

// Get user data for editing
$edit_user = null;
if ($action === 'edit' && $user_id) {
    $edit_user = fetchOne("SELECT * FROM users WHERE id = ?", [$user_id]);
    if (!$edit_user) {
        setFlashMessage('User not found.', 'error');
        header('Location: users.php');
        exit;
    }
}

// Get all users for listing
$users = [];
$search = $_GET['search'] ?? '';
$role_filter = $_GET['role'] ?? '';
$status_filter = $_GET['status'] ?? '';

try {
    $sql = "SELECT u.*, sp.name as project_name 
            FROM users u 
            LEFT JOIN sites_projects sp ON u.site_project = sp.name 
            WHERE 1=1";
    $params = [];
    
    if (!empty($search)) {
        $sql .= " AND (u.username LIKE ? OR u.full_name LIKE ? OR u.email LIKE ?)";
        $search_param = "%$search%";
        $params = array_merge($params, [$search_param, $search_param, $search_param]);
    }
    
    if (!empty($role_filter)) {
        $sql .= " AND u.role = ?";
        $params[] = $role_filter;
    }
    
    if (!empty($status_filter)) {
        $sql .= " AND u.status = ?";
        $params[] = $status_filter;
    }
    
    $sql .= " ORDER BY u.created_at DESC";
    $users = fetchAll($sql, $params);
    
} catch (Exception $e) {
    error_log("Error fetching users: " . $e->getMessage());
}

// Role definitions
$roles = [
    'super_admin' => 'Super Admin',
    'site_admin' => 'Site Admin Officer',
    'project_manager' => 'Project Manager',
    'general_manager' => 'General Manager',
    'hr_manager' => 'HR Manager',
    'accountant' => 'Accountant'
];
?>

<div class="row mb-4">
    <div class="col-12">
        <h4 class="mb-0">User Management</h4>
        <small class="text-muted">Manage system users and permissions</small>
    </div>
</div>

<?php if (!empty($errors)): ?>
<div class="alert alert-danger">
    <ul class="mb-0">
        <?php foreach ($errors as $error): ?>
            <li><?php echo htmlspecialchars($error); ?></li>
        <?php endforeach; ?>
    </ul>
</div>
<?php endif; ?>

<?php if ($action === 'create' || $action === 'edit'): ?>
<!-- User Form -->
<div class="card mb-4">
    <div class="card-header">
        <h5 class="mb-0">
            <i class="fas fa-<?php echo $action === 'create' ? 'plus' : 'edit'; ?>"></i>
            <?php echo $action === 'create' ? 'Create New User' : 'Edit User'; ?>
        </h5>
    </div>
    <div class="card-body">
        <form method="POST" id="userForm">
            <input type="hidden" name="csrf_token" value="<?php echo generateCSRFToken(); ?>">
            <input type="hidden" name="form_action" value="<?php echo $action; ?>">
            
            <div class="row mb-3">
                <div class="col-md-6">
                    <label class="form-label">Username *</label>
                    <input type="text" name="username" class="form-control" required 
                           value="<?php echo htmlspecialchars($edit_user['username'] ?? ''); ?>">
                    <small class="text-muted">Must be unique across the system</small>
                </div>
                <div class="col-md-6">
                    <label class="form-label">Full Name *</label>
                    <input type="text" name="full_name" class="form-control" required 
                           value="<?php echo htmlspecialchars($edit_user['full_name'] ?? ''); ?>">
                </div>
            </div>
            
            <div class="row mb-3">
                <div class="col-md-6">
                    <label class="form-label">Role *</label>
                    <select name="role" class="form-control" required onchange="toggleSiteProject()">
                        <option value="">Select role</option>
                        <?php foreach ($roles as $role_key => $role_name): ?>
                            <option value="<?php echo $role_key; ?>" 
                                    <?php echo ($edit_user['role'] ?? '') === $role_key ? 'selected' : ''; ?>>
                                <?php echo htmlspecialchars($role_name); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="col-md-6" id="siteProjectField">
                    <label class="form-label">Site/Project</label>
                    <select name="site_project" class="form-control">
                        <option value="">Select site/project (if applicable)</option>
                        <?php foreach ($sites_projects as $site): ?>
                            <option value="<?php echo htmlspecialchars($site['name']); ?>" 
                                    <?php echo ($edit_user['site_project'] ?? '') === $site['name'] ? 'selected' : ''; ?>>
                                <?php echo htmlspecialchars($site['name']); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                    <small class="text-muted">Required for Site Admins and Project Managers</small>
                </div>
            </div>
            
            <div class="row mb-3">
                <div class="col-md-6">
                    <label class="form-label">Email</label>
                    <input type="email" name="email" class="form-control" 
                           value="<?php echo htmlspecialchars($edit_user['email'] ?? ''); ?>">
                </div>
                <div class="col-md-6">
                    <label class="form-label">Phone</label>
                    <input type="tel" name="phone" class="form-control" 
                           value="<?php echo htmlspecialchars($edit_user['phone'] ?? ''); ?>">
                </div>
            </div>
            
            <div class="row mb-3">
                <div class="col-md-6">
                    <label class="form-label">
                        Password <?php echo $action === 'create' ? '*' : ''; ?>
                    </label>
                    <input type="password" name="password" class="form-control" 
                           <?php echo $action === 'create' ? 'required' : ''; ?>>
                    <?php if ($action === 'edit'): ?>
                        <small class="text-muted">Leave blank to keep current password</small>
                    <?php endif; ?>
                </div>
                <div class="col-md-6">
                    <label class="form-label">Status</label>
                    <select name="status" class="form-control">
                        <option value="active" <?php echo ($edit_user['status'] ?? 'active') === 'active' ? 'selected' : ''; ?>>
                            Active
                        </option>
                        <option value="inactive" <?php echo ($edit_user['status'] ?? '') === 'inactive' ? 'selected' : ''; ?>>
                            Inactive
                        </option>
                    </select>
                </div>
            </div>
            
            <div class="text-end">
                <a href="users.php" class="btn btn-outline-secondary">Cancel</a>
                <button type="submit" class="btn btn-primary">
                    <i class="fas fa-save"></i> 
                    <?php echo $action === 'create' ? 'Create User' : 'Update User'; ?>
                </button>
            </div>
        </form>
    </div>
</div>

<?php else: ?>
<!-- User List -->
<div class="card">
    <div class="card-header d-flex justify-content-between align-items-center">
        <h5 class="mb-0">System Users</h5>
        <a href="users.php?action=create" class="btn btn-primary btn-sm">
            <i class="fas fa-plus"></i> Add New User
        </a>
    </div>
    
    <!-- Filters -->
    <div class="card-body border-bottom">
        <form method="GET" class="row g-3">
            <div class="col-md-4">
                <input type="text" name="search" class="form-control" placeholder="Search users..." 
                       value="<?php echo htmlspecialchars($search); ?>">
            </div>
            <div class="col-md-3">
                <select name="role" class="form-control">
                    <option value="">All Roles</option>
                    <?php foreach ($roles as $role_key => $role_name): ?>
                        <option value="<?php echo $role_key; ?>" <?php echo $role_filter === $role_key ? 'selected' : ''; ?>>
                            <?php echo htmlspecialchars($role_name); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
            <div class="col-md-3">
                <select name="status" class="form-control">
                    <option value="">All Status</option>
                    <option value="active" <?php echo $status_filter === 'active' ? 'selected' : ''; ?>>Active</option>
                    <option value="inactive" <?php echo $status_filter === 'inactive' ? 'selected' : ''; ?>>Inactive</option>
                </select>
            </div>
            <div class="col-md-2">
                <button type="submit" class="btn btn-outline-primary w-100">
                    <i class="fas fa-search"></i> Filter
                </button>
            </div>
        </form>
    </div>
    
    <div class="card-body p-0">
        <?php if (empty($users)): ?>
            <div class="text-center py-5">
                <i class="fas fa-users fa-2x text-muted mb-3"></i>
                <h6>No users found</h6>
                <p class="text-muted">Create your first user to get started.</p>
                <a href="users.php?action=create" class="btn btn-primary">
                    <i class="fas fa-plus"></i> Add User
                </a>
            </div>
        <?php else: ?>
            <div class="table-responsive">
                <table class="table table-hover mb-0">
                    <thead class="table-light">
                        <tr>
                            <th>User</th>
                            <th>Role</th>
                            <th>Site/Project</th>
                            <th>Contact</th>
                            <th>Status</th>
                            <th>Created</th>
                            <th width="150">Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($users as $user): ?>
                        <tr>
                            <td>
                                <div>
                                    <strong><?php echo htmlspecialchars($user['full_name']); ?></strong>
                                    <br>
                                    <small class="text-muted">@<?php echo htmlspecialchars($user['username']); ?></small>
                                </div>
                            </td>
                            <td>
                                <span class="badge bg-secondary">
                                    <?php echo htmlspecialchars($roles[$user['role']] ?? $user['role']); ?>
                                </span>
                            </td>
                            <td>
                                <?php if (!empty($user['site_project'])): ?>
                                    <small><?php echo htmlspecialchars($user['site_project']); ?></small>
                                <?php else: ?>
                                    <span class="text-muted">-</span>
                                <?php endif; ?>
                            </td>
                            <td>
                                <?php if (!empty($user['email'])): ?>
                                    <div><small><?php echo htmlspecialchars($user['email']); ?></small></div>
                                <?php endif; ?>
                                <?php if (!empty($user['phone'])): ?>
                                    <div><small><?php echo htmlspecialchars($user['phone']); ?></small></div>
                                <?php endif; ?>
                                <?php if (empty($user['email']) && empty($user['phone'])): ?>
                                    <span class="text-muted">-</span>
                                <?php endif; ?>
                            </td>
                            <td>
                                <?php if ($user['status'] === 'active'): ?>
                                    <span class="badge bg-success">Active</span>
                                <?php else: ?>
                                    <span class="badge bg-danger">Inactive</span>
                                <?php endif; ?>
                            </td>
                            <td>
                                <small><?php echo date('M j, Y', strtotime($user['created_at'])); ?></small>
                            </td>
                            <td>
                                <div class="btn-group btn-group-sm">
                                    <a href="users.php?action=edit&id=<?php echo $user['id']; ?>" 
                                       class="btn btn-outline-primary" title="Edit">
                                        <i class="fas fa-edit"></i>
                                    </a>
                                    
                                    <?php if ($user['id'] != $currentUser['id']): ?>
                                        <button type="button" class="btn btn-outline-warning" 
                                                onclick="toggleStatus(<?php echo $user['id']; ?>, '<?php echo $user['status']; ?>')" 
                                                title="<?php echo $user['status'] === 'active' ? 'Deactivate' : 'Activate'; ?>">
                                            <i class="fas fa-<?php echo $user['status'] === 'active' ? 'ban' : 'check'; ?>"></i>
                                        </button>
                                        
                                        <button type="button" class="btn btn-outline-danger" 
                                                onclick="deleteUser(<?php echo $user['id']; ?>, '<?php echo htmlspecialchars($user['username'], ENT_QUOTES); ?>')" 
                                                title="Delete">
                                            <i class="fas fa-trash"></i>
                                        </button>
                                    <?php endif; ?>
                                </div>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        <?php endif; ?>
    </div>
</div>
<?php endif; ?>

<!-- Hidden forms for actions -->
<form id="statusForm" method="POST" style="display: none;">
    <input type="hidden" name="csrf_token" value="<?php echo generateCSRFToken(); ?>">
    <input type="hidden" name="form_action" value="toggle_status">
    <input type="hidden" name="user_id" id="statusUserId">
</form>

<form id="deleteForm" method="POST" style="display: none;">
    <input type="hidden" name="csrf_token" value="<?php echo generateCSRFToken(); ?>">
    <input type="hidden" name="form_action" value="delete">
    <input type="hidden" name="user_id" id="deleteUserId">
</form>

<script>
function toggleSiteProject() {
    const roleSelect = document.querySelector('select[name="role"]');
    const siteProjectField = document.getElementById('siteProjectField');
    const siteProjectSelect = document.querySelector('select[name="site_project"]');
    
    const rolesRequiringSite = ['site_admin', 'project_manager'];
    
    if (rolesRequiringSite.includes(roleSelect.value)) {
        siteProjectField.style.display = 'block';
        siteProjectSelect.required = true;
    } else {
        siteProjectField.style.display = 'none';
        siteProjectSelect.required = false;
        siteProjectSelect.value = '';
    }
}

function toggleStatus(userId, currentStatus) {
    const action = currentStatus === 'active' ? 'deactivate' : 'activate';
    
    if (confirm(`Are you sure you want to ${action} this user?`)) {
        document.getElementById('statusUserId').value = userId;
        document.getElementById('statusForm').submit();
    }
}

function deleteUser(userId, username) {
    if (confirm(`Are you sure you want to delete user "${username}"?\n\nThis action cannot be undone.`)) {
        document.getElementById('deleteUserId').value = userId;
        document.getElementById('deleteForm').submit();
    }
}

// Initialize site/project field visibility on page load
document.addEventListener('DOMContentLoaded', function() {
    toggleSiteProject();
});

// Form validation
document.getElementById('userForm')?.addEventListener('submit', function(e) {
    const role = document.querySelector('select[name="role"]').value;
    const siteProject = document.querySelector('select[name="site_project"]').value;
    
    const rolesRequiringSite = ['site_admin', 'project_manager'];
    
    if (rolesRequiringSite.includes(role) && !siteProject) {
        e.preventDefault();
        alert('Site/Project is required for Site Admins and Project Managers.');
        document.querySelector('select[name="site_project"]').focus();
        return false;
    }
});
</script>

<?php require_once '../includes/footer.php'; ?>