<?php
/**
 * Login Page
 * Employee Recruitment Management System
 * RR Construction (Pvt) Ltd
 */

// Define access constant
define('HRM_ACCESS', true);

// Start session
session_start();

// Include required files
require_once '../config/constants.php';
require_once '../config/database.php';
require_once '../includes/functions.php';

// Redirect if already logged in
if (isLoggedIn()) {
    $user = getCurrentUser();
    header('Location: ' . getDashboardUrl($user['role']));
    exit;
}

// Initialize variables
$username = '';
$error_message = '';
$success_message = '';

// Check for session timeout message
if (isset($_GET['timeout']) && $_GET['timeout'] == '1') {
    $error_message = 'Your session has expired. Please log in again.';
}

// Check for logout success message
if (isset($_GET['logout']) && $_GET['logout'] == 'success') {
    $success_message = 'You have been successfully logged out.';
}

// Process login form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Verify CSRF token
    if (!isset($_POST['csrf_token']) || !verifyCSRFToken($_POST['csrf_token'])) {
        $error_message = 'Invalid security token. Please try again.';
    } else {
        // Get and sanitize form data
        $username = cleanInput($_POST['username'] ?? '');
        $password = $_POST['password'] ?? '';
        $remember_me = isset($_POST['remember_me']);
        
        // Validate input
        if (empty($username) || empty($password)) {
            $error_message = 'Please enter both username and password.';
        } else {
            // Attempt login
            if (loginUser($username, $password)) {
                // Set remember me cookie if requested
                if ($remember_me) {
                    $token = generateRandomString(32);
                    setcookie('remember_token', $token, time() + REMEMBER_ME_DURATION, BASE_URL, '', false, true);
                    
                    // Store token in database (you may want to create a separate table for this)
                    try {
                        $sql = "UPDATE users SET remember_token = ? WHERE username = ?";
                        executeQuery($sql, [$token, $username]);
                    } catch (Exception $e) {
                        error_log("Remember token error: " . $e->getMessage());
                    }
                }
                
                // Get user and redirect to dashboard
                $user = getCurrentUser();
                $redirectUrl = getDashboardUrl($user['role']);
                
                // Check if there's a return URL
                if (isset($_SESSION['return_url'])) {
                    $redirectUrl = $_SESSION['return_url'];
                    unset($_SESSION['return_url']);
                }
                
                header('Location: ' . $redirectUrl);
                exit;
            } else {
                $error_message = 'Invalid username or password. Please try again.';
                
                // Log failed login attempt
                logSystemError("Failed login attempt for username: $username", [
                    'ip' => getClientIP(),
                    'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? 'Unknown'
                ]);
            }
        }
    }
}

// Check for remember me cookie
if (isset($_COOKIE['remember_token']) && !empty($_COOKIE['remember_token'])) {
    try {
        $sql = "SELECT * FROM users WHERE remember_token = ? AND status = 'active'";
        $user = fetchOne($sql, [$_COOKIE['remember_token']]);
        
        if ($user) {
            // Auto-login user
            $_SESSION['user_id'] = $user['id'];
            $_SESSION['username'] = $user['username'];
            $_SESSION['role'] = $user['role'];
            $_SESSION['full_name'] = $user['full_name'];
            $_SESSION['login_time'] = time();
            
            // Log auto-login
            logActivity($user['id'], 'auto_login', 'users', $user['id']);
            
            // Redirect to dashboard
            header('Location: ' . getDashboardUrl($user['role']));
            exit;
        } else {
            // Invalid token, clear cookie
            setcookie('remember_token', '', time() - 3600, BASE_URL);
        }
    } catch (Exception $e) {
        error_log("Remember me error: " . $e->getMessage());
    }
}

// Page title
$page_title = 'Login';
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $page_title; ?> - <?php echo APP_NAME; ?></title>
    
    <!-- Favicon -->
    <link rel="icon" type="image/x-icon" href="<?php echo ASSETS_URL; ?>images/favicon.ico">
    
    <!-- Bootstrap CSS -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    
    <!-- Font Awesome -->
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css" rel="stylesheet">
    
    <!-- Custom CSS -->
    <style>
        :root {
            --primary-color: <?php echo PRIMARY_COLOR; ?>;
            --secondary-color: <?php echo SECONDARY_COLOR; ?>;
        }
        
        body {
            font-family: 'Arial', sans-serif;
            background: white;
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 20px 0;
        }
        
        .login-container {
            background: white;
            border: 1px solid #e5e7eb;
            border-radius: 8px;
            box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
            overflow: hidden;
            width: 100%;
            max-width: 400px;
            min-height: auto;
        }
        
        .login-header {
            background: var(--primary-color);
            color: white;
            padding: 30px 30px 20px 30px;
            text-align: center;
        }
        
        .company-logo {
            width: 60px;
            height: 60px;
            background: rgba(255,255,255,0.2);
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            margin: 0 auto 15px;
        }
        
        .company-logo i {
            font-size: 1.8rem;
        }
        
        .login-header h1 {
            font-size: 1.5rem;
            font-weight: 600;
            margin-bottom: 5px;
        }
        
        .login-header p {
            font-size: 0.9rem;
            opacity: 0.9;
            margin: 0;
        }
        
        .login-form {
            padding: 30px;
        }
        
        .login-form h2 {
            color: var(--secondary-color);
            margin-bottom: 20px;
            font-weight: 600;
            font-size: 1.3rem;
            text-align: center;
        }
        
        .form-group {
            margin-bottom: 20px;
        }
        
        .form-label {
            font-weight: 500;
            color: var(--secondary-color);
            margin-bottom: 5px;
            display: block;
            font-size: 0.9rem;
        }
        
        .form-control {
            border: 1px solid #d1d5db;
            border-radius: 6px;
            padding: 12px 40px 12px 12px;
            font-size: 14px;
            transition: border-color 0.3s ease;
            background: white;
            width: 100%;
        }
        
        .form-control:focus {
            border-color: var(--primary-color);
            box-shadow: 0 0 0 0.1rem rgba(220, 38, 38, 0.25);
            outline: none;
        }
        
        .input-group-text {
            position: absolute;
            right: 12px;
            top: 50%;
            transform: translateY(-50%);
            background: none;
            border: none;
            color: #9ca3af;
            cursor: pointer;
            z-index: 10;
        }
        
        .btn-primary {
            background: var(--primary-color);
            border: none;
            border-radius: 6px;
            padding: 12px 20px;
            font-weight: 500;
            font-size: 14px;
            transition: background-color 0.3s ease;
            width: 100%;
        }
        
        .btn-primary:hover {
            background-color: #b91c1c;
        }
        
        .form-check {
            margin: 15px 0;
        }
        
        .form-check-input:checked {
            background-color: var(--primary-color);
            border-color: var(--primary-color);
        }
        
        .form-check-label {
            color: var(--secondary-color);
            font-size: 0.9rem;
        }
        
        .alert {
            border: none;
            border-radius: 6px;
            padding: 12px 15px;
            margin-bottom: 20px;
            font-size: 0.9rem;
        }
        
        .alert-danger {
            background-color: #fef2f2;
            color: #dc2626;
            border-left: 3px solid #dc2626;
        }
        
        .alert-success {
            background-color: #f0fdf4;
            color: #16a34a;
            border-left: 3px solid #16a34a;
        }
        
        .loading-spinner {
            display: none;
            width: 16px;
            height: 16px;
            border: 2px solid #ffffff;
            border-top: 2px solid transparent;
            border-radius: 50%;
            animation: spin 1s linear infinite;
            margin-right: 8px;
        }
        
        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
        
        .additional-info {
            text-align: center;
            margin-top: 20px;
            padding-top: 20px;
            border-top: 1px solid #e5e7eb;
        }
        
        .additional-info small {
            color: #6b7280;
            font-size: 0.8rem;
        }
        
        .demo-credentials {
            margin-top: 20px;
            padding: 15px;
            background: #f8fafc;
            border: 1px solid #e2e8f0;
            border-radius: 6px;
        }
        
        .demo-credentials h6 {
            color: var(--secondary-color);
            margin-bottom: 10px;
            font-size: 0.9rem;
        }
        
        .demo-credentials .row {
            margin-bottom: 8px;
        }
        
        .demo-credentials small {
            font-size: 0.8rem;
        }
        
        /* Responsive Design */
        @media (max-width: 768px) {
            .login-container {
                margin: 20px;
                max-width: none;
            }
            
            .login-header {
                padding: 25px 20px 15px 20px;
            }
            
            .login-header h1 {
                font-size: 1.3rem;
            }
            
            .login-form {
                padding: 25px 20px;
            }
        }
        
        /* Simple fade animation */
        .fade-in {
            animation: fadeIn 0.5s ease-in;
        }
        
        @keyframes fadeIn {
            from { opacity: 0; }
            to { opacity: 1; }
        }
    </style>
</head>
<body>
    <div class="login-container fade-in">
        <!-- Header -->
        <div class="login-header">
            <div class="company-logo">
                <i class="fas fa-building"></i>
            </div>
            <h1><?php echo COMPANY_NAME; ?></h1>
            <p><?php echo APP_NAME; ?></p>
        </div>
        
        <!-- Login Form -->
        <div class="login-form">
            <h2>Sign In</h2>
            
            <!-- Error/Success Messages -->
            <?php if (!empty($error_message)): ?>
                <div class="alert alert-danger" role="alert">
                    <i class="fas fa-exclamation-circle me-2"></i>
                    <?php echo htmlspecialchars($error_message); ?>
                </div>
            <?php endif; ?>
            
            <?php if (!empty($success_message)): ?>
                <div class="alert alert-success" role="alert">
                    <i class="fas fa-check-circle me-2"></i>
                    <?php echo htmlspecialchars($success_message); ?>
                </div>
            <?php endif; ?>
            
            <!-- Login Form -->
            <form method="POST" action="" id="loginForm" novalidate>
                <!-- CSRF Token -->
                <input type="hidden" name="csrf_token" value="<?php echo generateCSRFToken(); ?>">
                
                <!-- Username Field -->
                <div class="form-group">
                    <label for="username" class="form-label">Username</label>
                    <div class="position-relative">
                        <input type="text" 
                               class="form-control" 
                               id="username" 
                               name="username" 
                               value="<?php echo htmlspecialchars($username); ?>"
                               placeholder="Enter your username"
                               required
                               autocomplete="username">
                        <span class="input-group-text">
                            <i class="fas fa-user"></i>
                        </span>
                    </div>
                </div>
                
                <!-- Password Field -->
                <div class="form-group">
                    <label for="password" class="form-label">Password</label>
                    <div class="position-relative">
                        <input type="password" 
                               class="form-control" 
                               id="password" 
                               name="password" 
                               placeholder="Enter your password"
                               required
                               autocomplete="current-password">
                        <span class="input-group-text" onclick="togglePassword()">
                            <i class="fas fa-eye" id="passwordToggle"></i>
                        </span>
                    </div>
                </div>
                
                <!-- Remember Me -->
                <div class="form-check">
                    <input class="form-check-input" type="checkbox" id="remember_me" name="remember_me">
                    <label class="form-check-label" for="remember_me">
                        Remember me for 30 days
                    </label>
                </div>
                
                <!-- Submit Button -->
                <button type="submit" class="btn btn-primary" id="loginBtn">
                    <span class="loading-spinner" id="loginSpinner"></span>
                    <i class="fas fa-sign-in-alt me-2"></i>
                    Sign In
                </button>
            </form>
            
            <!-- Additional Info -->
            <div class="additional-info">
                <small class="text-muted">
                    Having trouble? Contact your system administrator.<br>
                    <i class="fas fa-phone me-1"></i><?php echo COMPANY_PHONE; ?>
                    <span class="mx-2">|</span>
                    <i class="fas fa-envelope me-1"></i><?php echo COMPANY_EMAIL; ?>
                </small>
            </div>
            
            <!-- Demo Credentials (Remove in production) -->
            <div class="demo-credentials">
                <h6>
                    <i class="fas fa-info-circle me-2"></i>Demo Credentials
                </h6>
                <div class="row">
                    <div class="col-6">
                        <small class="text-muted d-block">Super Admin:</small>
                        <small><strong>admin / admin123</strong></small>
                    </div>
                    <div class="col-6">
                        <small class="text-muted d-block">Site Admin:</small>
                        <small><strong>siteadmin_cmp1 / site123</strong></small>
                    </div>
                </div>
                <div class="row mt-2">
                    <div class="col-6">
                        <small class="text-muted d-block">Project Manager:</small>
                        <small><strong>pm_metro / pm123</strong></small>
                    </div>
                    <div class="col-6">
                        <small class="text-muted d-block">HR Manager:</small>
                        <small><strong>hr_manager / hr123</strong></small>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Scripts -->
    <script src="https://cdn.jsdelivr.net/npm/jquery@3.7.0/dist/jquery.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    
    <script>
        // Toggle password visibility
        function togglePassword() {
            const passwordField = document.getElementById('password');
            const passwordToggle = document.getElementById('passwordToggle');
            
            if (passwordField.type === 'password') {
                passwordField.type = 'text';
                passwordToggle.classList.remove('fa-eye');
                passwordToggle.classList.add('fa-eye-slash');
            } else {
                passwordField.type = 'password';
                passwordToggle.classList.remove('fa-eye-slash');
                passwordToggle.classList.add('fa-eye');
            }
        }
        
        // Form submission handling
        $('#loginForm').on('submit', function(e) {
            // Show loading spinner
            $('#loginSpinner').show();
            $('#loginBtn').prop('disabled', true);
            
            // Simple validation
            const username = $('#username').val().trim();
            const password = $('#password').val();
            
            if (!username || !password) {
                e.preventDefault();
                $('#loginSpinner').hide();
                $('#loginBtn').prop('disabled', false);
                
                // Show error
                showError('Please enter both username and password.');
                return false;
            }
        });
        
        // Show error message
        function showError(message) {
            const alertHtml = `
                <div class="alert alert-danger" role="alert">
                    <i class="fas fa-exclamation-circle me-2"></i>
                    ${message}
                </div>
            `;
            
            // Remove existing alerts
            $('.alert').remove();
            
            // Add new alert
            $('.login-form h2').after(alertHtml);
            
            // Auto-hide after 5 seconds
            setTimeout(() => {
                $('.alert').fadeOut();
            }, 5000);
        }
        
        // Auto-focus username field
        $(document).ready(function() {
            $('#username').focus();
            
            // Clear form on page load if there was an error
            <?php if (!empty($error_message)): ?>
            setTimeout(() => {
                $('.alert').fadeOut();
            }, 8000);
            <?php endif; ?>
        });
        
        // Keyboard shortcuts
        $(document).keydown(function(e) {
            // Alt + L to focus username
            if (e.altKey && e.keyCode === 76) {
                e.preventDefault();
                $('#username').focus();
            }
        });
        
        // Demo credential quick fill (remove in production)
        $('.demo-credentials').on('click', 'small', function() {
            const text = $(this).text();
            const parts = text.split(' / ');
            if (parts.length === 2) {
                $('#username').val(parts[0]);
                $('#password').val(parts[1]);
            }
        });
    </script>
</body>
</html>