<?php
/**
 * Logout Page
 * Employee Recruitment Management System
 * RR Construction (Pvt) Ltd
 */

// Define access constant
define('HRM_ACCESS', true);

// Start session
session_start();

// Include required files
require_once '../config/constants.php';
require_once '../config/database.php';
require_once '../includes/functions.php';

// Check if user is logged in
if (!isLoggedIn()) {
    // Already logged out, redirect to login
    header('Location: login.php?logout=already');
    exit;
}

// Get current user info before logout
$currentUser = getCurrentUser();
$userName = $currentUser ? $currentUser['full_name'] : 'User';

// Clear remember me token if exists
if (isset($_COOKIE['remember_token'])) {
    try {
        // Clear token from database
        if ($currentUser) {
            $sql = "UPDATE users SET remember_token = NULL WHERE id = ?";
            executeQuery($sql, [$currentUser['id']]);
        }
        
        // Clear cookie
        setcookie('remember_token', '', time() - 3600, BASE_URL, '', false, true);
    } catch (Exception $e) {
        error_log("Error clearing remember token: " . $e->getMessage());
    }
}

// Log logout activity
if ($currentUser) {
    logActivity($currentUser['id'], 'logout', 'users', $currentUser['id']);
}

// Check for timeout parameter
$isTimeout = isset($_GET['timeout']) && $_GET['timeout'] == '1';
$logoutReason = $isTimeout ? 'Session timeout' : 'User logout';

// Perform logout
logoutUser();

// Determine redirect URL and message
if ($isTimeout) {
    $redirectUrl = 'login.php?timeout=1';
    $message = 'Your session has expired for security reasons.';
    $messageType = 'warning';
} else {
    $redirectUrl = 'login.php?logout=success';
    $message = 'You have been successfully logged out.';
    $messageType = 'success';
}

// Check if this is an AJAX request
if (isAjaxRequest()) {
    sendJsonResponse([
        'success' => true,
        'message' => $message,
        'redirect' => $redirectUrl
    ]);
}

// For regular requests, show logout page with auto-redirect
$page_title = 'Logout';
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $page_title; ?> - <?php echo APP_NAME; ?></title>
    
    <!-- Favicon -->
    <link rel="icon" type="image/x-icon" href="<?php echo ASSETS_URL; ?>images/favicon.ico">
    
    <!-- Bootstrap CSS -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    
    <!-- Font Awesome -->
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css" rel="stylesheet">
    
    <!-- Custom CSS -->
    <style>
        :root {
            --primary-color: <?php echo PRIMARY_COLOR; ?>;
            --secondary-color: <?php echo SECONDARY_COLOR; ?>;
        }
        
        body {
            font-family: 'Arial', sans-serif;
            background: white;
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 20px;
        }
        
        .logout-container {
            background: white;
            border: 1px solid #e5e7eb;
            border-radius: 8px;
            box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
            padding: 40px 30px;
            text-align: center;
            max-width: 450px;
            width: 100%;
        }
        
        .logout-icon {
            width: 60px;
            height: 60px;
            background: var(--primary-color);
            color: white;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            margin: 0 auto 20px;
        }
        
        .logout-icon i {
            font-size: 1.5rem;
        }
        
        .logout-title {
            color: var(--secondary-color);
            font-size: 1.5rem;
            font-weight: 600;
            margin-bottom: 10px;
        }
        
        .logout-message {
            color: #6b7280;
            font-size: 1rem;
            margin-bottom: 25px;
            line-height: 1.5;
        }
        
        .user-info {
            background: #f9fafb;
            border: 1px solid #e5e7eb;
            border-radius: 6px;
            padding: 15px;
            margin-bottom: 25px;
            text-align: left;
        }
        
        .user-info h6 {
            color: var(--secondary-color);
            margin-bottom: 8px;
            font-weight: 600;
            font-size: 0.9rem;
        }
        
        .user-info p {
            color: #6b7280;
            margin: 0 0 5px 0;
            font-size: 0.85rem;
        }
        
        .countdown-container {
            background: #fef2f2;
            border: 1px solid #fecaca;
            border-radius: 6px;
            padding: 15px;
            margin-bottom: 25px;
        }
        
        .countdown-text {
            color: #dc2626;
            font-weight: 500;
            margin-bottom: 8px;
            font-size: 0.9rem;
        }
        
        .countdown-timer {
            font-size: 1.2rem;
            font-weight: 600;
            color: var(--primary-color);
        }d: white;
            border-radius: 15px;
            box-shadow: 0 15px 35px rgba(0, 0, 0, 0.1);
            padding: 60px 40px;
            text-align: center;
            max-width: 500px;
            width: 100%;
            position: relative;
            overflow: hidden;
        }
        
        .logout-container::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            height: 5px;
            background: linear-gradient(90deg, var(--primary-color), #b91c1c);
        }
        
        .logout-icon {
            width: 80px;
            height: 80px;
            background: linear-gradient(135deg, var(--primary-color), #b91c1c);
            color: white;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            margin: 0 auto 30px;
            animation: pulse 2s infinite;
        }
        
        .logout-icon i {
            font-size: 2rem;
        }
        
        @keyframes pulse {
            0% { transform: scale(1); }
            50% { transform: scale(1.1); }
            100% { transform: scale(1); }
        }
        
        .logout-title {
            color: var(--secondary-color);
            font-size: 2rem;
            font-weight: 600;
            margin-bottom: 15px;
        }
        
        .logout-message {
            color: #6b7280;
            font-size: 1.1rem;
            margin-bottom: 30px;
            line-height: 1.6;
        }
        
        .user-info {
            background: #f9fafb;
            border-radius: 10px;
            padding: 20px;
            margin-bottom: 30px;
            border-left: 4px solid var(--primary-color);
        }
        
        .user-info h6 {
            color: var(--secondary-color);
            margin-bottom: 5px;
            font-weight: 600;
        }
        
        .user-info p {
            color: #6b7280;
            margin: 0;
            font-size: 0.95rem;
        }
        
        .countdown-container {
            background: #fef3f2;
            border: 1px solid #fecaca;
            border-radius: 10px;
            padding: 20px;
            margin-bottom: 30px;
        }
        
        .countdown-text {
            color: #dc2626;
            font-weight: 600;
            margin-bottom: 10px;
        }
        
        .countdown-timer {
            font-size: 1.5rem;
            font-weight: bold;
            color: var(--primary-color);
            font-family: 'Courier New', monospace;
        }
        
        .action-buttons {
            display: flex;
            gap: 10px;
            justify-content: center;
            flex-wrap: wrap;
        }
        
        .btn-primary {
            background: var(--primary-color);
            border: none;
            border-radius: 6px;
            padding: 10px 20px;
            font-weight: 500;
            text-decoration: none;
            color: white;
            transition: background-color 0.3s ease;
            display: inline-flex;
            align-items: center;
            font-size: 0.9rem;
        }
        
        .btn-primary:hover {
            background-color: #b91c1c;
            color: white;
        }
        
        .btn-outline-secondary {
            border: 1px solid #d1d5db;
            border-radius: 6px;
            padding: 10px 20px;
            font-weight: 500;
            text-decoration: none;
            color: var(--secondary-color);
            background: white;
            transition: all 0.3s ease;
            display: inline-flex;
            align-items: center;
            font-size: 0.9rem;
        }
        
        .btn-outline-secondary:hover {
            background: #f9fafb;
            border-color: var(--primary-color);
            color: var(--primary-color);
        }
        
        .security-info {
            margin-top: 25px;
            padding-top: 20px;
            border-top: 1px solid #e5e7eb;
        }
        
        .security-info h6 {
            color: var(--secondary-color);
            font-size: 0.9rem;
            margin-bottom: 10px;
            font-weight: 600;
        }
        
        .security-tips {
            text-align: left;
            color: #6b7280;
            font-size: 0.8rem;
            line-height: 1.5;
        }
        
        .security-tips li {
            margin-bottom: 4px;
        }
        
        .footer-info {
            margin-top: 25px;
            padding-top: 15px;
            border-top: 1px solid #e5e7eb;
            color: #9ca3af;
            font-size: 0.8rem;
        }
        
        /* Mobile Responsive */
        @media (max-width: 768px) {
            .logout-container {
                margin: 20px;
                padding: 30px 20px;
            }
            
            .logout-title {
                font-size: 1.3rem;
            }
            
            .logout-message {
                font-size: 0.9rem;
            }
            
            .action-buttons {
                flex-direction: column;
            }
            
            .btn-primary,
            .btn-outline-secondary {
                width: 100%;
                justify-content: center;
            }
        }
        
        /* Simple animation */
        .fade-in {
            animation: fadeIn 0.5s ease-in;
        }
        
        @keyframes fadeIn {
            from { opacity: 0; }
            to { opacity: 1; }
        }
        
        .logout-success {
            color: #059669;
        }
        
        .logout-timeout {
            color: #dc2626;
        } {
            display: flex;
            gap: 15px;
            justify-content: center;
            flex-wrap: wrap;
        }
        
        .btn-primary {
            background: linear-gradient(135deg, var(--primary-color), #b91c1c);
            border: none;
            border-radius: 10px;
            padding: 12px 30px;
            font-weight: 600;
            text-decoration: none;
            color: white;
            transition: all 0.3s ease;
            display: inline-flex;
            align-items: center;
        }
        
        .btn-primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(220, 38, 38, 0.3);
            color: white;
        }
        
        .btn-outline-secondary {
            border: 2px solid #d1d5db;
            border-radius: 10px;
            padding: 10px 30px;
            font-weight: 600;
            text-decoration: none;
            color: var(--secondary-color);
            background: white;
            transition: all 0.3s ease;
            display: inline-flex;
            align-items: center;
        }
        
        .btn-outline-secondary:hover {
            background: #f9fafb;
            border-color: var(--primary-color);
            color: var(--primary-color);
        }
        
        .security-info {
            margin-top: 30px;
            padding-top: 20px;
            border-top: 1px solid #e5e7eb;
        }
        
        .security-info h6 {
            color: var(--secondary-color);
            font-size: 0.9rem;
            margin-bottom: 10px;
            font-weight: 600;
        }
        
        .security-tips {
            text-align: left;
            color: #6b7280;
            font-size: 0.85rem;
            line-height: 1.6;
        }
        
        .security-tips li {
            margin-bottom: 5px;
        }
        
        .footer-info {
            margin-top: 30px;
            padding-top: 20px;
            border-top: 1px solid #e5e7eb;
            color: #9ca3af;
            font-size: 0.8rem;
        }
        
        /* Mobile Responsive */
        @media (max-width: 768px) {
            .logout-container {
                margin: 20px;
                padding: 40px 20px;
            }
            
            .logout-title {
                font-size: 1.5rem;
            }
            
            .logout-message {
                font-size: 1rem;
            }
            
            .action-buttons {
                flex-direction: column;
            }
            
            .btn-primary,
            .btn-outline-secondary {
                width: 100%;
                justify-content: center;
            }
        }
        
        /* Animation */
        .fade-in {
            animation: fadeIn 0.8s ease-in;
        }
        
        @keyframes fadeIn {
            from { opacity: 0; transform: translateY(30px); }
            to { opacity: 1; transform: translateY(0); }
        }
        
        .logout-success {
            color: #059669;
        }
        
        .logout-timeout {
            color: #dc2626;
        }
    </style>
    
    <!-- Auto-redirect script -->
    <script>
        // Auto-redirect after 5 seconds
        let countdown = 5;
        
        function updateCountdown() {
            document.getElementById('countdown').textContent = countdown;
            countdown--;
            
            if (countdown < 0) {
                window.location.href = '<?php echo $redirectUrl; ?>';
            }
        }
        
        // Start countdown
        const countdownInterval = setInterval(updateCountdown, 1000);
        
        // Update immediately
        updateCountdown();
    </script>
</head>
<body>
    <div class="logout-container fade-in">
        <!-- Logout Icon -->
        <div class="logout-icon <?php echo $isTimeout ? 'logout-timeout' : 'logout-success'; ?>">
            <i class="fas <?php echo $isTimeout ? 'fa-clock' : 'fa-sign-out-alt'; ?>"></i>
        </div>
        
        <!-- Title -->
        <h1 class="logout-title">
            <?php echo $isTimeout ? 'Session Expired' : 'Logged Out Successfully'; ?>
        </h1>
        
        <!-- Message -->
        <p class="logout-message">
            <?php if ($isTimeout): ?>
                Your session has expired for security reasons. Please log in again to continue.
            <?php else: ?>
                Thank you for using <?php echo APP_NAME; ?>. You have been safely logged out.
            <?php endif; ?>
        </p>
        
        <!-- User Info -->
        <?php if (!$isTimeout && $currentUser): ?>
        <div class="user-info">
            <h6><i class="fas fa-user me-2"></i>Session Summary</h6>
            <p><strong>User:</strong> <?php echo htmlspecialchars($userName); ?></p>
            <p><strong>Role:</strong> <?php echo getRoleName($currentUser['role']); ?></p>
            <p><strong>Logout Time:</strong> <?php echo date('d/m/Y H:i:s'); ?></p>
            <p><strong>Session Duration:</strong> 
                <?php 
                if (isset($_SESSION['login_time'])) {
                    $duration = time() - $_SESSION['login_time'];
                    $hours = floor($duration / 3600);
                    $minutes = floor(($duration % 3600) / 60);
                    echo "{$hours}h {$minutes}m";
                } else {
                    echo "Unknown";
                }
                ?>
            </p>
        </div>
        <?php endif; ?>
        
        <!-- Countdown -->
        <div class="countdown-container">
            <div class="countdown-text">
                <i class="fas fa-clock me-2"></i>
                Redirecting to login page in:
            </div>
            <div class="countdown-timer">
                <span id="countdown">5</span> seconds
            </div>
        </div>
        
        <!-- Action Buttons -->
        <div class="action-buttons">
            <a href="login.php" class="btn-primary">
                <i class="fas fa-sign-in-alt me-2"></i>
                Login Again
            </a>
            <a href="<?php echo BASE_URL; ?>" class="btn-outline-secondary">
                <i class="fas fa-home me-2"></i>
                Home Page
            </a>
        </div>
        
        <!-- Security Information -->
        <div class="security-info">
            <h6><i class="fas fa-shield-alt me-2"></i>Security Information</h6>
            <ul class="security-tips">
                <li>All session data has been cleared</li>
                <li>Remember me tokens have been invalidated</li>
                <li>For security, close all browser windows when finished</li>
                <?php if ($isTimeout): ?>
                <li>Sessions expire after <?php echo SESSION_TIMEOUT / 60; ?> minutes of inactivity</li>
                <?php endif; ?>
            </ul>
        </div>
        
        <!-- Footer -->
        <div class="footer-info">
            <div class="mb-2">
                <i class="fas fa-building me-1"></i>
                <?php echo COMPANY_NAME; ?>
            </div>
            <div class="mb-1">
                <i class="fas fa-phone me-1"></i><?php echo COMPANY_PHONE; ?>
                <span class="mx-2">|</span>
                <i class="fas fa-envelope me-1"></i><?php echo COMPANY_EMAIL; ?>
            </div>
            <div>
                <?php echo APP_NAME; ?> v<?php echo APP_VERSION; ?>
            </div>
        </div>
    </div>

    <!-- Scripts -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    
    <script>
        // Handle manual navigation
        document.addEventListener('DOMContentLoaded', function() {
            // Clear any remaining session data on client side
            if (typeof(Storage) !== "undefined") {
                sessionStorage.clear();
                localStorage.removeItem('user_data');
                localStorage.removeItem('last_activity');
            }
            
            // Disable back button to prevent session issues
            window.history.pushState(null, null, window.location.href);
            window.onpopstate = function(event) {
                window.location.href = 'login.php';
            };
        });
        
        // Add click handlers for immediate redirect
        document.querySelectorAll('.btn-primary, .btn-outline-secondary').forEach(function(btn) {
            btn.addEventListener('click', function() {
                clearInterval(countdownInterval);
            });
        });
        
        // Clear any cached credentials
        if (navigator.credentials && navigator.credentials.preventSilentAccess) {
            navigator.credentials.preventSilentAccess();
        }
        
        // Log logout event for analytics (if needed)
        console.log('User logout completed:', {
            timestamp: new Date().toISOString(),
            reason: '<?php echo $logoutReason; ?>',
            redirect: '<?php echo $redirectUrl; ?>'
        });
    </script>
</body>
</html>