<?php
/**
 * Change Password
 * Employee Recruitment Management System
 * RR Construction (Pvt) Ltd
 */

// Include header first to load constants
require_once 'includes/header.php';

// Page configuration (after constants are loaded)
$page_title = "Change Password";
$active_menu = "profile";
$breadcrumbs = [
    ['title' => 'My Profile', 'url' => BASE_URL . 'profile.php'],
    ['title' => 'Change Password', 'url' => '#']
];

// Require login
if (!isLoggedIn()) {
    header('Location: ' . BASE_URL . 'auth/login.php');
    exit;
}

// Get current user details
$current_user = getCurrentUser();
if (!$current_user) {
    setFlashMessage('User session expired. Please login again.', 'danger');
    header('Location: ' . BASE_URL . 'auth/login.php');
    exit;
}

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        // Verify CSRF token
        if (!isset($_POST['csrf_token']) || !verifyCSRFToken($_POST['csrf_token'])) {
            throw new Exception('Invalid security token. Please try again.');
        }
        
        // Get form data
        $current_password = $_POST['current_password'] ?? '';
        $new_password = $_POST['new_password'] ?? '';
        $confirm_password = $_POST['confirm_password'] ?? '';
        
        // Validation
        $errors = [];
        
        // Check current password
        if (empty($current_password)) {
            $errors['current_password'] = 'Current password is required';
        } elseif ($current_password !== $current_user['password']) {
            $errors['current_password'] = 'Current password is incorrect';
        }
        
        // Validate new password
        if (empty($new_password)) {
            $errors['new_password'] = 'New password is required';
        } elseif (strlen($new_password) < MIN_PASSWORD_LENGTH) {
            $errors['new_password'] = 'Password must be at least ' . MIN_PASSWORD_LENGTH . ' characters long';
        }
        
        // Check if new password is same as current
        if (!empty($new_password) && $new_password === $current_password) {
            $errors['new_password'] = 'New password must be different from current password';
        }
        
        // Confirm password validation
        if (empty($confirm_password)) {
            $errors['confirm_password'] = 'Please confirm your new password';
        } elseif ($new_password !== $confirm_password) {
            $errors['confirm_password'] = 'Password confirmation does not match';
        }
        
        // Additional password strength validation (if enabled in constants)
        if (!empty($new_password)) {
            if (PASSWORD_REQUIRE_UPPERCASE && !preg_match('/[A-Z]/', $new_password)) {
                $errors['new_password'] = 'Password must contain at least one uppercase letter';
            }
            
            if (PASSWORD_REQUIRE_LOWERCASE && !preg_match('/[a-z]/', $new_password)) {
                $errors['new_password'] = 'Password must contain at least one lowercase letter';
            }
            
            if (PASSWORD_REQUIRE_NUMBERS && !preg_match('/[0-9]/', $new_password)) {
                $errors['new_password'] = 'Password must contain at least one number';
            }
            
            if (PASSWORD_REQUIRE_SYMBOLS && !preg_match('/[^A-Za-z0-9]/', $new_password)) {
                $errors['new_password'] = 'Password must contain at least one special character';
            }
        }
        
        if (empty($errors)) {
            // Update password in database
            $updated = updateRecord(
                'users', 
                ['password' => $new_password, 'updated_at' => date('Y-m-d H:i:s')], 
                'id = ?', 
                [$current_user['id']]
            );
            
            if ($updated) {
                // Log the activity
                logActivity(
                    $current_user['id'], 
                    'password_change', 
                    'users', 
                    $current_user['id']
                );
                
                setFlashMessage('Password changed successfully!', 'success');
                header('Location: ' . BASE_URL . 'profile.php');
                exit;
            } else {
                throw new Exception('Failed to update password. Please try again.');
            }
        }
        
    } catch (Exception $e) {
        setFlashMessage('Error: ' . $e->getMessage(), 'danger');
    }
}

// Get password requirements for display
$password_requirements = [];
$password_requirements[] = 'At least ' . MIN_PASSWORD_LENGTH . ' characters long';
if (PASSWORD_REQUIRE_UPPERCASE) $password_requirements[] = 'At least one uppercase letter (A-Z)';
if (PASSWORD_REQUIRE_LOWERCASE) $password_requirements[] = 'At least one lowercase letter (a-z)';
if (PASSWORD_REQUIRE_NUMBERS) $password_requirements[] = 'At least one number (0-9)';
if (PASSWORD_REQUIRE_SYMBOLS) $password_requirements[] = 'At least one special character (!@#$%^&*)';

// Get recent password changes (for security monitoring)
$recent_changes = fetchAll(
    "SELECT created_at, ip_address 
     FROM system_logs 
     WHERE user_id = ? AND action = 'password_change' 
     ORDER BY created_at DESC 
     LIMIT 5",
    [$current_user['id']]
);
?>

<div class="row justify-content-center">
    <div class="col-md-8">
        <!-- Change Password Form -->
        <div class="card">
            <div class="card-header">
                <h5 class="mb-0">
                    <i class="fas fa-key me-2"></i>Change Password
                </h5>
            </div>
            <div class="card-body">
                <!-- Security Notice -->
                <div class="alert alert-info">
                    <i class="fas fa-info-circle me-2"></i>
                    <strong>Security Notice:</strong> 
                    For your account security, please choose a strong password that you haven't used before.
                </div>

                <form method="POST" id="changePasswordForm">
                    <input type="hidden" name="csrf_token" value="<?php echo generateCSRFToken(); ?>">
                    
                    <!-- Current Password -->
                    <div class="mb-3">
                        <label for="current_password" class="form-label">
                            Current Password <span class="text-danger">*</span>
                        </label>
                        <div class="input-group">
                            <input type="password" 
                                   class="form-control <?php echo isset($errors['current_password']) ? 'is-invalid' : ''; ?>" 
                                   id="current_password" 
                                   name="current_password" 
                                   required
                                   autocomplete="current-password">
                            <button class="btn btn-outline-secondary" type="button" id="toggleCurrentPassword">
                                <i class="fas fa-eye"></i>
                            </button>
                        </div>
                        <?php if (isset($errors['current_password'])): ?>
                        <div class="invalid-feedback d-block"><?php echo $errors['current_password']; ?></div>
                        <?php endif; ?>
                    </div>

                    <hr class="my-4">

                    <!-- New Password -->
                    <div class="mb-3">
                        <label for="new_password" class="form-label">
                            New Password <span class="text-danger">*</span>
                        </label>
                        <div class="input-group">
                            <input type="password" 
                                   class="form-control <?php echo isset($errors['new_password']) ? 'is-invalid' : ''; ?>" 
                                   id="new_password" 
                                   name="new_password" 
                                   required
                                   autocomplete="new-password">
                            <button class="btn btn-outline-secondary" type="button" id="toggleNewPassword">
                                <i class="fas fa-eye"></i>
                            </button>
                        </div>
                        <?php if (isset($errors['new_password'])): ?>
                        <div class="invalid-feedback d-block"><?php echo $errors['new_password']; ?></div>
                        <?php endif; ?>
                        
                        <!-- Password Strength Indicator -->
                        <div class="mt-2">
                            <div class="progress" style="height: 5px;">
                                <div class="progress-bar" id="passwordStrength" role="progressbar" style="width: 0%"></div>
                            </div>
                            <small class="text-muted" id="strengthText">Password strength will appear here</small>
                        </div>
                    </div>

                    <!-- Confirm Password -->
                    <div class="mb-3">
                        <label for="confirm_password" class="form-label">
                            Confirm New Password <span class="text-danger">*</span>
                        </label>
                        <div class="input-group">
                            <input type="password" 
                                   class="form-control <?php echo isset($errors['confirm_password']) ? 'is-invalid' : ''; ?>" 
                                   id="confirm_password" 
                                   name="confirm_password" 
                                   required
                                   autocomplete="new-password">
                            <button class="btn btn-outline-secondary" type="button" id="toggleConfirmPassword">
                                <i class="fas fa-eye"></i>
                            </button>
                        </div>
                        <?php if (isset($errors['confirm_password'])): ?>
                        <div class="invalid-feedback d-block"><?php echo $errors['confirm_password']; ?></div>
                        <?php endif; ?>
                        
                        <!-- Match Indicator -->
                        <div class="mt-1">
                            <small id="matchIndicator" class="text-muted"></small>
                        </div>
                    </div>

                    <!-- Password Requirements -->
                    <?php if (!empty($password_requirements)): ?>
                    <div class="mb-4">
                        <h6 class="text-muted">Password Requirements:</h6>
                        <ul class="list-unstyled mb-0">
                            <?php foreach ($password_requirements as $requirement): ?>
                            <li class="small text-muted">
                                <i class="fas fa-check-circle text-success me-2"></i>
                                <?php echo htmlspecialchars($requirement); ?>
                            </li>
                            <?php endforeach; ?>
                        </ul>
                    </div>
                    <?php endif; ?>

                    <!-- Form Actions -->
                    <div class="d-flex justify-content-between">
                        <a href="<?php echo BASE_URL; ?>profile.php" class="btn btn-outline-secondary">
                            <i class="fas fa-arrow-left me-2"></i>Back to Profile
                        </a>
                        <button type="submit" class="btn btn-primary" id="submitBtn">
                            <i class="fas fa-save me-2"></i>Change Password
                        </button>
                    </div>
                </form>
            </div>
        </div>

        <!-- Recent Password Changes -->
        <?php if (!empty($recent_changes)): ?>
        <div class="card mt-4">
            <div class="card-header">
                <h6 class="mb-0">
                    <i class="fas fa-history me-2"></i>Recent Password Changes
                </h6>
            </div>
            <div class="card-body">
                <div class="table-responsive">
                    <table class="table table-sm">
                        <thead>
                            <tr>
                                <th>Date & Time</th>
                                <th>IP Address</th>
                                <th>Status</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($recent_changes as $change): ?>
                            <tr>
                                <td>
                                    <small><?php echo formatDisplayDateTime($change['created_at']); ?></small>
                                </td>
                                <td>
                                    <small class="font-monospace"><?php echo htmlspecialchars($change['ip_address']); ?></small>
                                </td>
                                <td>
                                    <span class="badge bg-success">
                                        <i class="fas fa-check me-1"></i>Success
                                    </span>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
                <small class="text-muted">
                    <i class="fas fa-info-circle me-1"></i>
                    If you see any unfamiliar password changes, please contact your administrator immediately.
                </small>
            </div>
        </div>
        <?php endif; ?>

        <!-- Security Tips -->
        <div class="card mt-4">
            <div class="card-header">
                <h6 class="mb-0">
                    <i class="fas fa-shield-alt me-2"></i>Password Security Tips
                </h6>
            </div>
            <div class="card-body">
                <div class="row">
                    <div class="col-md-6">
                        <h6 class="text-success">
                            <i class="fas fa-check me-2"></i>Do:
                        </h6>
                        <ul class="small text-muted">
                            <li>Use a unique password for this account</li>
                            <li>Include a mix of letters, numbers, and symbols</li>
                            <li>Make it at least <?php echo MIN_PASSWORD_LENGTH; ?> characters long</li>
                            <li>Change your password regularly</li>
                        </ul>
                    </div>
                    <div class="col-md-6">
                        <h6 class="text-danger">
                            <i class="fas fa-times me-2"></i>Don't:
                        </h6>
                        <ul class="small text-muted">
                            <li>Use personal information (names, dates)</li>
                            <li>Reuse passwords from other accounts</li>
                            <li>Share your password with anyone</li>
                            <li>Write it down in obvious places</li>
                        </ul>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
// Page specific JavaScript
function pageInit() {
    // Password visibility toggles
    $('#toggleCurrentPassword').on('click', function() {
        togglePasswordVisibility('current_password', this);
    });
    
    $('#toggleNewPassword').on('click', function() {
        togglePasswordVisibility('new_password', this);
    });
    
    $('#toggleConfirmPassword').on('click', function() {
        togglePasswordVisibility('confirm_password', this);
    });
    
    // Password strength checker
    $('#new_password').on('input', function() {
        checkPasswordStrength($(this).val());
        checkPasswordMatch();
    });
    
    // Password match checker
    $('#confirm_password').on('input', function() {
        checkPasswordMatch();
    });
    
    // Form validation
    $('#changePasswordForm').on('submit', function(e) {
        if (!validatePasswordForm()) {
            e.preventDefault();
            showAlert('Please correct the errors and try again.', 'warning');
        }
    });
    
    // Real-time current password validation
    $('#current_password').on('input', function() {
        const currentPassword = $(this).val();
        if (currentPassword.length > 0) {
            // Remove any existing validation states
            $(this).removeClass('is-invalid is-valid');
        }
    });
}

function togglePasswordVisibility(fieldId, button) {
    const field = document.getElementById(fieldId);
    const icon = button.querySelector('i');
    
    if (field.type === 'password') {
        field.type = 'text';
        icon.className = 'fas fa-eye-slash';
    } else {
        field.type = 'password';
        icon.className = 'fas fa-eye';
    }
}

function checkPasswordStrength(password) {
    const strengthBar = $('#passwordStrength');
    const strengthText = $('#strengthText');
    
    if (!password) {
        strengthBar.css('width', '0%').removeClass().addClass('progress-bar');
        strengthText.text('Password strength will appear here').removeClass();
        return;
    }
    
    let score = 0;
    const checks = {
        length: password.length >= <?php echo MIN_PASSWORD_LENGTH; ?>,
        uppercase: /[A-Z]/.test(password),
        lowercase: /[a-z]/.test(password),
        numbers: /[0-9]/.test(password),
        symbols: /[^A-Za-z0-9]/.test(password)
    };
    
    // Calculate score
    if (checks.length) score += 20;
    if (checks.uppercase) score += 20;
    if (checks.lowercase) score += 20;
    if (checks.numbers) score += 20;
    if (checks.symbols) score += 20;
    
    // Update strength indicator
    strengthBar.css('width', score + '%');
    
    if (score < 40) {
        strengthBar.removeClass().addClass('progress-bar bg-danger');
        strengthText.text('Weak').removeClass().addClass('text-danger');
    } else if (score < 80) {
        strengthBar.removeClass().addClass('progress-bar bg-warning');
        strengthText.text('Medium').removeClass().addClass('text-warning');
    } else {
        strengthBar.removeClass().addClass('progress-bar bg-success');
        strengthText.text('Strong').removeClass().addClass('text-success');
    }
}

function checkPasswordMatch() {
    const newPassword = $('#new_password').val();
    const confirmPassword = $('#confirm_password').val();
    const matchIndicator = $('#matchIndicator');
    
    if (!confirmPassword) {
        matchIndicator.text('').removeClass();
        $('#confirm_password').removeClass('is-invalid is-valid');
        return;
    }
    
    if (newPassword === confirmPassword) {
        matchIndicator.text('Passwords match').removeClass().addClass('text-success');
        $('#confirm_password').removeClass('is-invalid').addClass('is-valid');
    } else {
        matchIndicator.text('Passwords do not match').removeClass().addClass('text-danger');
        $('#confirm_password').removeClass('is-valid').addClass('is-invalid');
    }
}

function validatePasswordForm() {
    let isValid = true;
    
    // Clear previous validation
    $('.form-control').removeClass('is-invalid');
    $('.invalid-feedback').remove();
    
    // Validate current password
    const currentPassword = $('#current_password').val().trim();
    if (!currentPassword) {
        markFieldInvalid($('#current_password'), 'Current password is required');
        isValid = false;
    }
    
    // Validate new password
    const newPassword = $('#new_password').val();
    if (!newPassword) {
        markFieldInvalid($('#new_password'), 'New password is required');
        isValid = false;
    } else if (newPassword.length < <?php echo MIN_PASSWORD_LENGTH; ?>) {
        markFieldInvalid($('#new_password'), 'Password must be at least <?php echo MIN_PASSWORD_LENGTH; ?> characters long');
        isValid = false;
    } else if (newPassword === currentPassword) {
        markFieldInvalid($('#new_password'), 'New password must be different from current password');
        isValid = false;
    }
    
    // Validate confirm password
    const confirmPassword = $('#confirm_password').val();
    if (!confirmPassword) {
        markFieldInvalid($('#confirm_password'), 'Please confirm your new password');
        isValid = false;
    } else if (newPassword !== confirmPassword) {
        markFieldInvalid($('#confirm_password'), 'Password confirmation does not match');
        isValid = false;
    }
    
    return isValid;
}

function markFieldInvalid(field, message) {
    field.addClass('is-invalid');
    field.after('<div class="invalid-feedback d-block">' + message + '</div>');
}
</script>

<?php require_once 'includes/footer.php'; ?>