<?php
/**
 * System Constants
 * Employee Recruitment Management System
 * RR Construction (Pvt) Ltd
 */

// Prevent direct access
if (!defined('HRM_ACCESS')) {
    die('Direct access not permitted');
}

// =============================================
// SYSTEM CONFIGURATION
// =============================================

// Application Info
define('APP_NAME', 'Employee Recruitment Management System');
define('APP_VERSION', '1.0.0');
define('COMPANY_NAME', 'RR Construction (Pvt) Ltd');
define('COMPANY_ADDRESS', 'Colombo, Sri Lanka');
define('COMPANY_PHONE', '+94 11 2345678');
define('COMPANY_EMAIL', 'info@rrconstruction.lk');

// Base paths
define('BASE_PATH', dirname(__DIR__));
define('BASE_URL', '/hrm/');
define('ASSETS_URL', BASE_URL . 'assets/');
define('UPLOADS_URL', BASE_URL . 'uploads/');

// Directory paths
define('UPLOADS_DIR', BASE_PATH . '/uploads/');
define('DOCUMENTS_DIR', UPLOADS_DIR . 'documents/');
define('PHOTOS_DIR', UPLOADS_DIR . 'photos/');
define('TEMP_DIR', UPLOADS_DIR . 'temp/');

// =============================================
// SECURITY SETTINGS
// =============================================

// Session settings
define('SESSION_TIMEOUT', 3600); // 1 hour in seconds
define('REMEMBER_ME_DURATION', 2592000); // 30 days in seconds
define('SESSION_NAME', 'hrm_session');

// Password settings
define('MIN_PASSWORD_LENGTH', 6);
define('PASSWORD_REQUIRE_UPPERCASE', false);
define('PASSWORD_REQUIRE_LOWERCASE', false);
define('PASSWORD_REQUIRE_NUMBERS', false);
define('PASSWORD_REQUIRE_SYMBOLS', false);

// Security tokens
define('CSRF_TOKEN_NAME', 'csrf_token');
define('CSRF_TOKEN_EXPIRY', 3600); // 1 hour

// =============================================
// USER ROLES AND PERMISSIONS
// =============================================

// User roles
define('ROLE_SUPER_ADMIN', 'super_admin');
define('ROLE_SITE_ADMIN', 'site_admin');
define('ROLE_PROJECT_MANAGER', 'project_manager');
define('ROLE_GENERAL_MANAGER', 'general_manager');
define('ROLE_HR_MANAGER', 'hr_manager');
define('ROLE_ACCOUNTANT', 'accountant');

// Role hierarchy (higher number = more permissions)
define('ROLE_LEVELS', [
    ROLE_SITE_ADMIN => 1,
    ROLE_PROJECT_MANAGER => 2,
    ROLE_GENERAL_MANAGER => 3,
    ROLE_HR_MANAGER => 4,
    ROLE_ACCOUNTANT => 4,
    ROLE_SUPER_ADMIN => 5
]);

// Role names for display
define('ROLE_NAMES', [
    ROLE_SUPER_ADMIN => 'Super Administrator',
    ROLE_SITE_ADMIN => 'Site Admin Officer',
    ROLE_PROJECT_MANAGER => 'Project Manager',
    ROLE_GENERAL_MANAGER => 'General Manager',
    ROLE_HR_MANAGER => 'HR Manager',
    ROLE_ACCOUNTANT => 'Accountant'
]);

// =============================================
// REQUEST AND APPLICATION STATUS
// =============================================

// Employee request statuses
define('REQUEST_STATUS_PENDING', 'pending');
define('REQUEST_STATUS_APPROVED_BY_PM', 'approved_by_pm');
define('REQUEST_STATUS_APPROVED_BY_GM', 'approved_by_gm');
define('REQUEST_STATUS_APPROVED_FINAL', 'approved_final');
define('REQUEST_STATUS_REJECTED', 'rejected');

define('REQUEST_STATUSES', [
    REQUEST_STATUS_PENDING => 'Pending Review',
    REQUEST_STATUS_APPROVED_BY_PM => 'Approved by Project Manager',
    REQUEST_STATUS_APPROVED_BY_GM => 'Approved by General Manager',
    REQUEST_STATUS_APPROVED_FINAL => 'Final Approval',
    REQUEST_STATUS_REJECTED => 'Rejected'
]);

// Application statuses - REMOVED DUPLICATES
define('APP_STATUS_SUBMITTED', 'submitted');
define('APP_STATUS_UNDER_REVIEW', 'under_review');
define('APP_STATUS_APPROVED_BY_PM', 'approved_by_pm');
define('APP_STATUS_APPROVED_BY_HR', 'approved_by_hr');
define('APP_STATUS_APPROVED_BY_ACCOUNTANT', 'approved_by_accountant');
define('APP_STATUS_HIRED', 'hired');
define('APP_STATUS_REJECTED', 'rejected');

define('APPLICATION_STATUSES', [
    APP_STATUS_SUBMITTED => 'Submitted',
    APP_STATUS_UNDER_REVIEW => 'Under Review',
    APP_STATUS_APPROVED_BY_PM => 'Approved by Project Manager',
    APP_STATUS_APPROVED_BY_HR => 'Approved by HR Manager',
    APP_STATUS_APPROVED_BY_ACCOUNTANT => 'Approved by Accountant',
    APP_STATUS_HIRED => 'Hired',
    APP_STATUS_REJECTED => 'Rejected'
]);

// =============================================
// EMPLOYEE CATEGORIES AND TYPES
// =============================================

define('EMPLOYEE_CATEGORIES', [
    'Technical' => 'Technical Staff',
    'Labor' => 'Labor/Skilled Workers',
    'Administrative' => 'Administrative Staff',
    'Management' => 'Management',
    'Support' => 'Support Staff'
]);

define('REQUIREMENT_REASONS', [
    'replacement' => 'Employee Replacement',
    'new_requirement' => 'New Employee Requirement'
]);

define('TRANSPORTATION_METHODS', [
    'own_vehicle' => 'Own Vehicle',
    'public_transport' => 'Public Transport',
    'company_transport' => 'Company Transport',
    'other' => 'Other'
]);

define('GENDER_OPTIONS', [
    'male' => 'Male',
    'female' => 'Female'
]);

define('MARITAL_STATUS_OPTIONS', [
    'single' => 'Single',
    'married' => 'Married',
    'divorced' => 'Divorced',
    'widowed' => 'Widowed'
]);

// =============================================
// FILE UPLOAD SETTINGS
// =============================================

// Maximum file sizes (in bytes)
define('MAX_FILE_SIZE', 10485760); // 10MB
define('MAX_PHOTO_SIZE', 2097152); // 2MB

// Allowed file types
define('ALLOWED_DOCUMENT_TYPES', [
    'pdf' => 'application/pdf',
    'doc' => 'application/msword',
    'docx' => 'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
    'jpg' => 'image/jpeg',
    'jpeg' => 'image/jpeg',
    'png' => 'image/png',
    'gif' => 'image/gif'
]);

define('ALLOWED_PHOTO_TYPES', [
    'jpg' => 'image/jpeg',
    'jpeg' => 'image/jpeg',
    'png' => 'image/png'
]);

// Document types
define('DOCUMENT_TYPES', [
    'nic_copy' => 'NIC Copy',
    'birth_certificate' => 'Birth Certificate',
    'cv' => 'Curriculum Vitae',
    'certificates' => 'Educational Certificates',
    'photo' => 'Passport Photo',
    'bank_details' => 'Bank Account Details',
    'medical_report' => 'Medical Report',
    'police_report' => 'Police Clearance',
    'other' => 'Other Documents'
]);

// =============================================
// PAGINATION AND DISPLAY SETTINGS
// =============================================

define('RECORDS_PER_PAGE', 20);
define('SEARCH_MIN_LENGTH', 3);
define('DATE_FORMAT', 'Y-m-d');
define('DATETIME_FORMAT', 'Y-m-d H:i:s');
define('DISPLAY_DATE_FORMAT', 'd/m/Y');
define('DISPLAY_DATETIME_FORMAT', 'd/m/Y H:i');

// =============================================
// EMAIL SETTINGS
// =============================================

define('SMTP_HOST', 'localhost');
define('SMTP_PORT', 587);
define('SMTP_USERNAME', '');
define('SMTP_PASSWORD', '');
define('SMTP_SECURE', 'tls');
define('FROM_EMAIL', 'noreply@rrconstruction.lk');
define('FROM_NAME', 'RR Construction HRM System');

// =============================================
// NOTIFICATION SETTINGS
// =============================================

define('ENABLE_EMAIL_NOTIFICATIONS', true);
define('ENABLE_SMS_NOTIFICATIONS', false);

// Notification types
define('NOTIFICATION_TYPES', [
    'request_submitted' => 'Employee Request Submitted',
    'request_approved' => 'Employee Request Approved',
    'request_rejected' => 'Employee Request Rejected',
    'application_submitted' => 'Job Application Submitted',
    'application_approved' => 'Job Application Approved',
    'application_rejected' => 'Job Application Rejected',
    'user_created' => 'New User Account Created'
]);

// =============================================
// APPROVAL WORKFLOW
// =============================================

// Approval levels for employee requests
define('REQUEST_APPROVAL_FLOW', [
    REQUEST_STATUS_PENDING => ROLE_PROJECT_MANAGER,
    REQUEST_STATUS_APPROVED_BY_PM => ROLE_GENERAL_MANAGER,
    REQUEST_STATUS_APPROVED_BY_GM => [ROLE_HR_MANAGER, ROLE_ACCOUNTANT]
]);

// Approval levels for job applications
define('APPLICATION_APPROVAL_FLOW', [
    APP_STATUS_SUBMITTED => ROLE_PROJECT_MANAGER,
    APP_STATUS_APPROVED_BY_PM => ROLE_HR_MANAGER,
    APP_STATUS_APPROVED_BY_HR => ROLE_ACCOUNTANT
]);

// =============================================
// COLORS AND STYLING
// =============================================

// Brand colors
define('PRIMARY_COLOR', '#DC2626'); // Red
define('SECONDARY_COLOR', '#374151'); // Dark Gray
define('SUCCESS_COLOR', '#10B981'); // Green
define('WARNING_COLOR', '#F59E0B'); // Yellow
define('DANGER_COLOR', '#EF4444'); // Red
define('INFO_COLOR', '#3B82F6'); // Blue

// Status colors
define('STATUS_COLORS', [
    'pending' => '#F59E0B',
    'approved' => '#10B981',
    'rejected' => '#EF4444',
    'submitted' => '#3B82F6',
    'under_review' => '#8B5CF6',
    'hired' => '#059669'
]);

// =============================================
// VALIDATION RULES
// =============================================

define('VALIDATION_RULES', [
    'nic' => '/^([0-9]{9}[xXvV]|[0-9]{12})$/',
    'phone' => '/^[+]?[0-9\s\-\(\)]{7,15}$/',
    'email' => '/^[^\s@]+@[^\s@]+\.[^\s@]+$/',
    'password' => '/^.{6,}$/'
]);

// =============================================
// ERROR MESSAGES
// =============================================

define('ERROR_MESSAGES', [
    'required_field' => 'This field is required',
    'invalid_email' => 'Please enter a valid email address',
    'invalid_phone' => 'Please enter a valid phone number',
    'invalid_nic' => 'Please enter a valid NIC number',
    'weak_password' => 'Password must be at least 6 characters long',
    'file_too_large' => 'File size exceeds maximum limit',
    'invalid_file_type' => 'File type not allowed',
    'upload_failed' => 'File upload failed',
    'access_denied' => 'Access denied',
    'invalid_credentials' => 'Invalid username or password',
    'session_expired' => 'Your session has expired. Please login again',
    'csrf_token_invalid' => 'Invalid security token. Please refresh and try again'
]);

// =============================================
// SUCCESS MESSAGES
// =============================================

define('SUCCESS_MESSAGES', [
    'record_created' => 'Record created successfully',
    'record_updated' => 'Record updated successfully',
    'record_deleted' => 'Record deleted successfully',
    'login_success' => 'Login successful',
    'logout_success' => 'Logged out successfully',
    'password_changed' => 'Password changed successfully',
    'file_uploaded' => 'File uploaded successfully',
    'request_submitted' => 'Employee request submitted successfully',
    'application_submitted' => 'Job application submitted successfully',
    'approval_success' => 'Approval processed successfully'
]);

// =============================================
// HELPER FUNCTIONS
// =============================================

// Get status color
function getStatusColor($status) {
    return STATUS_COLORS[$status] ?? '#6B7280';
}

// Check if user has permission
function hasPermission($userRole, $requiredRole) {
    $userLevel = ROLE_LEVELS[$userRole] ?? 0;
    $requiredLevel = ROLE_LEVELS[$requiredRole] ?? 99;
    return $userLevel >= $requiredLevel;
}

// Format date for display
function formatDate($date, $format = DISPLAY_DATE_FORMAT) {
    if (empty($date)) return '-';
    return date($format, strtotime($date));
}

// Format currency
function formatCurrency($amount) {
    if (empty($amount)) return '-';
    return 'Rs. ' . number_format($amount, 2);
}

// Get role display name
function getRoleName($role) {
    return ROLE_NAMES[$role] ?? $role;
}

// Get status display name
function getStatusName($status, $type = 'request') {
    if ($type === 'application') {
        return APPLICATION_STATUSES[$status] ?? $status;
    }
    return REQUEST_STATUSES[$status] ?? $status;
}

// Generate CSRF token
function generateCSRFToken() {
    if (!isset($_SESSION[CSRF_TOKEN_NAME])) {
        $_SESSION[CSRF_TOKEN_NAME] = bin2hex(random_bytes(32));
        $_SESSION[CSRF_TOKEN_NAME . '_time'] = time();
    }
    return $_SESSION[CSRF_TOKEN_NAME];
}

// Verify CSRF token
function verifyCSRFToken($token) {
    if (!isset($_SESSION[CSRF_TOKEN_NAME]) || 
        !isset($_SESSION[CSRF_TOKEN_NAME . '_time'])) {
        return false;
    }
    
    // Check if token has expired
    if (time() - $_SESSION[CSRF_TOKEN_NAME . '_time'] > CSRF_TOKEN_EXPIRY) {
        unset($_SESSION[CSRF_TOKEN_NAME]);
        unset($_SESSION[CSRF_TOKEN_NAME . '_time']);
        return false;
    }
    
    return hash_equals($_SESSION[CSRF_TOKEN_NAME], $token);
}

// Create directory if not exists
function createDirectory($path, $permissions = 0755) {
    if (!file_exists($path)) {
        return mkdir($path, $permissions, true);
    }
    return true;
}

// Ensure upload directories exist
createDirectory(DOCUMENTS_DIR);
createDirectory(PHOTOS_DIR);
createDirectory(TEMP_DIR);

/**
 * Get application status name
 */
if (!function_exists('getApplicationStatusName')) {
    function getApplicationStatusName($status) {
        $statuses = [
            APP_STATUS_SUBMITTED => 'Submitted',
            APP_STATUS_UNDER_REVIEW => 'Under Review',
            APP_STATUS_APPROVED_BY_PM => 'Approved by PM',
            APP_STATUS_APPROVED_BY_HR => 'Approved by HR',
            APP_STATUS_APPROVED_BY_ACCOUNTANT => 'Approved by Accountant',
            APP_STATUS_HIRED => 'Hired',
            APP_STATUS_REJECTED => 'Rejected'
        ];
        
        return $statuses[$status] ?? 'Unknown';
    }
}

/**
 * Get application status color
 */
if (!function_exists('getApplicationStatusColor')) {
    function getApplicationStatusColor($status) {
        $colors = [
            APP_STATUS_SUBMITTED => '#17a2b8',
            APP_STATUS_UNDER_REVIEW => '#ffc107',
            APP_STATUS_APPROVED_BY_PM => '#fd7e14',
            APP_STATUS_APPROVED_BY_HR => '#6f42c1',
            APP_STATUS_APPROVED_BY_ACCOUNTANT => '#20c997',
            APP_STATUS_HIRED => '#28a745',
            APP_STATUS_REJECTED => '#dc3545'
        ];
        
        return $colors[$status] ?? '#6c757d';
    }
}

?>