<?php
/**
 * Document Download
 * Employee Recruitment Management System
 * RR Construction (Pvt) Ltd
 */

// Define access constant
define('HRM_ACCESS', true);

// Start session
session_start();

// Include required files
require_once '../config/constants.php';
require_once '../config/database.php';
require_once '../includes/functions.php';

// Check authentication
if (!isLoggedIn()) {
    header('HTTP/1.0 403 Forbidden');
    die('Access denied');
}

$currentUser = getCurrentUser();
if (!$currentUser) {
    header('HTTP/1.0 403 Forbidden');
    die('Access denied');
}

// Get document ID
$document_id = (int)($_GET['id'] ?? 0);
$view_mode = isset($_GET['view']) && $_GET['view'] == '1';

if (!$document_id) {
    header('HTTP/1.0 400 Bad Request');
    die('Invalid document ID');
}

try {
    // Get document details with application info
    $sql = "SELECT ed.*, ja.submitted_by, er.site_project, ja.full_name_english
            FROM employee_documents ed
            JOIN job_applications ja ON ed.application_id = ja.id
            JOIN employee_requests er ON ja.request_id = er.id
            WHERE ed.id = ?";
    
    $document = fetchOne($sql, [$document_id]);
    
    if (!$document) {
        header('HTTP/1.0 404 Not Found');
        die('Document not found');
    }
    
    // Check permissions
    $has_access = false;
    
    if ($currentUser['role'] === ROLE_SUPER_ADMIN) {
        // Super admin can access all documents
        $has_access = true;
    } elseif ($currentUser['role'] === ROLE_SITE_ADMIN) {
        // Site admin can access documents they uploaded
        $has_access = ($document['submitted_by'] == $currentUser['id']);
    } elseif ($currentUser['role'] === ROLE_PROJECT_MANAGER) {
        // Project manager can access documents from their project
        $has_access = ($document['site_project'] === $currentUser['site_project']);
    } elseif (in_array($currentUser['role'], [ROLE_GENERAL_MANAGER, ROLE_HR_MANAGER, ROLE_ACCOUNTANT])) {
        // GM, HR, and Accountant can access all documents
        $has_access = true;
    }
    
    if (!$has_access) {
        header('HTTP/1.0 403 Forbidden');
        die('Access denied');
    }
    
    // Check if file exists
    if (!file_exists($document['file_path'])) {
        header('HTTP/1.0 404 Not Found');
        die('File not found on server');
    }
    
    // Log download activity
    logActivity($currentUser['id'], 'download', 'employee_documents', $document_id);
    
    // Get file info
    $file_path = $document['file_path'];
    $file_name = $document['original_filename'];
    $file_size = $document['file_size'];
    $mime_type = $document['mime_type'];
    
    // Clean filename for download
    $safe_filename = preg_replace('/[^a-zA-Z0-9._-]/', '_', $file_name);
    
    // Set headers
    header('Content-Type: ' . $mime_type);
    header('Content-Length: ' . $file_size);
    header('Cache-Control: private, must-revalidate');
    header('Pragma: public');
    header('Last-Modified: ' . gmdate('D, d M Y H:i:s', filemtime($file_path)) . ' GMT');
    
    if ($view_mode && in_array($mime_type, ['application/pdf', 'image/jpeg', 'image/png', 'image/gif'])) {
        // Display in browser
        header('Content-Disposition: inline; filename="' . $safe_filename . '"');
    } else {
        // Force download
        header('Content-Disposition: attachment; filename="' . $safe_filename . '"');
    }
    
    // Prevent any output before file
    while (ob_get_level()) {
        ob_end_clean();
    }
    
    // Output file
    if ($file_size > 10 * 1024 * 1024) { // Files larger than 10MB
        // Stream large files
        $handle = fopen($file_path, 'rb');
        if ($handle) {
            while (!feof($handle)) {
                echo fread($handle, 8192);
                flush();
            }
            fclose($handle);
        }
    } else {
        // Output smaller files directly
        readfile($file_path);
    }
    
    exit;
    
} catch (Exception $e) {
    error_log("Document download error: " . $e->getMessage());
    header('HTTP/1.0 500 Internal Server Error');
    die('Download failed');
}

// If we reach here, something went wrong
header('HTTP/1.0 500 Internal Server Error');
die('Download failed');
?>