<?php
/**
 * Document Upload
 * Employee Recruitment Management System
 * RR Construction (Pvt) Ltd
 */

$page_title = 'Upload Documents';
$active_menu = 'documents';

require_once '../includes/header.php';
requirePermission(ROLE_SITE_ADMIN);

$currentUser = getCurrentUser();
$errors = [];
$success = false;
$uploaded_files = [];

// Get application ID from URL
$application_id = (int)($_GET['app_id'] ?? 0);

// Verify application exists and belongs to current user
$application = null;
if ($application_id) {
    try {
        $sql = "SELECT ja.*, er.request_number, er.site_project 
                FROM job_applications ja 
                JOIN employee_requests er ON ja.request_id = er.id 
                WHERE ja.id = ? AND ja.submitted_by = ?";
        $application = fetchOne($sql, [$application_id, $currentUser['id']]);
    } catch (Exception $e) {
        error_log("Error fetching application: " . $e->getMessage());
    }
}

if (!$application) {
    setFlashMessage('Application not found or access denied.', 'danger');
    header('Location: ../employees/view_applications.php');
    exit;
}

// Process file upload
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!verifyCSRFToken($_POST['csrf_token'] ?? '')) {
        $errors[] = 'Invalid security token.';
    } else {
        $document_type = $_POST['document_type'] ?? '';
        $document_name = cleanInput($_POST['document_name'] ?? '');
        
        if (empty($document_type)) {
            $errors[] = 'Please select document type.';
        }
        
        if (empty($document_name)) {
            $errors[] = 'Please enter document name.';
        }
        
        if (empty($_FILES['document_file']['name'])) {
            $errors[] = 'Please select a file to upload.';
        }
        
        if (empty($errors)) {
            try {
                // Validate file
                $file = $_FILES['document_file'];
                $allowed_types = array_keys(ALLOWED_DOCUMENT_TYPES);
                $validation = validateFileUpload($file, $allowed_types, MAX_FILE_SIZE);
                
                if ($validation === true) {
                    // Upload file
                    $upload_result = uploadFile($file, DOCUMENTS_DIR, $allowed_types, MAX_FILE_SIZE);
                    
                    if ($upload_result['success']) {
                        beginTransaction();
                        
                        // Save document record
                        $document_data = [
                            'application_id' => $application_id,
                            'document_name' => $document_name,
                            'document_type' => $document_type,
                            'original_filename' => $upload_result['original_name'],
                            'file_path' => $upload_result['filepath'],
                            'file_size' => $upload_result['size'],
                            'mime_type' => $upload_result['type'],
                            'uploaded_by' => $currentUser['id']
                        ];
                        
                        $document_id = insertRecord('employee_documents', $document_data);
                        
                        // Log activity
                        logActivity($currentUser['id'], 'upload', 'employee_documents', $document_id);
                        
                        commit();
                        $success = true;
                        $uploaded_files[] = $upload_result['filename'];
                        setFlashMessage("Document '{$document_name}' uploaded successfully!", 'success');
                        
                    } else {
                        $errors[] = $upload_result['error'];
                    }
                } else {
                    $errors[] = $validation;
                }
            } catch (Exception $e) {
                if (isset($upload_result['filepath'])) {
                    deleteFile($upload_result['filepath']);
                }
                rollback();
                error_log("Document upload error: " . $e->getMessage());
                $errors[] = 'Failed to upload document. Please try again.';
            }
        }
    }
}

// Get existing documents for this application
$existing_documents = [];
try {
    $sql = "SELECT ed.*, u.full_name as uploaded_by_name 
            FROM employee_documents ed 
            JOIN users u ON ed.uploaded_by = u.id 
            WHERE ed.application_id = ? 
            ORDER BY ed.uploaded_at DESC";
    $existing_documents = fetchAll($sql, [$application_id]);
} catch (Exception $e) {
    error_log("Error fetching documents: " . $e->getMessage());
}
?>

<div class="row mb-4">
    <div class="col-12">
        <h4 class="mb-0">Upload Documents</h4>
        <small class="text-muted">
            Application: <?php echo htmlspecialchars($application['application_number']); ?> - 
            <?php echo htmlspecialchars($application['full_name_english']); ?>
        </small>
    </div>
</div>

<!-- Application Info -->
<div class="card mb-3">
    <div class="card-header">Application Information</div>
    <div class="card-body">
        <div class="row">
            <div class="col-md-6">
                <table class="table table-sm table-borderless">
                    <tr>
                        <td width="120"><strong>Applicant:</strong></td>
                        <td><?php echo htmlspecialchars($application['full_name_english']); ?></td>
                    </tr>
                    <tr>
                        <td><strong>Position:</strong></td>
                        <td><?php echo htmlspecialchars($application['position_applied']); ?></td>
                    </tr>
                    <tr>
                        <td><strong>NIC:</strong></td>
                        <td><?php echo htmlspecialchars($application['nic_number']); ?></td>
                    </tr>
                </table>
            </div>
            <div class="col-md-6">
                <table class="table table-sm table-borderless">
                    <tr>
                        <td width="120"><strong>Request:</strong></td>
                        <td><?php echo htmlspecialchars($application['request_number']); ?></td>
                    </tr>
                    <tr>
                        <td><strong>Project:</strong></td>
                        <td><?php echo htmlspecialchars($application['site_project']); ?></td>
                    </tr>
                    <tr>
                        <td><strong>Status:</strong></td>
                        <td><?php echo getStatusBadge($application['status'], 'application'); ?></td>
                    </tr>
                </table>
            </div>
        </div>
    </div>
</div>

<?php if (!empty($errors)): ?>
<div class="alert alert-danger">
    <ul class="mb-0">
        <?php foreach ($errors as $error): ?>
            <li><?php echo htmlspecialchars($error); ?></li>
        <?php endforeach; ?>
    </ul>
</div>
<?php endif; ?>

<!-- Upload Form -->
<div class="card mb-4">
    <div class="card-header">Upload New Document</div>
    <div class="card-body">
        <form method="POST" enctype="multipart/form-data" id="uploadForm">
            <input type="hidden" name="csrf_token" value="<?php echo generateCSRFToken(); ?>">
            
            <div class="row mb-3">
                <div class="col-md-4">
                    <label class="form-label">Document Type *</label>
                    <select name="document_type" class="form-control" required>
                        <option value="">Select document type</option>
                        <?php foreach (DOCUMENT_TYPES as $key => $value): ?>
                            <option value="<?php echo $key; ?>"><?php echo $value; ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="col-md-4">
                    <label class="form-label">Document Name *</label>
                    <input type="text" name="document_name" class="form-control" required
                           placeholder="e.g., Original NIC Copy, CV Latest">
                </div>
                <div class="col-md-4">
                    <label class="form-label">File *</label>
                    <input type="file" name="document_file" class="form-control" required
                           accept=".pdf,.doc,.docx,.jpg,.jpeg,.png">
                </div>
            </div>
            
            <div class="mb-3">
                <small class="text-muted">
                    <i class="fas fa-info-circle"></i>
                    Allowed formats: PDF, DOC, DOCX, JPG, PNG | Maximum size: <?php echo formatFileSize(MAX_FILE_SIZE); ?>
                </small>
            </div>
            
            <div class="text-end">
                <a href="../employees/view_applications.php" class="btn btn-outline-secondary">
                    <i class="fas fa-arrow-left"></i> Back to Applications
                </a>
                <button type="submit" class="btn btn-primary">
                    <i class="fas fa-upload"></i> Upload Document
                </button>
            </div>
        </form>
    </div>
</div>

<!-- Existing Documents -->
<div class="card">
    <div class="card-header d-flex justify-content-between align-items-center">
        <span>Uploaded Documents (<?php echo count($existing_documents); ?>)</span>
        <?php if (!empty($existing_documents)): ?>
        <small class="text-muted">Total size: 
            <?php 
            $total_size = array_sum(array_column($existing_documents, 'file_size'));
            echo formatFileSize($total_size);
            ?>
        </small>
        <?php endif; ?>
    </div>
    <div class="card-body p-0">
        <?php if (empty($existing_documents)): ?>
        <div class="text-center py-4">
            <i class="fas fa-folder-open fa-2x text-muted mb-3"></i>
            <p class="text-muted">No documents uploaded yet</p>
            <small class="text-muted">Upload required documents using the form above</small>
        </div>
        <?php else: ?>
        <div class="table-responsive">
            <table class="table table-hover mb-0">
                <thead class="bg-light">
                    <tr>
                        <th>Document</th>
                        <th>Type</th>
                        <th>File Info</th>
                        <th>Uploaded</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($existing_documents as $doc): ?>
                    <tr>
                        <td>
                            <div class="d-flex align-items-center">
                                <i class="<?php echo getFileIcon($doc['original_filename']); ?> me-2 text-primary"></i>
                                <div>
                                    <strong><?php echo htmlspecialchars($doc['document_name']); ?></strong><br>
                                    <small class="text-muted"><?php echo htmlspecialchars($doc['original_filename']); ?></small>
                                </div>
                            </div>
                        </td>
                        <td>
                            <span class="badge bg-secondary">
                                <?php echo DOCUMENT_TYPES[$doc['document_type']] ?? $doc['document_type']; ?>
                            </span>
                        </td>
                        <td>
                            <div><?php echo formatFileSize($doc['file_size']); ?></div>
                            <small class="text-muted"><?php echo htmlspecialchars($doc['mime_type']); ?></small>
                        </td>
                        <td>
                            <div><?php echo formatDisplayDate($doc['uploaded_at']); ?></div>
                            <small class="text-muted">
                                by <?php echo htmlspecialchars($doc['uploaded_by_name']); ?>
                            </small>
                        </td>
                        <td>
                            <div class="btn-group btn-group-sm">
                                <a href="download.php?id=<?php echo $doc['id']; ?>" 
                                   class="btn btn-outline-primary" target="_blank" title="Download">
                                    <i class="fas fa-download"></i>
                                </a>
                                <a href="download.php?id=<?php echo $doc['id']; ?>&view=1" 
                                   class="btn btn-outline-info" target="_blank" title="View">
                                    <i class="fas fa-eye"></i>
                                </a>
                                <button type="button" class="btn btn-outline-danger" 
                                        onclick="deleteDocument(<?php echo $doc['id']; ?>)" title="Delete">
                                    <i class="fas fa-trash"></i>
                                </button>
                            </div>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
        <?php endif; ?>
    </div>
</div>

<!-- Required Documents Checklist -->
<div class="card mt-3">
    <div class="card-header">Required Documents Checklist</div>
    <div class="card-body">
        <div class="row">
            <?php 
            $required_docs = ['nic_copy', 'cv', 'certificates', 'photo'];
            $uploaded_types = array_column($existing_documents, 'document_type');
            
            foreach ($required_docs as $req_type):
                $is_uploaded = in_array($req_type, $uploaded_types);
            ?>
            <div class="col-md-3 mb-2">
                <div class="d-flex align-items-center">
                    <i class="fas <?php echo $is_uploaded ? 'fa-check-circle text-success' : 'fa-circle text-muted'; ?> me-2"></i>
                    <span class="<?php echo $is_uploaded ? 'text-success' : 'text-muted'; ?>">
                        <?php echo DOCUMENT_TYPES[$req_type]; ?>
                    </span>
                </div>
            </div>
            <?php endforeach; ?>
        </div>
        
        <?php 
        $missing_count = count($required_docs) - count(array_intersect($required_docs, $uploaded_types));
        if ($missing_count > 0):
        ?>
        <div class="alert alert-warning mt-2 mb-0">
            <i class="fas fa-exclamation-triangle me-2"></i>
            <?php echo $missing_count; ?> required document(s) still need to be uploaded.
        </div>
        <?php else: ?>
        <div class="alert alert-success mt-2 mb-0">
            <i class="fas fa-check-circle me-2"></i>
            All required documents have been uploaded.
        </div>
        <?php endif; ?>
    </div>
</div>

<script>
function deleteDocument(documentId) {
    if (confirm('Are you sure you want to delete this document? This action cannot be undone.')) {
        // Create form and submit
        const form = document.createElement('form');
        form.method = 'POST';
        form.action = 'delete_document.php';
        
        const csrfInput = document.createElement('input');
        csrfInput.type = 'hidden';
        csrfInput.name = 'csrf_token';
        csrfInput.value = '<?php echo generateCSRFToken(); ?>';
        
        const idInput = document.createElement('input');
        idInput.type = 'hidden';
        idInput.name = 'document_id';
        idInput.value = documentId;
        
        const appInput = document.createElement('input');
        appInput.type = 'hidden';
        appInput.name = 'application_id';
        appInput.value = '<?php echo $application_id; ?>';
        
        form.appendChild(csrfInput);
        form.appendChild(idInput);
        form.appendChild(appInput);
        
        document.body.appendChild(form);
        form.submit();
    }
}

// File size validation
document.querySelector('input[type="file"]').addEventListener('change', function() {
    const file = this.files[0];
    if (file) {
        const maxSize = <?php echo MAX_FILE_SIZE; ?>;
        if (file.size > maxSize) {
            alert('File size exceeds maximum limit of <?php echo formatFileSize(MAX_FILE_SIZE); ?>');
            this.value = '';
        }
    }
});

// Auto-fill document name based on type
document.querySelector('select[name="document_type"]').addEventListener('change', function() {
    const nameInput = document.querySelector('input[name="document_name"]');
    if (this.value && !nameInput.value) {
        const typeName = this.options[this.selectedIndex].text;
        nameInput.value = typeName + ' - <?php echo htmlspecialchars($application['full_name_english']); ?>';
    }
});
</script>

<?php require_once '../includes/footer.php'; ?>