<?php
/**
 * View Documents (Read-only)
 * Employee Recruitment Management System
 * RR Construction (Pvt) Ltd
 */

$page_title = 'View Documents';
$active_menu = 'applications';

require_once '../includes/header.php';
requirePermission(ROLE_SITE_ADMIN);

$currentUser = getCurrentUser();
$application_id = (int)($_GET['app_id'] ?? 0);

if (!$application_id) {
    setFlashMessage('Invalid application ID.', 'danger');
    header('Location: ../employees/view_applications.php');
    exit;
}

// Get application details with access control
$application = null;
try {
    $sql = "SELECT ja.*, er.request_number, er.site_project, ec.position_title,
                   u.full_name as submitted_by_name
            FROM job_applications ja
            JOIN employee_requests er ON ja.request_id = er.id
            JOIN employee_categories ec ON ja.category_id = ec.id
            JOIN users u ON ja.submitted_by = u.id
            WHERE ja.id = ?";
    
    $params = [$application_id];
    
    // Add role-based access control
    switch ($currentUser['role']) {
        case ROLE_SUPER_ADMIN:
            // Super admin can view all documents
            break;
            
        case ROLE_SITE_ADMIN:
            // Site admin can only view their own applications
            $sql .= " AND ja.submitted_by = ?";
            $params[] = $currentUser['id'];
            break;
            
        case ROLE_PROJECT_MANAGER:
            // Project manager can view documents from their project
            $sql .= " AND er.site_project = ?";
            $params[] = $currentUser['site_project'];
            break;
            
        case ROLE_GENERAL_MANAGER:
        case ROLE_HR_MANAGER:
        case ROLE_ACCOUNTANT:
            // These roles can view all documents
            break;
            
        default:
            // No access for other roles
            $sql .= " AND 1=0";
            break;
    }
    
    $application = fetchOne($sql, $params);
    
    if (!$application) {
        setFlashMessage('Application not found or access denied.', 'danger');
        header('Location: ../employees/view_applications.php');
        exit;
    }
    
} catch (Exception $e) {
    error_log("Error fetching application: " . $e->getMessage());
    setFlashMessage('Error loading application details.', 'danger');
    header('Location: ../employees/view_applications.php');
    exit;
}

// Get documents for this application
$documents = [];
try {
    $sql = "SELECT ed.*, u.full_name as uploaded_by_name 
            FROM employee_documents ed 
            JOIN users u ON ed.uploaded_by = u.id 
            WHERE ed.application_id = ? 
            ORDER BY ed.document_type, ed.uploaded_at DESC";
    $documents = fetchAll($sql, [$application_id]);
} catch (Exception $e) {
    error_log("Error fetching documents: " . $e->getMessage());
}
?>

<div class="row mb-4">
    <div class="col-12">
        <div class="d-flex justify-content-between align-items-center">
            <div>
                <h4 class="mb-0">Application Documents</h4>
                <small class="text-muted">
                    Application: <?php echo htmlspecialchars($application['application_number']); ?> - 
                    <?php echo htmlspecialchars($application['full_name_english']); ?>
                </small>
            </div>
            <div class="btn-group">
                <a href="../employees/view_applications.php" class="btn btn-outline-secondary">
                    <i class="fas fa-arrow-left"></i> Back to Applications
                </a>
                <a href="../employees/view_application.php?id=<?php echo $application_id; ?>" class="btn btn-outline-primary">
                    <i class="fas fa-user"></i> View Application
                </a>
            </div>
        </div>
    </div>
</div>

<!-- Application Info -->
<div class="card mb-3">
    <div class="card-header">Application Information</div>
    <div class="card-body">
        <div class="row">
            <div class="col-md-6">
                <table class="table table-sm table-borderless">
                    <tr>
                        <td width="120"><strong>Applicant:</strong></td>
                        <td><?php echo htmlspecialchars($application['full_name_english']); ?></td>
                    </tr>
                    <tr>
                        <td><strong>Position:</strong></td>
                        <td><?php echo htmlspecialchars($application['position_title']); ?></td>
                    </tr>
                    <tr>
                        <td><strong>NIC:</strong></td>
                        <td><?php echo htmlspecialchars($application['nic_number']); ?></td>
                    </tr>
                </table>
            </div>
            <div class="col-md-6">
                <table class="table table-sm table-borderless">
                    <tr>
                        <td width="120"><strong>Request:</strong></td>
                        <td><?php echo htmlspecialchars($application['request_number']); ?></td>
                    </tr>
                    <tr>
                        <td><strong>Project:</strong></td>
                        <td><?php echo htmlspecialchars($application['site_project']); ?></td>
                    </tr>
                    <tr>
                        <td><strong>Status:</strong></td>
                        <td><?php echo getStatusBadge($application['status'], 'application'); ?></td>
                    </tr>
                </table>
            </div>
        </div>
    </div>
</div>

<!-- Documents -->
<div class="card">
    <div class="card-header d-flex justify-content-between align-items-center">
        <span>Documents (<?php echo count($documents); ?>)</span>
        <?php if (!empty($documents)): ?>
        <small class="text-muted">Total size: 
            <?php 
            $total_size = array_sum(array_column($documents, 'file_size'));
            echo formatFileSize($total_size);
            ?>
        </small>
        <?php endif; ?>
    </div>
    <div class="card-body p-0">
        <?php if (empty($documents)): ?>
        <div class="text-center py-5">
            <i class="fas fa-folder-open fa-3x text-muted mb-3"></i>
            <h5>No Documents Found</h5>
            <p class="text-muted">No documents have been uploaded for this application yet.</p>
        </div>
        <?php else: ?>
        <div class="table-responsive">
            <table class="table table-hover mb-0">
                <thead class="bg-light">
                    <tr>
                        <th>Document</th>
                        <th>Type</th>
                        <th>File Info</th>
                        <th>Uploaded</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($documents as $doc): ?>
                    <tr>
                        <td>
                            <div class="d-flex align-items-center">
                                <i class="<?php echo getFileIcon($doc['original_filename']); ?> me-2 text-primary"></i>
                                <div>
                                    <strong><?php echo htmlspecialchars($doc['document_name']); ?></strong><br>
                                    <small class="text-muted"><?php echo htmlspecialchars($doc['original_filename']); ?></small>
                                </div>
                            </div>
                        </td>
                        <td>
                            <span class="badge bg-secondary">
                                <?php echo DOCUMENT_TYPES[$doc['document_type']] ?? $doc['document_type']; ?>
                            </span>
                        </td>
                        <td>
                            <div><?php echo formatFileSize($doc['file_size']); ?></div>
                            <small class="text-muted"><?php echo htmlspecialchars($doc['mime_type']); ?></small>
                        </td>
                        <td>
                            <div><?php echo formatDisplayDate($doc['uploaded_at']); ?></div>
                            <small class="text-muted">
                                by <?php echo htmlspecialchars($doc['uploaded_by_name']); ?>
                            </small>
                        </td>
                        <td>
                            <div class="btn-group btn-group-sm">
                                <a href="download.php?id=<?php echo $doc['id']; ?>" 
                                   class="btn btn-outline-primary" target="_blank" title="Download">
                                    <i class="fas fa-download"></i>
                                </a>
                                <?php if (in_array($doc['mime_type'], ['application/pdf', 'image/jpeg', 'image/png', 'image/gif'])): ?>
                                <a href="download.php?id=<?php echo $doc['id']; ?>&view=1" 
                                   class="btn btn-outline-info" target="_blank" title="View">
                                    <i class="fas fa-eye"></i>
                                </a>
                                <?php endif; ?>
                            </div>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
        <?php endif; ?>
    </div>
</div>

<!-- Document Requirements Status -->
<div class="card mt-3">
    <div class="card-header">Document Requirements</div>
    <div class="card-body">
        <div class="row">
            <div class="col-md-6">
                <h6>Required Documents:</h6>
                <?php 
                $required_docs = ['nic_copy', 'cv', 'certificates', 'photo'];
                $uploaded_types = array_column($documents, 'document_type');
                
                foreach ($required_docs as $req_type):
                    $is_uploaded = in_array($req_type, $uploaded_types);
                ?>
                <div class="d-flex align-items-center mb-1">
                    <i class="fas <?php echo $is_uploaded ? 'fa-check-circle text-success' : 'fa-circle text-muted'; ?> me-2"></i>
                    <span class="<?php echo $is_uploaded ? 'text-success' : 'text-muted'; ?>">
                        <?php echo DOCUMENT_TYPES[$req_type]; ?>
                    </span>
                </div>
                <?php endforeach; ?>
            </div>
            <div class="col-md-6">
                <h6>Optional Documents:</h6>
                <?php 
                $optional_docs = ['birth_certificate', 'bank_details', 'medical_report', 'police_report', 'other'];
                
                foreach ($optional_docs as $opt_type):
                    $is_uploaded = in_array($opt_type, $uploaded_types);
                    if ($is_uploaded):
                ?>
                <div class="d-flex align-items-center mb-1">
                    <i class="fas fa-check-circle text-info me-2"></i>
                    <span class="text-info">
                        <?php echo DOCUMENT_TYPES[$opt_type]; ?>
                    </span>
                </div>
                <?php 
                    endif;
                endforeach; 
                ?>
            </div>
        </div>
        
        <?php 
        $missing_count = count($required_docs) - count(array_intersect($required_docs, $uploaded_types));
        if ($missing_count > 0):
        ?>
        <div class="alert alert-warning mt-3 mb-0">
            <i class="fas fa-exclamation-triangle me-2"></i>
            <?php echo $missing_count; ?> required document(s) are missing.
        </div>
        <?php else: ?>
        <div class="alert alert-success mt-3 mb-0">
            <i class="fas fa-check-circle me-2"></i>
            All required documents have been uploaded.
        </div>
        <?php endif; ?>
    </div>
</div>

<script>
// Bulk download functionality
function downloadAllDocuments() {
    const documentIds = <?php echo json_encode(array_column($documents, 'id')); ?>;
    
    if (documentIds.length === 0) {
        alert('No documents to download.');
        return;
    }
    
    if (confirm(`Download all ${documentIds.length} documents?`)) {
        documentIds.forEach((id, index) => {
            setTimeout(() => {
                window.open(`download.php?id=${id}`, '_blank');
            }, index * 500); // Stagger downloads
        });
    }
}

// Add bulk download button if there are documents
document.addEventListener('DOMContentLoaded', function() {
    const documentCount = <?php echo count($documents); ?>;
    if (documentCount > 1) {
        const header = document.querySelector('.card-header');
        const downloadAllBtn = document.createElement('button');
        downloadAllBtn.className = 'btn btn-sm btn-outline-success';
        downloadAllBtn.innerHTML = '<i class="fas fa-download"></i> Download All';
        downloadAllBtn.onclick = downloadAllDocuments;
        header.appendChild(downloadAllBtn);
    }
});
</script>

<?php require_once '../includes/footer.php'; ?>