<?php
/**
 * Employee Application Form
 * Employee Recruitment Management System
 * RR Construction (Pvt) Ltd
 */

$page_title = 'Employee Application';
$active_menu = 'applications';

require_once '../includes/header.php';
requirePermission(ROLE_SITE_ADMIN);

$currentUser = getCurrentUser();
$errors = [];
$success = false;

// Get approved requests for this user's project with availability check
$approved_requests = [];
try {
    $sql = "SELECT er.id, er.request_number, ec.id as category_id, ec.position_title, ec.category, ec.quantity,
                   COUNT(ja.id) as applications_count,
                   COUNT(CASE WHEN ja.status = 'hired' THEN 1 END) as hired_count,
                   (ec.quantity - COUNT(CASE WHEN ja.status = 'hired' THEN 1 END)) as available_positions
            FROM employee_requests er
            JOIN employee_categories ec ON er.id = ec.request_id
            LEFT JOIN job_applications ja ON ec.id = ja.category_id
            WHERE er.requested_by = ? AND er.status IN ('approved_by_pm', 'approved_by_gm', 'approved_final')
            GROUP BY er.id, ec.id
            HAVING available_positions > 0
            ORDER BY er.created_at DESC";
    $approved_requests = fetchAll($sql, [$currentUser['id']]);
} catch (Exception $e) {
    error_log("Error fetching requests: " . $e->getMessage());
}

// Process form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!verifyCSRFToken($_POST['csrf_token'] ?? '')) {
        $errors[] = 'Invalid security token.';
    } else {
        // Validate required fields
        $required = [
            'request_id', 'category_id', 'full_name_english', 'nic_number', 
            'date_of_birth', 'gender', 'marital_status', 'permanent_address',
            'phone_mobile', 'position_applied'
        ];
        
        $validation_errors = validateRequired($_POST, $required);
        
        // Check if position is still available
        if (!empty($_POST['category_id'])) {
            $availabilityCheck = fetchOne(
                "SELECT ec.quantity,
                        COUNT(CASE WHEN ja.status = 'hired' THEN 1 END) as hired_count,
                        (ec.quantity - COUNT(CASE WHEN ja.status = 'hired' THEN 1 END)) as available_positions
                 FROM employee_categories ec
                 LEFT JOIN job_applications ja ON ec.id = ja.category_id
                 WHERE ec.id = ?
                 GROUP BY ec.id",
                [(int)$_POST['category_id']]
            );
            
            if (!$availabilityCheck || $availabilityCheck['available_positions'] <= 0) {
                $validation_errors['category_id'] = 'This position is no longer available for applications';
            }
        }
        
        // Check for duplicate NIC applications for the same category
        if (!empty($_POST['nic_number']) && !empty($_POST['category_id'])) {
            $duplicateCheck = fetchOne(
                "SELECT ja.id FROM job_applications ja 
                 WHERE ja.nic_number = ? AND ja.category_id = ? AND ja.status != 'rejected'",
                [$_POST['nic_number'], (int)$_POST['category_id']]
            );
            
            if ($duplicateCheck) {
                $validation_errors['nic_number'] = 'An application with this NIC already exists for this position';
            }
        }
        
        // Validate NIC
        if (!empty($_POST['nic_number']) && !validateNIC($_POST['nic_number'])) {
            $validation_errors['nic_number'] = 'Invalid NIC number format';
        }
        
        // Validate email if provided
        if (!empty($_POST['email']) && !validateEmail($_POST['email'])) {
            $validation_errors['email'] = 'Invalid email address';
        }
        
        // Validate phone
        if (!empty($_POST['phone_mobile']) && !validatePhone($_POST['phone_mobile'])) {
            $validation_errors['phone_mobile'] = 'Invalid phone number format';
        }
        
        if (empty($validation_errors)) {
            try {
                beginTransaction();
                
                // Double-check availability before inserting
                $finalAvailabilityCheck = fetchOne(
                    "SELECT ec.quantity,
                            COUNT(CASE WHEN ja.status = 'hired' THEN 1 END) as hired_count,
                            (ec.quantity - COUNT(CASE WHEN ja.status = 'hired' THEN 1 END)) as available_positions
                     FROM employee_categories ec
                     LEFT JOIN job_applications ja ON ec.id = ja.category_id
                     WHERE ec.id = ?
                     GROUP BY ec.id
                     FOR UPDATE", // Lock the row to prevent race conditions
                    [(int)$_POST['category_id']]
                );
                
                if (!$finalAvailabilityCheck || $finalAvailabilityCheck['available_positions'] <= 0) {
                    throw new Exception('Position is no longer available');
                }
                
                // Generate application number
                $application_number = generateApplicationNumber();
                
                // Calculate age
                $age = calculateAge($_POST['date_of_birth']);
                
                // Prepare application data
                $application_data = [
                    'request_id' => (int)$_POST['request_id'],
                    'category_id' => (int)$_POST['category_id'],
                    'application_number' => $application_number,
                    
                    // Personal Information
                    'full_name_english' => cleanInput($_POST['full_name_english']),
                    'full_name_sinhala' => cleanInput($_POST['full_name_sinhala'] ?? ''),
                    'nic_number' => cleanInput($_POST['nic_number']),
                    'date_of_birth' => $_POST['date_of_birth'],
                    'age' => $age,
                    'gender' => $_POST['gender'],
                    'marital_status' => $_POST['marital_status'],
                    'nationality' => cleanInput($_POST['nationality'] ?? 'Sri Lankan'),
                    'religion' => cleanInput($_POST['religion'] ?? ''),
                    'blood_group' => cleanInput($_POST['blood_group'] ?? ''),
                    'district' => cleanInput($_POST['district'] ?? 'Gampaha'),
                    'police_station' => cleanInput($_POST['police_station'] ?? ''),
                    'medical_condition' => cleanInput($_POST['medical_condition'] ?? ''),
                    
                    // Contact Information
                    'permanent_address' => cleanInput($_POST['permanent_address']),
                    'current_address' => cleanInput($_POST['current_address'] ?? ''),
                    'phone_mobile' => cleanInput($_POST['phone_mobile']),
                    'phone_home' => cleanInput($_POST['phone_home'] ?? ''),
                    'email' => cleanInput($_POST['email'] ?? ''),
                    'emergency_contact_name' => cleanInput($_POST['emergency_contact_name'] ?? ''),
                    'emergency_contact_phone' => cleanInput($_POST['emergency_contact_phone'] ?? ''),
                    'emergency_contact_relationship' => cleanInput($_POST['emergency_contact_relationship'] ?? ''),
                    
                    // Employment Information
                    'position_applied' => cleanInput($_POST['position_applied']),
                    'expected_salary' => !empty($_POST['expected_salary']) ? (float)$_POST['expected_salary'] : null,
                    'available_start_date' => $_POST['available_start_date'] ?? null,
                    'transportation_method' => $_POST['transportation_method'] ?? null,
                    'driving_license_no' => cleanInput($_POST['driving_license_no'] ?? ''),
                    'driving_license_type' => $_POST['driving_license_type'] ?? 'none',
                    
                    // Current Employment
                    'currently_employed' => isset($_POST['currently_employed']) ? 1 : 0,
                    'current_employer' => cleanInput($_POST['current_employer'] ?? ''),
                    'current_position' => cleanInput($_POST['current_position'] ?? ''),
                    'current_salary' => !empty($_POST['current_salary']) ? (float)$_POST['current_salary'] : null,
                    'employment_start_date' => $_POST['employment_start_date'] ?? null,
                    'reason_for_leaving' => cleanInput($_POST['reason_for_leaving'] ?? ''),
                    
                    // Previous Employment 1
                    'prev1_employer' => cleanInput($_POST['prev1_employer'] ?? ''),
                    'prev1_position' => cleanInput($_POST['prev1_position'] ?? ''),
                    'prev1_duration' => cleanInput($_POST['prev1_duration'] ?? ''),
                    'prev1_salary' => !empty($_POST['prev1_salary']) ? (float)$_POST['prev1_salary'] : null,
                    'prev1_reason_leaving' => cleanInput($_POST['prev1_reason_leaving'] ?? ''),
                    
                    // Previous Employment 2
                    'prev2_employer' => cleanInput($_POST['prev2_employer'] ?? ''),
                    'prev2_position' => cleanInput($_POST['prev2_position'] ?? ''),
                    'prev2_duration' => cleanInput($_POST['prev2_duration'] ?? ''),
                    'prev2_salary' => !empty($_POST['prev2_salary']) ? (float)$_POST['prev2_salary'] : null,
                    'prev2_reason_leaving' => cleanInput($_POST['prev2_reason_leaving'] ?? ''),
                    
                    // Previous Employment 1
                    'prev1_employer' => cleanInput($_POST['prev1_employer'] ?? ''),
                    'prev1_position' => cleanInput($_POST['prev1_position'] ?? ''),
                    'prev1_duration' => cleanInput($_POST['prev1_duration'] ?? ''),
                    'prev1_salary' => !empty($_POST['prev1_salary']) ? (float)$_POST['prev1_salary'] : null,
                    'prev1_reason_leaving' => cleanInput($_POST['prev1_reason_leaving'] ?? ''),
                    
                    // Previous Employment 2
                    'prev2_employer' => cleanInput($_POST['prev2_employer'] ?? ''),
                    'prev2_position' => cleanInput($_POST['prev2_position'] ?? ''),
                    'prev2_duration' => cleanInput($_POST['prev2_duration'] ?? ''),
                    'prev2_salary' => !empty($_POST['prev2_salary']) ? (float)$_POST['prev2_salary'] : null,
                    'prev2_reason_leaving' => cleanInput($_POST['prev2_reason_leaving'] ?? ''),
                    
                    // Education
                    'elementary_school' => cleanInput($_POST['elementary_school'] ?? ''),
                    'elementary_year_completed' => !empty($_POST['elementary_year_completed']) ? (int)$_POST['elementary_year_completed'] : null,
                    'elementary_school' => cleanInput($_POST['elementary_school'] ?? ''),
                    'elementary_year_completed' => !empty($_POST['elementary_year_completed']) ? (int)$_POST['elementary_year_completed'] : null,
                    'high_school' => cleanInput($_POST['high_school'] ?? ''),
                    'high_school_year_completed' => !empty($_POST['high_school_year']) ? (int)$_POST['high_school_year'] : null,
                    'ol_results' => cleanInput($_POST['ol_results'] ?? ''),
                    'al_results' => cleanInput($_POST['al_results'] ?? ''),
                    'college_university' => cleanInput($_POST['college_university'] ?? ''),
                    'degree_diploma' => cleanInput($_POST['degree_diploma'] ?? ''),
                    'graduation_year' => !empty($_POST['graduation_year']) ? (int)$_POST['graduation_year'] : null,
                    'gpa_class' => cleanInput($_POST['gpa_class'] ?? ''),
                    'other_education' => cleanInput($_POST['other_education'] ?? ''),
                    'certifications' => cleanInput($_POST['certifications'] ?? ''),
                    'training_courses' => cleanInput($_POST['training_courses'] ?? ''),
                    
                    // Banking
                    'bank_name' => cleanInput($_POST['bank_name'] ?? ''),
                    'bank_branch' => cleanInput($_POST['bank_branch'] ?? ''),
                    'account_number' => cleanInput($_POST['account_number'] ?? ''),
                    'account_holder_name' => cleanInput($_POST['account_holder_name'] ?? ''),
                    
                    // Skills and References
                    'special_skills' => cleanInput($_POST['special_skills'] ?? ''),
                    'computer_skills' => cleanInput($_POST['computer_skills'] ?? ''),
                    'language_skills' => cleanInput($_POST['language_skills'] ?? ''),
                    'hobbies' => cleanInput($_POST['hobbies'] ?? ''),
                    'employee_references' => cleanInput($_POST['employee_references'] ?? ''),
                    
                    // System fields
                    'status' => APP_STATUS_SUBMITTED,
                    'submitted_by' => $currentUser['id']
                ];
                
                $application_id = insertRecord('job_applications', $application_data);
                
                // Handle document uploads if any files were provided
                $uploaded_documents = [];
                if (!empty($_FILES['documents'])) {
                    foreach ($_FILES['documents']['name'] as $index => $files) {
                        if (!empty($files['file']) && $files['file']['error'] === UPLOAD_ERR_OK) {
                            $doc_type = $_POST['documents'][$index]['type'] ?? '';
                            $doc_name = cleanInput($_POST['documents'][$index]['name'] ?? '');
                            
                            if (!empty($doc_type) && !empty($doc_name)) {
                                try {
                                    // Validate and upload file
                                    $file_data = [
                                        'name' => $files['file']['name'],
                                        'type' => $files['file']['type'],
                                        'tmp_name' => $files['file']['tmp_name'],
                                        'error' => $files['file']['error'],
                                        'size' => $files['file']['size']
                                    ];
                                    
                                    $allowed_types = ['pdf', 'doc', 'docx', 'jpg', 'jpeg', 'png'];
                                    $max_size = 5 * 1024 * 1024; // 5MB
                                    
                                    if ($file_data['size'] > $max_size) {
                                        throw new Exception("File {$file_data['name']} exceeds maximum size limit");
                                    }
                                    
                                    $file_extension = strtolower(pathinfo($file_data['name'], PATHINFO_EXTENSION));
                                    if (!in_array($file_extension, $allowed_types)) {
                                        throw new Exception("File type not allowed for {$file_data['name']}");
                                    }
                                    
                                    // Create upload directory if it doesn't exist
                                    $upload_dir = '../uploads/documents/';
                                    if (!is_dir($upload_dir)) {
                                        mkdir($upload_dir, 0755, true);
                                    }
                                    
                                    // Generate unique filename
                                    $filename = uniqid() . '_' . time() . '.' . $file_extension;
                                    $filepath = $upload_dir . $filename;
                                    
                                    if (move_uploaded_file($file_data['tmp_name'], $filepath)) {
                                        // Save document record
                                        $document_data = [
                                            'application_id' => $application_id,
                                            'document_name' => $doc_name,
                                            'document_type' => $doc_type,
                                            'original_filename' => $file_data['name'],
                                            'file_path' => $filepath,
                                            'file_size' => $file_data['size'],
                                            'mime_type' => $file_data['type'],
                                            'uploaded_by' => $currentUser['id']
                                        ];
                                        
                                        $document_id = insertRecord('employee_documents', $document_data);
                                        $uploaded_documents[] = $doc_name;
                                        
                                        // Log document upload
                                        logActivity($currentUser['id'], 'upload', 'employee_documents', $document_id);
                                    }
                                } catch (Exception $doc_error) {
                                    error_log("Document upload error: " . $doc_error->getMessage());
                                    // Continue with application submission even if document upload fails
                                }
                            }
                        }
                    }
                }
                
                // Log activity
                logActivity($currentUser['id'], 'create', 'job_applications', $application_id);
                
                commit();
                $success = true;
                
                $success_message = "Application $application_number submitted successfully!";
                if (!empty($uploaded_documents)) {
                    $success_message .= " Documents uploaded: " . implode(', ', $uploaded_documents);
                }
                setFlashMessage($success_message, 'success');
                
            } catch (Exception $e) {
                rollback();
                error_log("Error creating application: " . $e->getMessage());
                $errors[] = 'Failed to submit application. Please try again.';
            }
        } else {
            $errors = array_merge($errors, array_values($validation_errors));
        }
    }
}
?>

<div class="row mb-4">
    <div class="col-12">
        <h4 class="mb-0">Employee Application Form</h4>
        <small class="text-muted">Submit new employee application</small>
    </div>
</div>

<?php if (!empty($errors)): ?>
<div class="alert alert-danger">
    <ul class="mb-0">
        <?php foreach ($errors as $error): ?>
            <li><?php echo htmlspecialchars($error); ?></li>
        <?php endforeach; ?>
    </ul>
</div>
<?php endif; ?>

<?php if ($success): ?>
<div class="alert alert-success">
    Application submitted successfully! <a href="view_applications.php">View all applications</a>
</div>
<?php endif; ?>

<?php if (empty($approved_requests)): ?>
<div class="card">
    <div class="card-body text-center py-5">
        <i class="fas fa-exclamation-triangle fa-2x text-warning mb-3"></i>
        <h5>No Available Positions</h5>
        <p class="text-muted">
            Either you have no approved employee requests, or all requested positions have been filled. 
            Please create a new request or wait for positions to become available.
        </p>
        <a href="../requests/create_request.php" class="btn btn-primary">Create New Request</a>
    </div>
</div>
<?php else: ?>

<!-- Position Availability Summary -->
<div class="card mb-3">
    <div class="card-header bg-info text-white">
        <i class="fas fa-info-circle me-2"></i>Available Positions Summary
    </div>
    <div class="card-body">
        <div class="row">
            <?php 
            $grouped_requests = [];
            foreach ($approved_requests as $req) {
                $grouped_requests[$req['request_number']][] = $req;
            }
            foreach ($grouped_requests as $req_num => $positions): 
            ?>
            <div class="col-md-6 mb-3">
                <div class="border rounded p-3">
                    <h6 class="text-primary mb-2"><?php echo htmlspecialchars($req_num); ?></h6>
                    <?php foreach ($positions as $pos): ?>
                    <div class="d-flex justify-content-between align-items-center mb-1">
                        <span class="small"><?php echo htmlspecialchars($pos['position_title']); ?></span>
                        <span class="badge bg-success">
                            <?php echo $pos['available_positions']; ?> available
                            <small class="text-white-50">
                                (<?php echo $pos['hired_count']; ?>/<?php echo $pos['quantity']; ?> filled)
                            </small>
                        </span>
                    </div>
                    <?php endforeach; ?>
                </div>
            </div>
            <?php endforeach; ?>
        </div>
    </div>
</div>

<form method="POST" id="applicationForm" enctype="multipart/form-data">
    <input type="hidden" name="csrf_token" value="<?php echo generateCSRFToken(); ?>">
    
    <!-- Request Selection -->
    <div class="card mb-3">
        <div class="card-header">Position Selection</div>
        <div class="card-body">
            <div class="row">
                <div class="col-md-6">
                    <label class="form-label">Request Number *</label>
                    <select name="request_id" class="form-control" required onchange="loadCategories()">
                        <option value="">Select approved request</option>
                        <?php 
                        foreach ($grouped_requests as $req_num => $positions):
                        ?>
                            <option value="<?php echo $positions[0]['id']; ?>"><?php echo htmlspecialchars($req_num); ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="col-md-6">
                    <label class="form-label">Position Applied *</label>
                    <select name="category_id" class="form-control" required onchange="updatePositionTitle()">
                        <option value="">Select position</option>
                    </select>
                    <input type="hidden" name="position_applied" id="position_applied">
                    <div id="position_availability" class="mt-2"></div>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Personal Information -->
    <div class="card mb-3">
        <div class="card-header">Personal Information</div>
        <div class="card-body">
            <div class="row mb-3">
                <div class="col-md-6">
                    <label class="form-label">Full Name (English) *</label>
                    <input type="text" name="full_name_english" class="form-control" required>
                </div>
                <div class="col-md-6">
                    <label class="form-label">Full Name (Sinhala)</label>
                    <input type="text" name="full_name_sinhala" class="form-control">
                </div>
            </div>
            
            <div class="row mb-3">
                <div class="col-md-4">
                    <label class="form-label">NIC Number *</label>
                    <input type="text" name="nic_number" class="form-control" required>
                    <small class="text-muted">This will be checked for duplicates</small>
                </div>
                <div class="col-md-4">
                    <label class="form-label">Date of Birth *</label>
                    <input type="date" name="date_of_birth" class="form-control" required>
                </div>
                <div class="col-md-4">
                    <label class="form-label">Gender *</label>
                    <select name="gender" class="form-control" required>
                        <option value="">Select gender</option>
                        <option value="male">Male</option>
                        <option value="female">Female</option>
                    </select>
                </div>
            </div>
            
            <div class="row mb-3">
                <div class="col-md-3">
                    <label class="form-label">Marital Status *</label>
                    <select name="marital_status" class="form-control" required>
                        <option value="">Select status</option>
                        <option value="single">Single</option>
                        <option value="married">Married</option>
                        <option value="divorced">Divorced</option>
                        <option value="widowed">Widowed</option>
                    </select>
                </div>
                <div class="col-md-3">
                    <label class="form-label">Nationality</label>
                    <input type="text" name="nationality" class="form-control" value="Sri Lankan">
                </div>
                <div class="col-md-3">
                    <label class="form-label">Religion</label>
                    <input type="text" name="religion" class="form-control">
                </div>
                <div class="col-md-3">
                    <label class="form-label">Blood Group</label>
                    <select name="blood_group" class="form-control">
                        <option value="">Select blood group</option>
                        <option value="A+">A+</option>
                        <option value="A-">A-</option>
                        <option value="B+">B+</option>
                        <option value="B-">B-</option>
                        <option value="AB+">AB+</option>
                        <option value="AB-">AB-</option>
                        <option value="O+">O+</option>
                        <option value="O-">O-</option>
                    </select>
                </div>
            </div>
            
            <div class="row mb-3">
                <div class="col-md-4">
                    <label class="form-label">District</label>
                    <input type="text" name="district" class="form-control" value="Gampaha">
                </div>
                <div class="col-md-4">
                    <label class="form-label">Nearest Police Station</label>
                    <input type="text" name="police_station" class="form-control" placeholder="e.g., Sapugaskanda">
                </div>
                <div class="col-md-4">
                    <label class="form-label">Medical Condition</label>
                    <textarea name="medical_condition" class="form-control" rows="2" placeholder="Any medical conditions or disabilities"></textarea>
                </div>
            </div>
            
            <div class="row mb-3">
                <div class="col-md-4">
                    <label class="form-label">Driving License Number</label>
                    <input type="text" name="driving_license_no" class="form-control">
                </div>
                <div class="col-md-4">
                    <label class="form-label">License Type</label>
                    <select name="driving_license_type" class="form-control">
                        <option value="none">No License</option>
                        <option value="light_vehicle">Light Vehicle</option>
                        <option value="heavy_vehicle">Heavy Vehicle</option>
                        <option value="both">Both</option>
                    </select>
                </div>
                <div class="col-md-4">
                    <label class="form-label">Language Skills</label>
                    <textarea name="language_skills" class="form-control" rows="2" 
                              placeholder="e.g., Sinhala (Native), English (Good)"></textarea>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Contact Information -->
    <div class="card mb-3">
        <div class="card-header">Contact Information</div>
        <div class="card-body">
            <div class="row mb-3">
                <div class="col-md-6">
                    <label class="form-label">Permanent Address *</label>
                    <textarea name="permanent_address" class="form-control" rows="2" required></textarea>
                </div>
                <div class="col-md-6">
                    <label class="form-label">Current Address</label>
                    <textarea name="current_address" class="form-control" rows="2"></textarea>
                </div>
            </div>
            
            <div class="row mb-3">
                <div class="col-md-4">
                    <label class="form-label">Mobile Phone *</label>
                    <input type="tel" name="phone_mobile" class="form-control" required>
                </div>
                <div class="col-md-4">
                    <label class="form-label">Home Phone</label>
                    <input type="tel" name="phone_home" class="form-control">
                </div>
                <div class="col-md-4">
                    <label class="form-label">Email</label>
                    <input type="email" name="email" class="form-control">
                </div>
            </div>
            
            <div class="row mb-3">
                <div class="col-md-4">
                    <label class="form-label">Emergency Contact Name</label>
                    <input type="text" name="emergency_contact_name" class="form-control">
                </div>
                <div class="col-md-4">
                    <label class="form-label">Emergency Contact Phone</label>
                    <input type="tel" name="emergency_contact_phone" class="form-control">
                </div>
                <div class="col-md-4">
                    <label class="form-label">Relationship</label>
                    <input type="text" name="emergency_contact_relationship" class="form-control">
                </div>
            </div>
        </div>
    </div>
    
    <!-- Employment Information -->
    <div class="card mb-3">
        <div class="card-header">Employment Information</div>
        <div class="card-body">
            <div class="row mb-3">
                <div class="col-md-4">
                    <label class="form-label">Expected Salary</label>
                    <input type="number" name="expected_salary" class="form-control" step="0.01">
                </div>
                <div class="col-md-4">
                    <label class="form-label">Available Start Date</label>
                    <input type="date" name="available_start_date" class="form-control">
                </div>
                <div class="col-md-4">
                    <label class="form-label">Transportation</label>
                    <select name="transportation_method" class="form-control">
                        <option value="">Select method</option>
                        <option value="own_vehicle">Own Vehicle</option>
                        <option value="public_transport">Public Transport</option>
                        <option value="company_transport">Company Transport</option>
                        <option value="other">Other</option>
                    </select>
                </div>
            </div>
            
            <div class="form-check mb-3">
                <input type="checkbox" name="currently_employed" class="form-check-input" id="currentlyEmployed" 
                       onchange="toggleCurrentEmployment()">
                <label class="form-check-label" for="currentlyEmployed">Currently Employed</label>
            </div>
            
            <div id="currentEmploymentFields" style="display: none;">
                <h6>Current Employment Details</h6>
                <div class="row mb-3">
                    <div class="col-md-6">
                        <label class="form-label">Current Employer</label>
                        <input type="text" name="current_employer" class="form-control">
                    </div>
                    <div class="col-md-6">
                        <label class="form-label">Current Position</label>
                        <input type="text" name="current_position" class="form-control">
                    </div>
                </div>
                <div class="row mb-3">
                    <div class="col-md-4">
                        <label class="form-label">Current Salary</label>
                        <input type="number" name="current_salary" class="form-control" step="0.01">
                    </div>
                    <div class="col-md-4">
                        <label class="form-label">Employment Start Date</label>
                        <input type="date" name="employment_start_date" class="form-control">
                    </div>
                    <div class="col-md-4">
                        <label class="form-label">Reason for Leaving</label>
                        <input type="text" name="reason_for_leaving" class="form-control">
                    </div>
                </div>
            </div>
            
            <!-- Previous Employment History -->
            <h6>Previous Employment History</h6>
            
            <!-- Previous Job 1 -->
            <div class="border rounded p-3 mb-3">
                <h6 class="text-primary">Previous Employment 1</h6>
                <div class="row mb-2">
                    <div class="col-md-4">
                        <label class="form-label">Employer Name</label>
                        <input type="text" name="prev1_employer" class="form-control">
                    </div>
                    <div class="col-md-4">
                        <label class="form-label">Position</label>
                        <input type="text" name="prev1_position" class="form-control">
                    </div>
                    <div class="col-md-4">
                        <label class="form-label">Duration</label>
                        <input type="text" name="prev1_duration" class="form-control" placeholder="e.g., 2 years">
                    </div>
                </div>
                <div class="row mb-2">
                    <div class="col-md-4">
                        <label class="form-label">Salary</label>
                        <input type="number" name="prev1_salary" class="form-control" step="0.01">
                    </div>
                    <div class="col-md-8">
                        <label class="form-label">Reason for Leaving</label>
                        <input type="text" name="prev1_reason_leaving" class="form-control">
                    </div>
                </div>
            </div>
            
            <!-- Previous Job 2 -->
            <div class="border rounded p-3 mb-3">
                <h6 class="text-primary">Previous Employment 2</h6>
                <div class="row mb-2">
                    <div class="col-md-4">
                        <label class="form-label">Employer Name</label>
                        <input type="text" name="prev2_employer" class="form-control">
                    </div>
                    <div class="col-md-4">
                        <label class="form-label">Position</label>
                        <input type="text" name="prev2_position" class="form-control">
                    </div>
                    <div class="col-md-4">
                        <label class="form-label">Duration</label>
                        <input type="text" name="prev2_duration" class="form-control" placeholder="e.g., 1.5 years">
                    </div>
                </div>
                <div class="row mb-2">
                    <div class="col-md-4">
                        <label class="form-label">Salary</label>
                        <input type="number" name="prev2_salary" class="form-control" step="0.01">
                    </div>
                    <div class="col-md-8">
                        <label class="form-label">Reason for Leaving</label>
                        <input type="text" name="prev2_reason_leaving" class="form-control">
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Education -->
    <div class="card mb-3">
        <div class="card-header">Education Background</div>
        <div class="card-body">
            <!-- Elementary Education -->
            <div class="row mb-3">
                <div class="col-md-6">
                    <label class="form-label">Elementary School</label>
                    <input type="text" name="elementary_school" class="form-control">
                </div>
                <div class="col-md-6">
                    <label class="form-label">Elementary Year Completed</label>
                    <input type="number" name="elementary_year_completed" class="form-control" min="1950" max="<?php echo date('Y'); ?>">
                </div>
            </div>
            
            <!-- High School -->
            <div class="row mb-3">
                <div class="col-md-6">
                    <label class="form-label">High School</label>
                    <input type="text" name="high_school" class="form-control">
                </div>
                <div class="col-md-6">
                    <label class="form-label">High School Year Completed</label>
                    <input type="number" name="high_school_year" class="form-control" min="1950" max="<?php echo date('Y'); ?>">
                </div>
            </div>
            
            <div class="row mb-3">
                <div class="col-md-6">
                    <label class="form-label">O/L Results</label>
                    <input type="text" name="ol_results" class="form-control" placeholder="e.g., 6 passes including Maths">
                </div>
                <div class="col-md-6">
                    <label class="form-label">A/L Results</label>
                    <input type="text" name="al_results" class="form-control" placeholder="e.g., 2 passes in Science stream">
                </div>
            </div>
            
            <!-- Higher Education -->
            <div class="row mb-3">
                <div class="col-md-4">
                    <label class="form-label">College/University</label>
                    <input type="text" name="college_university" class="form-control">
                </div>
                <div class="col-md-4">
                    <label class="form-label">Degree/Diploma</label>
                    <input type="text" name="degree_diploma" class="form-control">
                </div>
                <div class="col-md-4">
                    <label class="form-label">Graduation Year</label>
                    <input type="number" name="graduation_year" class="form-control" min="1950" max="<?php echo date('Y'); ?>">
                </div>
            </div>
            
            <div class="row mb-3">
                <div class="col-md-4">
                    <label class="form-label">GPA/Class</label>
                    <input type="text" name="gpa_class" class="form-control" placeholder="e.g., First Class, 3.5 GPA">
                </div>
                <div class="col-md-4">
                    <label class="form-label">Certifications</label>
                    <textarea name="certifications" class="form-control" rows="2" placeholder="Professional certifications"></textarea>
                </div>
                <div class="col-md-4">
                    <label class="form-label">Training Courses</label>
                    <textarea name="training_courses" class="form-control" rows="2" placeholder="Additional training courses"></textarea>
                </div>
            </div>
            
            <div class="row mb-3">
                <div class="col-md-12">
                    <label class="form-label">Other Education</label>
                    <textarea name="other_education" class="form-control" rows="2" placeholder="Any other educational qualifications"></textarea>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Banking Information -->
    <div class="card mb-3">
        <div class="card-header">Banking Information</div>
        <div class="card-body">
            <div class="row mb-3">
                <div class="col-md-6">
                    <label class="form-label">Bank Name</label>
                    <input type="text" name="bank_name" class="form-control">
                </div>
                <div class="col-md-6">
                    <label class="form-label">Branch</label>
                    <input type="text" name="bank_branch" class="form-control">
                </div>
            </div>
            
            <div class="row mb-3">
                <div class="col-md-6">
                    <label class="form-label">Account Number</label>
                    <input type="text" name="account_number" class="form-control">
                </div>
                <div class="col-md-6">
                    <label class="form-label">Account Holder Name</label>
                    <input type="text" name="account_holder_name" class="form-control">
                </div>
            </div>
        </div>
    </div>
    
    <!-- Skills & References -->
    <div class="card mb-3">
        <div class="card-header">Skills & References</div>
        <div class="card-body">
            <div class="row mb-3">
                <div class="col-md-4">
                    <label class="form-label">Special Skills</label>
                    <textarea name="special_skills" class="form-control" rows="3" 
                              placeholder="e.g., Welding, Heavy machinery operation"></textarea>
                </div>
                <div class="col-md-4">
                    <label class="form-label">Computer Skills</label>
                    <textarea name="computer_skills" class="form-control" rows="3" 
                              placeholder="e.g., MS Office, AutoCAD"></textarea>
                </div>
                <div class="col-md-4">
                    <label class="form-label">Hobbies</label>
                    <textarea name="hobbies" class="form-control" rows="3" 
                              placeholder="Personal interests and hobbies"></textarea>
                </div>
            </div>
            
            <div class="row mb-3">
                <div class="col-md-12">
                    <label class="form-label">Employee References</label>
                    <textarea name="employee_references" class="form-control" rows="3" 
                              placeholder="Names and contact details of previous supervisors or colleagues who can provide references"></textarea>
                </div>
            </div>
        </div>
    </div>

    <!-- Document Upload -->
    <div class="card mb-3">
        <div class="card-header">
            <i class="fas fa-file-upload me-2"></i>Document Upload
            <small class="text-muted ms-2">(Optional - you can upload documents after submitting the application)</small>
        </div>
        <div class="card-body">
            <div class="row mb-3">
                <div class="col-12">
                    <div class="alert alert-info">
                        <i class="fas fa-info-circle me-2"></i>
                        <strong>Required Documents:</strong> NIC Copy, CV/Resume, Educational Certificates, Passport Photo
                        <br><small>You can upload these documents now or after submitting your application.</small>
                    </div>
                </div>
            </div>
            
            <div id="documentUploadSection">
                <div class="row mb-3 document-upload-row">
                    <div class="col-md-3">
                        <label class="form-label">Document Type</label>
                        <select name="documents[0][type]" class="form-control">
                            <option value="">Select type</option>
                            <option value="nic_copy">NIC Copy</option>
                            <option value="cv">CV/Resume</option>
                            <option value="certificates">Educational Certificates</option>
                            <option value="photo">Passport Photo</option>
                            <option value="birth_certificate">Birth Certificate</option>
                            <option value="bank_details">Bank Details</option>
                            <option value="other">Other</option>
                        </select>
                    </div>
                    <div class="col-md-3">
                        <label class="form-label">Document Name</label>
                        <input type="text" name="documents[0][name]" class="form-control" 
                               placeholder="e.g., Original NIC Copy">
                    </div>
                    <div class="col-md-4">
                        <label class="form-label">File</label>
                        <input type="file" name="documents[0][file]" class="form-control" 
                               accept=".pdf,.doc,.docx,.jpg,.jpeg,.png">
                    </div>
                    <div class="col-md-2 d-flex align-items-end">
                        <button type="button" class="btn btn-outline-danger btn-sm" 
                                onclick="removeDocumentRow(this)" style="display: none;">
                            <i class="fas fa-times"></i>
                        </button>
                    </div>
                </div>
            </div>
            
            <div class="row">
                <div class="col-12">
                    <button type="button" class="btn btn-outline-primary btn-sm" onclick="addDocumentRow()">
                        <i class="fas fa-plus me-1"></i>Add Another Document
                    </button>
                    <small class="text-muted ms-3">
                        Allowed formats: PDF, DOC, DOCX, JPG, PNG | Max size: 5MB per file
                    </small>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Submit -->
    <div class="d-flex justify-content-between">
        <div>
            <a href="view_applications.php" class="btn btn-outline-secondary">
                <i class="fas fa-arrow-left"></i> Cancel
            </a>
        </div>
        <div>
            <button type="button" class="btn btn-outline-info me-2" onclick="saveDraft()">
                <i class="fas fa-save"></i> Save as Draft
            </button>
            <button type="submit" class="btn btn-primary" id="submitBtn">
                <i class="fas fa-paper-plane"></i> Submit Application
            </button>
        </div>
    </div>
</form>

<?php endif; ?>

<script>
// Store request data for category loading
const requestData = <?php echo json_encode($approved_requests); ?>;

function loadCategories() {
    const requestId = document.querySelector('select[name="request_id"]').value;
    const categorySelect = document.querySelector('select[name="category_id"]');
    const availabilityDiv = document.getElementById('position_availability');
    
    // Clear categories and availability info
    categorySelect.innerHTML = '<option value="">Select position</option>';
    availabilityDiv.innerHTML = '';
    
    if (requestId) {
        // Filter categories for selected request
        const categories = requestData.filter(item => item.id == requestId);
        categories.forEach(cat => {
            const option = document.createElement('option');
            option.value = cat.category_id;
            option.textContent = cat.position_title;
            option.dataset.position = cat.position_title;
            option.dataset.available = cat.available_positions;
            option.dataset.hired = cat.hired_count;
            option.dataset.quantity = cat.quantity;
            categorySelect.appendChild(option);
        });
    }
}

function updatePositionTitle() {
    const categorySelect = document.querySelector('select[name="category_id"]');
    const positionInput = document.getElementById('position_applied');
    const availabilityDiv = document.getElementById('position_availability');
    const submitBtn = document.getElementById('submitBtn');
    
    if (categorySelect.value) {
        const selectedOption = categorySelect.options[categorySelect.selectedIndex];
        positionInput.value = selectedOption.dataset.position || selectedOption.textContent;
        
        const available = parseInt(selectedOption.dataset.available);
        const hired = parseInt(selectedOption.dataset.hired);
        const quantity = parseInt(selectedOption.dataset.quantity);
        
        if (available > 0) {
            availabilityDiv.innerHTML = `
                <div class="alert alert-success alert-sm py-2">
                    <i class="fas fa-check-circle me-1"></i>
                    <strong>${available} position(s) available</strong> 
                    <small class="text-muted">(${hired}/${quantity} filled)</small>
                </div>
            `;
            submitBtn.disabled = false;
        } else {
            availabilityDiv.innerHTML = `
                <div class="alert alert-warning alert-sm py-2">
                    <i class="fas fa-exclamation-triangle me-1"></i>
                    <strong>No positions available</strong> 
                    <small class="text-muted">(${hired}/${quantity} filled)</small>
                </div>
            `;
            submitBtn.disabled = true;
        }
    } else {
        positionInput.value = '';
        availabilityDiv.innerHTML = '';
        submitBtn.disabled = false;
    }
}

function toggleCurrentEmployment() {
    const checkbox = document.getElementById('currentlyEmployed');
    const fields = document.getElementById('currentEmploymentFields');
    fields.style.display = checkbox.checked ? 'block' : 'none';
}

// Form validation before submission
document.getElementById('applicationForm').addEventListener('submit', function(e) {
    const categorySelect = document.querySelector('select[name="category_id"]');
    
    if (categorySelect.value) {
        const selectedOption = categorySelect.options[categorySelect.selectedIndex];
        const available = parseInt(selectedOption.dataset.available);
        
        if (available <= 0) {
            e.preventDefault();
            alert('This position is no longer available for applications.');
            return false;
        }
    }
});

// Check for real-time updates every 30 seconds
setInterval(function() {
    const requestId = document.querySelector('select[name="request_id"]').value;
    const categoryId = document.querySelector('select[name="category_id"]').value;
    
    if (requestId && categoryId) {
        // Make AJAX call to check current availability
        fetch('check_availability.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/x-www-data-form-data'
            },
            body: new URLSearchParams({
                'category_id': categoryId,
                'csrf_token': '<?php echo generateCSRFToken(); ?>'
            })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                const availabilityDiv = document.getElementById('position_availability');
                const submitBtn = document.getElementById('submitBtn');
                
                if (data.available_positions > 0) {
                    availabilityDiv.innerHTML = `
                        <div class="alert alert-success alert-sm py-2">
                            <i class="fas fa-check-circle me-1"></i>
                            <strong>${data.available_positions} position(s) available</strong> 
                            <small class="text-muted">(${data.hired_count}/${data.quantity} filled)</small>
                        </div>
                    `;
                    submitBtn.disabled = false;
                } else {
                    availabilityDiv.innerHTML = `
                        <div class="alert alert-danger alert-sm py-2">
                            <i class="fas fa-times-circle me-1"></i>
                            <strong>Position no longer available</strong> 
                            <small class="text-muted">(${data.hired_count}/${data.quantity} filled)</small>
                        </div>
                    `;
                    submitBtn.disabled = true;
                }
            }
        })
        .catch(error => {
            console.error('Error checking availability:', error);
        });
    }
    }, 30000); // Check every 30 seconds

// Document upload functions
let documentRowIndex = 1;

function addDocumentRow() {
    const container = document.getElementById('documentUploadSection');
    const newRow = document.createElement('div');
    newRow.className = 'row mb-3 document-upload-row';
    newRow.innerHTML = `
        <div class="col-md-3">
            <select name="documents[${documentRowIndex}][type]" class="form-control">
                <option value="">Select type</option>
                <option value="nic_copy">NIC Copy</option>
                <option value="cv">CV/Resume</option>
                <option value="certificates">Educational Certificates</option>
                <option value="photo">Passport Photo</option>
                <option value="birth_certificate">Birth Certificate</option>
                <option value="bank_details">Bank Details</option>
                <option value="other">Other</option>
            </select>
        </div>
        <div class="col-md-3">
            <input type="text" name="documents[${documentRowIndex}][name]" class="form-control" 
                   placeholder="e.g., Original NIC Copy">
        </div>
        <div class="col-md-4">
            <input type="file" name="documents[${documentRowIndex}][file]" class="form-control" 
                   accept=".pdf,.doc,.docx,.jpg,.jpeg,.png">
        </div>
        <div class="col-md-2 d-flex align-items-end">
            <button type="button" class="btn btn-outline-danger btn-sm" 
                    onclick="removeDocumentRow(this)">
                <i class="fas fa-times"></i>
            </button>
        </div>
    `;
    
    container.appendChild(newRow);
    documentRowIndex++;
    
    // Show remove button for all rows if we have more than one
    const removeButtons = container.querySelectorAll('.btn-outline-danger');
    if (removeButtons.length > 1) {
        removeButtons.forEach(btn => btn.style.display = 'block');
    }
}

function removeDocumentRow(button) {
    const row = button.closest('.document-upload-row');
    row.remove();
    
    // Hide remove button if only one row remains
    const container = document.getElementById('documentUploadSection');
    const removeButtons = container.querySelectorAll('.btn-outline-danger');
    if (removeButtons.length === 1) {
        removeButtons[0].style.display = 'none';
    }
}

// Auto-fill document name based on type
document.addEventListener('change', function(e) {
    if (e.target.name && e.target.name.includes('[type]')) {
        const row = e.target.closest('.document-upload-row');
        const nameInput = row.querySelector('input[type="text"]');
        const applicantName = '<?php echo htmlspecialchars($currentUser['full_name'] ?? ''); ?>';
        
        if (e.target.value && !nameInput.value) {
            const typeName = e.target.options[e.target.selectedIndex].text;
            nameInput.value = typeName + (applicantName ? ' - ' + applicantName : '');
        }
    }
});

// File validation
document.addEventListener('change', function(e) {
    if (e.target.type === 'file' && e.target.name && e.target.name.includes('[file]')) {
        const file = e.target.files[0];
        if (file) {
            const maxSize = 5 * 1024 * 1024; // 5MB
            const allowedTypes = ['pdf', 'doc', 'docx', 'jpg', 'jpeg', 'png'];
            const fileExtension = file.name.split('.').pop().toLowerCase();
            
            if (file.size > maxSize) {
                alert('File size exceeds maximum limit of 5MB');
                e.target.value = '';
                return;
            }
            
            if (!allowedTypes.includes(fileExtension)) {
                alert('File type not allowed. Please use: ' + allowedTypes.join(', '));
                e.target.value = '';
                return;
            }
        }
    }
});

// Save draft functionality
function saveDraft() {
    const formData = new FormData(document.getElementById('applicationForm'));
    formData.append('save_draft', '1');
    
    fetch('save_draft.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            alert('Draft saved successfully!');
        } else {
            alert('Failed to save draft: ' + (data.error || 'Unknown error'));
        }
    })
    .catch(error => {
        console.error('Error saving draft:', error);
        alert('Failed to save draft');
    });
}
</script>

<?php require_once '../includes/footer.php'; ?>