<?php
/**
 * Approve Employee Applications
 * Employee Recruitment Management System
 * RR Construction (Pvt) Ltd
 */

$page_title = 'Approve Applications';
$active_menu = 'applications';

require_once '../includes/header.php';
requirePermission(ROLE_PROJECT_MANAGER);

$currentUser = getCurrentUser();
$success = false;
$errors = [];

// Process approval/rejection
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!verifyCSRFToken($_POST['csrf_token'] ?? '')) {
        $errors[] = 'Invalid security token.';
    } else {
        $application_id = (int)($_POST['application_id'] ?? 0);
        $action = $_POST['action'] ?? '';
        $comments = cleanInput($_POST['comments'] ?? '');
        $approved_salary = !empty($_POST['approved_salary']) ? (float)$_POST['approved_salary'] : null;
        $start_date = $_POST['start_date'] ?? null;
        
        if ($application_id && in_array($action, ['approve', 'reject'])) {
            try {
                // Get application details
                $sql = "SELECT ja.*, er.site_project 
                        FROM job_applications ja 
                        JOIN employee_requests er ON ja.request_id = er.id 
                        WHERE ja.id = ?";
                $application = fetchOne($sql, [$application_id]);
                
                if ($application) {
                    // Check if user can approve this application
                    $can_approve = false;
                    if ($currentUser['role'] === ROLE_PROJECT_MANAGER) {
                        $can_approve = ($application['site_project'] === $currentUser['site_project']) && 
                                     in_array($application['status'], ['submitted', 'under_review']);
                    } elseif ($currentUser['role'] === ROLE_HR_MANAGER) {
                        $can_approve = $application['status'] === 'approved_by_pm';
                    } elseif ($currentUser['role'] === ROLE_ACCOUNTANT) {
                        $can_approve = $application['status'] === 'approved_by_hr';
                    }
                    
                    if ($can_approve) {
                        beginTransaction();
                        
                        if ($action === 'approve') {
                            // Determine next status
                            if ($currentUser['role'] === ROLE_PROJECT_MANAGER) {
                                $new_status = 'approved_by_pm';
                            } elseif ($currentUser['role'] === ROLE_HR_MANAGER) {
                                $new_status = 'approved_by_hr';
                            } elseif ($currentUser['role'] === ROLE_ACCOUNTANT) {
                                $new_status = 'hired';
                            }
                        } else {
                            $new_status = 'rejected';
                        }
                        
                        // Update application status
                        $update_data = [
                            'status' => $new_status,
                            'rejection_reason' => $action === 'reject' ? $comments : null
                        ];
                        
                        // Update salary if approved by accountant
                        if ($action === 'approve' && $currentUser['role'] === ROLE_ACCOUNTANT && $approved_salary) {
                            $update_data['expected_salary'] = $approved_salary;
                        }
                        
                        // FIXED: Use named parameters consistently
                        updateRecord('job_applications', $update_data, 'id = :id', ['id' => $application_id]);
                        
                        // Log approval
                        $approval_data = [
                            'application_id' => $application_id,
                            'approved_by' => $currentUser['id'],
                            'approval_level' => $currentUser['role'],
                            'action' => $action === 'approve' ? 'approved' : 'rejected',
                            'comments' => $comments,
                            'salary_approved' => $approved_salary,
                            'start_date_approved' => $start_date
                        ];
                        insertRecord('application_approval_logs', $approval_data);
                        
                        // Log activity
                        logActivity($currentUser['id'], $action, 'job_applications', $application_id);
                        
                        commit();
                        $success = true;
                        setFlashMessage("Application " . ($action === 'approve' ? 'approved' : 'rejected') . " successfully!", 'success');
                        
                        // Redirect to avoid resubmission
                        header('Location: ' . $_SERVER['PHP_SELF']);
                        exit();
                        
                    } else {
                        $errors[] = 'You do not have permission to approve this application.';
                    }
                } else {
                    $errors[] = 'Application not found.';
                }
            } catch (Exception $e) {
                rollback();
                error_log("Application approval error: " . $e->getMessage());
                $errors[] = 'Failed to process approval. Please try again.';
            }
        } else {
            $errors[] = 'Invalid request data.';
        }
    }
}

// Get pending applications for approval
$where_conditions = ['1=1'];
$params = [];

// Determine which applications this user can approve
if ($currentUser['role'] === ROLE_PROJECT_MANAGER) {
    $where_conditions[] = 'ja.status IN (?, ?)';
    $where_conditions[] = 'er.site_project = ?';
    $params[] = 'submitted';
    $params[] = 'under_review';
    $params[] = $currentUser['site_project'];
} elseif ($currentUser['role'] === ROLE_HR_MANAGER) {
    $where_conditions[] = 'ja.status = ?';
    $params[] = 'approved_by_pm';
} elseif ($currentUser['role'] === ROLE_ACCOUNTANT) {
    $where_conditions[] = 'ja.status = ?';
    $params[] = 'approved_by_hr';
}

$where_clause = implode(' AND ', $where_conditions);

$sql = "SELECT ja.*, er.request_number, er.site_project, u.full_name as submitted_by_name
        FROM job_applications ja
        LEFT JOIN employee_requests er ON ja.request_id = er.id
        LEFT JOIN users u ON ja.submitted_by = u.id
        WHERE $where_clause
        ORDER BY ja.submitted_at ASC";

$pending_applications = fetchAll($sql, $params);

// Set breadcrumbs
$breadcrumbs = [
    ['title' => 'Job Applications', 'url' => BASE_URL . 'employees/view_applications.php'],
    ['title' => 'Pending Approvals', 'url' => '']
];
?>

<div class="row mb-4">
    <div class="col-12">
        <div class="card">
            <div class="card-header d-flex justify-content-between align-items-center">
                <h5 class="mb-0">
                    <i class="fas fa-user-check me-2"></i>
                    Pending Application Approvals
                </h5>
                <div>
                    <span class="badge bg-warning text-dark me-2">
                        <?php echo count($pending_applications); ?> Pending
                    </span>
                    <span class="text-muted small">
                        <?php 
                        if ($currentUser['role'] === ROLE_PROJECT_MANAGER) {
                            echo 'Project Manager Level - ' . htmlspecialchars($currentUser['site_project']);
                        } elseif ($currentUser['role'] === ROLE_HR_MANAGER) {
                            echo 'HR Manager Level - Human Resources Review';
                        } elseif ($currentUser['role'] === ROLE_ACCOUNTANT) {
                            echo 'Accountant Level - Financial Approval';
                        }
                        ?>
                    </span>
                </div>
            </div>
            <div class="card-body">
                
                <?php if (!empty($errors)): ?>
                    <div class="alert alert-danger">
                        <i class="fas fa-exclamation-circle me-2"></i>
                        <ul class="mb-0">
                            <?php foreach ($errors as $error): ?>
                                <li><?php echo htmlspecialchars($error); ?></li>
                            <?php endforeach; ?>
                        </ul>
                    </div>
                <?php endif; ?>

                <?php if (empty($pending_applications)): ?>
                    <div class="text-center py-5">
                        <i class="fas fa-check-circle text-success" style="font-size: 4rem; opacity: 0.3;"></i>
                        <h4 class="mt-3 text-muted">No Pending Applications</h4>
                        <p class="text-muted">
                            All applications at your approval level have been processed.
                        </p>
                        <a href="view_applications.php" class="btn btn-outline-primary">
                            <i class="fas fa-list me-2"></i>View All Applications
                        </a>
                    </div>
                <?php else: ?>

                    <?php foreach ($pending_applications as $app): ?>
                    <div class="card mb-4">
                        <div class="card-header d-flex justify-content-between align-items-center">
                            <div>
                                <strong><?php echo htmlspecialchars($app['application_number']); ?></strong>
                                <span class="badge bg-info ms-2"><?php echo htmlspecialchars($app['request_number']); ?></span>
                                <span class="badge bg-secondary ms-1"><?php echo getStatusName($app['status'], 'application'); ?></span>
                            </div>
                            <small class="text-muted"><?php echo getRelativeTime($app['submitted_at']); ?></small>
                        </div>
                        <div class="card-body">
                            <!-- Applicant Information -->
                            <div class="row mb-3">
                                <div class="col-md-6">
                                    <h6 class="text-primary mb-2">
                                        <i class="fas fa-user me-2"></i>Applicant Information
                                    </h6>
                                    <table class="table table-sm table-borderless">
                                        <tr>
                                            <td width="120"><strong>Name:</strong></td>
                                            <td><?php echo htmlspecialchars($app['full_name_english']); ?></td>
                                        </tr>
                                        <tr>
                                            <td><strong>NIC:</strong></td>
                                            <td><?php echo htmlspecialchars($app['nic_number']); ?></td>
                                        </tr>
                                        <tr>
                                            <td><strong>Age/Gender:</strong></td>
                                            <td><?php echo $app['age']; ?> years, <?php echo ucfirst($app['gender']); ?></td>
                                        </tr>
                                        <tr>
                                            <td><strong>Phone:</strong></td>
                                            <td><?php echo htmlspecialchars($app['phone_mobile']); ?></td>
                                        </tr>
                                        <tr>
                                            <td><strong>Email:</strong></td>
                                            <td><?php echo htmlspecialchars($app['email'] ?: 'Not provided'); ?></td>
                                        </tr>
                                    </table>
                                </div>
                                <div class="col-md-6">
                                    <h6 class="text-primary mb-2">
                                        <i class="fas fa-briefcase me-2"></i>Position & Project
                                    </h6>
                                    <table class="table table-sm table-borderless">
                                        <tr>
                                            <td width="120"><strong>Position:</strong></td>
                                            <td><?php echo htmlspecialchars($app['position_applied']); ?></td>
                                        </tr>
                                        <tr>
                                            <td><strong>Project:</strong></td>
                                            <td><?php echo htmlspecialchars($app['site_project']); ?></td>
                                        </tr>
                                        <tr>
                                            <td><strong>Expected Salary:</strong></td>
                                            <td><?php echo $app['expected_salary'] ? formatCurrencyAmount($app['expected_salary']) : 'Not specified'; ?></td>
                                        </tr>
                                        <tr>
                                            <td><strong>Available From:</strong></td>
                                            <td><?php echo $app['available_start_date'] ? formatDisplayDate($app['available_start_date']) : 'Immediately'; ?></td>
                                        </tr>
                                        <tr>
                                            <td><strong>Submitted By:</strong></td>
                                            <td><?php echo htmlspecialchars($app['submitted_by_name']); ?></td>
                                        </tr>
                                    </table>
                                </div>
                            </div>
                            
                            <!-- Education & Experience -->
                            <div class="row mb-3">
                                <div class="col-md-6">
                                    <h6 class="text-secondary mb-2">
                                        <i class="fas fa-graduation-cap me-2"></i>Education
                                    </h6>
                                    <div class="small">
                                        <?php if ($app['high_school']): ?>
                                        <div><strong>School:</strong> <?php echo htmlspecialchars($app['high_school']); ?> 
                                        (<?php echo $app['high_school_year_completed'] ?: 'Year not specified'; ?>)</div>
                                        <?php endif; ?>
                                        
                                        <?php if ($app['ol_results']): ?>
                                        <div><strong>O/L:</strong> <?php echo htmlspecialchars($app['ol_results']); ?></div>
                                        <?php endif; ?>
                                        
                                        <?php if ($app['al_results']): ?>
                                        <div><strong>A/L:</strong> <?php echo htmlspecialchars($app['al_results']); ?></div>
                                        <?php endif; ?>
                                        
                                        <?php if ($app['college_university']): ?>
                                        <div><strong>Higher Education:</strong> <?php echo htmlspecialchars($app['degree_diploma']); ?> 
                                        from <?php echo htmlspecialchars($app['college_university']); ?></div>
                                        <?php endif; ?>
                                    </div>
                                </div>
                                <div class="col-md-6">
                                    <h6 class="text-secondary mb-2">
                                        <i class="fas fa-tools me-2"></i>Experience & Skills
                                    </h6>
                                    <div class="small">
                                        <?php if ($app['currently_employed']): ?>
                                        <div><strong>Current Job:</strong> <?php echo htmlspecialchars($app['current_position']); ?> 
                                        at <?php echo htmlspecialchars($app['current_employer']); ?></div>
                                        <?php if ($app['current_salary']): ?>
                                        <div><strong>Current Salary:</strong> <?php echo formatCurrencyAmount($app['current_salary']); ?></div>
                                        <?php endif; ?>
                                        <?php endif; ?>
                                        
                                        <?php if ($app['special_skills']): ?>
                                        <div><strong>Special Skills:</strong> <?php echo htmlspecialchars($app['special_skills']); ?></div>
                                        <?php endif; ?>
                                        
                                        <?php if ($app['computer_skills']): ?>
                                        <div><strong>Computer Skills:</strong> <?php echo htmlspecialchars($app['computer_skills']); ?></div>
                                        <?php endif; ?>
                                        
                                        <?php if ($app['language_skills']): ?>
                                        <div><strong>Languages:</strong> <?php echo htmlspecialchars($app['language_skills']); ?></div>
                                        <?php endif; ?>
                                    </div>
                                </div>
                            </div>
                            
                            <!-- Approval Actions -->
                            <div class="border-top pt-3">
                                <div class="row">
                                    <!-- Approval Form -->
                                    <div class="col-md-8">
                                        <form method="POST" class="approval-form">
                                            <input type="hidden" name="csrf_token" value="<?php echo generateCSRFToken(); ?>">
                                            <input type="hidden" name="application_id" value="<?php echo $app['id']; ?>">
                                            <input type="hidden" name="action" value="approve">
                                            
                                            <div class="row">
                                                <div class="col-md-6">
                                                    <label class="form-label">Comments</label>
                                                    <input type="text" name="comments" class="form-control form-control-sm" 
                                                           placeholder="Approval comments...">
                                                </div>
                                                <?php if ($currentUser['role'] === ROLE_ACCOUNTANT): ?>
                                                <div class="col-md-3">
                                                    <label class="form-label">Approved Salary <span class="text-danger">*</span></label>
                                                    <input type="number" name="approved_salary" class="form-control form-control-sm" 
                                                           step="0.01" value="<?php echo $app['expected_salary']; ?>" required>
                                                </div>
                                                <div class="col-md-3">
                                                    <label class="form-label">Start Date</label>
                                                    <input type="date" name="start_date" class="form-control form-control-sm">
                                                </div>
                                                <?php else: ?>
                                                <div class="col-md-6">
                                                    <label class="form-label">&nbsp;</label>
                                                    <div></div>
                                                </div>
                                                <?php endif; ?>
                                            </div>
                                            
                                            <div class="mt-2">
                                                <button type="submit" class="btn btn-success btn-sm me-2" 
                                                        onclick="return confirm('Are you sure you want to approve this application?')">
                                                    <i class="fas fa-check"></i> Approve
                                                </button>
                                            </div>
                                        </form>
                                    </div>
                                    
                                    <!-- Rejection Form -->
                                    <div class="col-md-4">
                                        <form method="POST" class="rejection-form">
                                            <input type="hidden" name="csrf_token" value="<?php echo generateCSRFToken(); ?>">
                                            <input type="hidden" name="application_id" value="<?php echo $app['id']; ?>">
                                            <input type="hidden" name="action" value="reject">
                                            
                                            <label class="form-label">Rejection Reason <span class="text-danger">*</span></label>
                                            <input type="text" name="comments" class="form-control form-control-sm" 
                                                   placeholder="Reason for rejection..." required>
                                            
                                            <div class="mt-2">
                                                <button type="submit" class="btn btn-outline-danger btn-sm" 
                                                        onclick="return confirm('Are you sure you want to reject this application?')">
                                                    <i class="fas fa-times"></i> Reject
                                                </button>
                                            </div>
                                        </form>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    <?php endforeach; ?>

                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<!-- Recent Approval History -->
<div class="card mt-4">
    <div class="card-header">
        <i class="fas fa-history me-2"></i>Recent Approval History
    </div>
    <div class="card-body">
        <?php
        try {
            $history_sql = "SELECT ja.application_number, ja.full_name_english, ja.position_applied, 
                                   aal.action, aal.comments, aal.approved_at
                           FROM application_approval_logs aal
                           JOIN job_applications ja ON aal.application_id = ja.id
                           WHERE aal.approved_by = ?
                           ORDER BY aal.approved_at DESC
                           LIMIT 10";
            $history = fetchAll($history_sql, [$currentUser['id']]);
            
            if ($history):
        ?>
        <div class="table-responsive">
            <table class="table table-sm">
                <thead>
                    <tr>
                        <th>Application</th>
                        <th>Applicant</th>
                        <th>Position</th>
                        <th>Action</th>
                        <th>Comments</th>
                        <th>Date</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($history as $item): ?>
                    <tr>
                        <td><?php echo htmlspecialchars($item['application_number']); ?></td>
                        <td><?php echo htmlspecialchars($item['full_name_english']); ?></td>
                        <td><?php echo htmlspecialchars($item['position_applied']); ?></td>
                        <td>
                            <span class="badge <?php echo $item['action'] === 'approved' ? 'bg-success' : 'bg-danger'; ?>">
                                <?php echo ucfirst($item['action']); ?>
                            </span>
                        </td>
                        <td><?php echo htmlspecialchars($item['comments'] ?: '-'); ?></td>
                        <td><?php echo formatDisplayDateTime($item['approved_at']); ?></td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
        <?php else: ?>
        <p class="text-muted">No approval history found.</p>
        <?php 
            endif;
        } catch (Exception $e) {
            echo '<p class="text-muted">Unable to load approval history.</p>';
        }
        ?>
    </div>
</div>

<script>
// Form validation for salary approval
document.addEventListener('DOMContentLoaded', function() {
    const forms = document.querySelectorAll('.approval-form');
    forms.forEach(function(form) {
        form.addEventListener('submit', function(e) {
            const salaryField = form.querySelector('input[name="approved_salary"]');
            if (salaryField && salaryField.hasAttribute('required')) {
                const salary = parseFloat(salaryField.value);
                if (!salary || salary <= 0) {
                    e.preventDefault();
                    showAlert('Please enter a valid approved salary amount.', 'warning');
                    salaryField.focus();
                    return false;
                }
            }
        });
    });
    
    // Auto-refresh every 60 seconds if there are pending applications
    <?php if (!empty($pending_applications)): ?>
    setInterval(function() {
        // Only refresh if no forms are being filled
        const activeElement = document.activeElement;
        if (!activeElement || activeElement.tagName !== 'INPUT') {
            location.reload();
        }
    }, 60000);
    <?php endif; ?>
});
</script>

<?php require_once '../includes/footer.php'; ?>