<?php
/**
 * Check Position Availability (AJAX Endpoint)
 * Employee Recruitment Management System
 * RR Construction (Pvt) Ltd
 */

define('HRM_ACCESS', true);

// Start session if not already started
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Include required files
require_once '../config/constants.php';
require_once '../config/database.php';
require_once '../includes/functions.php';

// Set content type to JSON
header('Content-Type: application/json');

// Check if user is logged in
if (!isLoggedIn()) {
    http_response_code(401);
    echo json_encode(['success' => false, 'error' => 'Unauthorized']);
    exit;
}

// Check if this is a POST request
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'error' => 'Method not allowed']);
    exit;
}

// Verify CSRF token
if (!verifyCSRFToken($_POST['csrf_token'] ?? '')) {
    http_response_code(403);
    echo json_encode(['success' => false, 'error' => 'Invalid security token']);
    exit;
}

// Validate category_id
if (empty($_POST['category_id']) || !is_numeric($_POST['category_id'])) {
    http_response_code(400);
    echo json_encode(['success' => false, 'error' => 'Invalid category ID']);
    exit;
}

$categoryId = (int)$_POST['category_id'];

try {
    // Get current availability for the position
    $sql = "SELECT ec.id, ec.position_title, ec.quantity,
                   COUNT(CASE WHEN ja.status = 'hired' THEN 1 END) as hired_count,
                   (ec.quantity - COUNT(CASE WHEN ja.status = 'hired' THEN 1 END)) as available_positions,
                   er.request_number, er.site_project
            FROM employee_categories ec
            JOIN employee_requests er ON ec.request_id = er.id
            LEFT JOIN job_applications ja ON ec.id = ja.category_id
            WHERE ec.id = ?
            GROUP BY ec.id";
    
    $result = fetchOne($sql, [$categoryId]);
    
    if (!$result) {
        http_response_code(404);
        echo json_encode(['success' => false, 'error' => 'Position not found']);
        exit;
    }
    
    // Return availability data
    echo json_encode([
        'success' => true,
        'category_id' => $result['id'],
        'position_title' => $result['position_title'],
        'quantity' => (int)$result['quantity'],
        'hired_count' => (int)$result['hired_count'],
        'available_positions' => (int)$result['available_positions'],
        'request_number' => $result['request_number'],
        'site_project' => $result['site_project']
    ]);
    
} catch (Exception $e) {
    error_log("Error checking availability: " . $e->getMessage());
    http_response_code(500);
    echo json_encode(['success' => false, 'error' => 'Server error']);
}
?>