<?php
/**
 * Edit Employee Application
 * Employee Recruitment Management System
 * RR Construction (Pvt) Ltd
 */

$page_title = 'Edit Application';
$active_menu = 'applications';

require_once '../includes/header.php';
requirePermission(ROLE_SITE_ADMIN);

$currentUser = getCurrentUser();
$application_id = (int)($_GET['id'] ?? 0);
$errors = [];
$success = false;

// Get application details
$application = null;
if ($application_id) {
    try {
        $sql = "SELECT ja.*, er.request_number, er.site_project, er.status as request_status,
                       ec.position_title, ec.category, ec.quantity, ec.salary_range_min, ec.salary_range_max,
                       u.full_name as submitted_by_name
                FROM job_applications ja
                JOIN employee_requests er ON ja.request_id = er.id
                JOIN employee_categories ec ON ja.category_id = ec.id
                JOIN users u ON ja.submitted_by = u.id
                WHERE ja.id = ? AND ja.submitted_by = ?";
        
        $application = fetchOne($sql, [$application_id, $currentUser['id']]);
    } catch (Exception $e) {
        error_log("Error fetching application: " . $e->getMessage());
    }
}

if (!$application) {
    setFlashMessage('Application not found or access denied.', 'danger');
    header('Location: view_applications.php');
    exit;
}

// Check if application can be edited
if ($application['status'] !== 'submitted') {
    setFlashMessage('Application cannot be edited. Only submitted applications can be modified.', 'warning');
    header('Location: view_application.php?id=' . $application_id);
    exit;
}

// Get available categories for this request
$available_categories = [];
try {
    $sql = "SELECT ec.id, ec.position_title, ec.category, ec.quantity,
                   COUNT(ja.id) as applications_count,
                   COUNT(CASE WHEN ja.status = 'hired' THEN 1 END) as hired_count,
                   (ec.quantity - COUNT(CASE WHEN ja.status = 'hired' THEN 1 END)) as available_positions
            FROM employee_categories ec
            LEFT JOIN job_applications ja ON ec.id = ja.category_id AND ja.id != ?
            WHERE ec.request_id = ?
            GROUP BY ec.id
            HAVING available_positions > 0 OR ec.id = ?
            ORDER BY ec.position_title";
    $available_categories = fetchAll($sql, [$application_id, $application['request_id'], $application['category_id']]);
} catch (Exception $e) {
    error_log("Error fetching categories: " . $e->getMessage());
}

// Process form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!verifyCSRFToken($_POST['csrf_token'] ?? '')) {
        $errors[] = 'Invalid security token.';
    } else {
        // Validate required fields
        $required = [
            'category_id', 'full_name_english', 'nic_number', 
            'date_of_birth', 'gender', 'marital_status', 'permanent_address',
            'phone_mobile', 'position_applied'
        ];
        
        $validation_errors = validateRequired($_POST, $required);
        
        // Check if position is still available (if changing category)
        if (!empty($_POST['category_id']) && $_POST['category_id'] != $application['category_id']) {
            $availabilityCheck = fetchOne(
                "SELECT ec.quantity,
                        COUNT(CASE WHEN ja.status = 'hired' THEN 1 END) as hired_count,
                        (ec.quantity - COUNT(CASE WHEN ja.status = 'hired' THEN 1 END)) as available_positions
                 FROM employee_categories ec
                 LEFT JOIN job_applications ja ON ec.id = ja.category_id
                 WHERE ec.id = ?
                 GROUP BY ec.id",
                [(int)$_POST['category_id']]
            );
            
            if (!$availabilityCheck || $availabilityCheck['available_positions'] <= 0) {
                $validation_errors['category_id'] = 'This position is no longer available for applications';
            }
        }
        
        // Check for duplicate NIC applications for the same category (excluding current application)
        if (!empty($_POST['nic_number']) && !empty($_POST['category_id'])) {
            $duplicateCheck = fetchOne(
                "SELECT ja.id FROM job_applications ja 
                 WHERE ja.nic_number = ? AND ja.category_id = ? AND ja.status != 'rejected' AND ja.id != ?",
                [$_POST['nic_number'], (int)$_POST['category_id'], $application_id]
            );
            
            if ($duplicateCheck) {
                $validation_errors['nic_number'] = 'An application with this NIC already exists for this position';
            }
        }
        
        // Validate NIC
        if (!empty($_POST['nic_number']) && !validateNIC($_POST['nic_number'])) {
            $validation_errors['nic_number'] = 'Invalid NIC number format';
        }
        
        // Validate email if provided
        if (!empty($_POST['email']) && !validateEmail($_POST['email'])) {
            $validation_errors['email'] = 'Invalid email address';
        }
        
        // Validate phone
        if (!empty($_POST['phone_mobile']) && !validatePhone($_POST['phone_mobile'])) {
            $validation_errors['phone_mobile'] = 'Invalid phone number format';
        }
        
        if (empty($validation_errors)) {
            try {
                beginTransaction();
                
                // Calculate age
                $age = calculateAge($_POST['date_of_birth']);
                
                // Prepare update data
                $update_data = [
                    'category_id' => (int)$_POST['category_id'],
                    
                    // Personal Information
                    'full_name_english' => cleanInput($_POST['full_name_english']),
                    'full_name_sinhala' => cleanInput($_POST['full_name_sinhala'] ?? ''),
                    'nic_number' => cleanInput($_POST['nic_number']),
                    'date_of_birth' => $_POST['date_of_birth'],
                    'age' => $age,
                    'gender' => $_POST['gender'],
                    'marital_status' => $_POST['marital_status'],
                    'nationality' => cleanInput($_POST['nationality'] ?? 'Sri Lankan'),
                    'religion' => cleanInput($_POST['religion'] ?? ''),
                    'blood_group' => cleanInput($_POST['blood_group'] ?? ''),
                    'district' => cleanInput($_POST['district'] ?? 'Gampaha'),
                    'police_station' => cleanInput($_POST['police_station'] ?? ''),
                    'medical_condition' => cleanInput($_POST['medical_condition'] ?? ''),
                    
                    // Contact Information
                    'permanent_address' => cleanInput($_POST['permanent_address']),
                    'current_address' => cleanInput($_POST['current_address'] ?? ''),
                    'phone_mobile' => cleanInput($_POST['phone_mobile']),
                    'phone_home' => cleanInput($_POST['phone_home'] ?? ''),
                    'email' => cleanInput($_POST['email'] ?? ''),
                    'emergency_contact_name' => cleanInput($_POST['emergency_contact_name'] ?? ''),
                    'emergency_contact_phone' => cleanInput($_POST['emergency_contact_phone'] ?? ''),
                    'emergency_contact_relationship' => cleanInput($_POST['emergency_contact_relationship'] ?? ''),
                    
                    // Employment Information
                    'position_applied' => cleanInput($_POST['position_applied']),
                    'expected_salary' => !empty($_POST['expected_salary']) ? (float)$_POST['expected_salary'] : null,
                    'available_start_date' => $_POST['available_start_date'] ?? null,
                    'transportation_method' => $_POST['transportation_method'] ?? null,
                    'driving_license_no' => cleanInput($_POST['driving_license_no'] ?? ''),
                    'driving_license_type' => $_POST['driving_license_type'] ?? 'none',
                    
                    // Current Employment
                    'currently_employed' => isset($_POST['currently_employed']) ? 1 : 0,
                    'current_employer' => cleanInput($_POST['current_employer'] ?? ''),
                    'current_position' => cleanInput($_POST['current_position'] ?? ''),
                    'current_salary' => !empty($_POST['current_salary']) ? (float)$_POST['current_salary'] : null,
                    'employment_start_date' => $_POST['employment_start_date'] ?? null,
                    'reason_for_leaving' => cleanInput($_POST['reason_for_leaving'] ?? ''),
                    
                    // Previous Employment
                    'prev1_employer' => cleanInput($_POST['prev1_employer'] ?? ''),
                    'prev1_position' => cleanInput($_POST['prev1_position'] ?? ''),
                    'prev1_duration' => cleanInput($_POST['prev1_duration'] ?? ''),
                    'prev1_salary' => !empty($_POST['prev1_salary']) ? (float)$_POST['prev1_salary'] : null,
                    'prev1_reason_leaving' => cleanInput($_POST['prev1_reason_leaving'] ?? ''),
                    
                    'prev2_employer' => cleanInput($_POST['prev2_employer'] ?? ''),
                    'prev2_position' => cleanInput($_POST['prev2_position'] ?? ''),
                    'prev2_duration' => cleanInput($_POST['prev2_duration'] ?? ''),
                    'prev2_salary' => !empty($_POST['prev2_salary']) ? (float)$_POST['prev2_salary'] : null,
                    'prev2_reason_leaving' => cleanInput($_POST['prev2_reason_leaving'] ?? ''),
                    
                    // Education
                    'elementary_school' => cleanInput($_POST['elementary_school'] ?? ''),
                    'elementary_year_completed' => !empty($_POST['elementary_year_completed']) ? (int)$_POST['elementary_year_completed'] : null,
                    'high_school' => cleanInput($_POST['high_school'] ?? ''),
                    'high_school_year_completed' => !empty($_POST['high_school_year']) ? (int)$_POST['high_school_year'] : null,
                    'ol_results' => cleanInput($_POST['ol_results'] ?? ''),
                    'al_results' => cleanInput($_POST['al_results'] ?? ''),
                    'college_university' => cleanInput($_POST['college_university'] ?? ''),
                    'degree_diploma' => cleanInput($_POST['degree_diploma'] ?? ''),
                    'graduation_year' => !empty($_POST['graduation_year']) ? (int)$_POST['graduation_year'] : null,
                    'gpa_class' => cleanInput($_POST['gpa_class'] ?? ''),
                    'other_education' => cleanInput($_POST['other_education'] ?? ''),
                    'certifications' => cleanInput($_POST['certifications'] ?? ''),
                    'training_courses' => cleanInput($_POST['training_courses'] ?? ''),
                    
                    // Banking
                    'bank_name' => cleanInput($_POST['bank_name'] ?? ''),
                    'bank_branch' => cleanInput($_POST['bank_branch'] ?? ''),
                    'account_number' => cleanInput($_POST['account_number'] ?? ''),
                    'account_holder_name' => cleanInput($_POST['account_holder_name'] ?? ''),
                    
                    // Skills and References
                    'special_skills' => cleanInput($_POST['special_skills'] ?? ''),
                    'computer_skills' => cleanInput($_POST['computer_skills'] ?? ''),
                    'language_skills' => cleanInput($_POST['language_skills'] ?? ''),
                    'hobbies' => cleanInput($_POST['hobbies'] ?? ''),
                    'employee_references' => cleanInput($_POST['employee_references'] ?? ''),
                    
                    // System fields
                    'updated_at' => date('Y-m-d H:i:s')
                ];
                
                // Store old values for logging
                $old_values = $application;
                
                // Update application
                updateRecord('job_applications', $update_data, 'id = ?', [$application_id]);
                
                // Log activity
                logActivity($currentUser['id'], 'update', 'job_applications', $application_id, $old_values, $update_data);
                
                commit();
                $success = true;
                setFlashMessage('Application updated successfully!', 'success');
                
                // Redirect to view page
                header('Location: view_application.php?id=' . $application_id);
                exit;
                
            } catch (Exception $e) {
                rollback();
                error_log("Error updating application: " . $e->getMessage());
                $errors[] = 'Failed to update application. Please try again.';
            }
        } else {
            $errors = array_merge($errors, array_values($validation_errors));
        }
    }
}
?>

<div class="row mb-4">
    <div class="col-12">
        <div class="d-flex justify-content-between align-items-center">
            <div>
                <h4 class="mb-0">Edit Application</h4>
                <small class="text-muted">
                    <?php echo htmlspecialchars($application['application_number']); ?> - 
                    <?php echo htmlspecialchars($application['full_name_english']); ?>
                </small>
            </div>
            <div class="btn-group">
                <a href="view_applications.php" class="btn btn-outline-secondary">
                    <i class="fas fa-arrow-left"></i> Back to Applications
                </a>
                <a href="view_application.php?id=<?php echo $application_id; ?>" class="btn btn-outline-primary">
                    <i class="fas fa-eye"></i> View Application
                </a>
            </div>
        </div>
    </div>
</div>

<!-- Application Status Info -->
<div class="card mb-3">
    <div class="card-header bg-info text-white">
        <i class="fas fa-info-circle me-2"></i>Application Information
    </div>
    <div class="card-body">
        <div class="row">
            <div class="col-md-6">
                <table class="table table-sm table-borderless">
                    <tr>
                        <td width="150"><strong>Application Number:</strong></td>
                        <td><?php echo htmlspecialchars($application['application_number']); ?></td>
                    </tr>
                    <tr>
                        <td><strong>Request Number:</strong></td>
                        <td><?php echo htmlspecialchars($application['request_number']); ?></td>
                    </tr>
                    <tr>
                        <td><strong>Project:</strong></td>
                        <td><?php echo htmlspecialchars($application['site_project']); ?></td>
                    </tr>
                </table>
            </div>
            <div class="col-md-6">
                <table class="table table-sm table-borderless">
                    <tr>
                        <td width="150"><strong>Current Status:</strong></td>
                        <td><?php echo getStatusBadge($application['status'], 'application'); ?></td>
                    </tr>
                    <tr>
                        <td><strong>Submitted:</strong></td>
                        <td><?php echo formatDisplayDateTime($application['submitted_at']); ?></td>
                    </tr>
                    <tr>
                        <td><strong>Last Updated:</strong></td>
                        <td><?php echo formatDisplayDateTime($application['updated_at']); ?></td>
                    </tr>
                </table>
            </div>
        </div>
        <div class="alert alert-warning mb-0">
            <i class="fas fa-exclamation-triangle me-2"></i>
            <strong>Note:</strong> You can only edit applications that are in "Submitted" status. 
            Once approved or under review, modifications are not allowed.
        </div>
    </div>
</div>

<?php if (!empty($errors)): ?>
<div class="alert alert-danger">
    <ul class="mb-0">
        <?php foreach ($errors as $error): ?>
            <li><?php echo htmlspecialchars($error); ?></li>
        <?php endforeach; ?>
    </ul>
</div>
<?php endif; ?>

<!-- Position Availability Summary -->
<div class="card mb-3">
    <div class="card-header bg-success text-white">
        <i class="fas fa-chart-pie me-2"></i>Available Positions for This Request
    </div>
    <div class="card-body">
        <div class="row">
            <?php foreach ($available_categories as $cat): ?>
            <div class="col-md-6 mb-3">
                <div class="border rounded p-3 <?php echo $cat['id'] == $application['category_id'] ? 'border-primary bg-light' : ''; ?>">
                    <h6 class="text-primary mb-2">
                        <?php echo htmlspecialchars($cat['position_title']); ?>
                        <?php if ($cat['id'] == $application['category_id']): ?>
                        <span class="badge bg-primary">Current</span>
                        <?php endif; ?>
                    </h6>
                    <div class="d-flex justify-content-between align-items-center">
                        <span class="small text-muted"><?php echo htmlspecialchars($cat['category']); ?></span>
                        <span class="badge bg-success">
                            <?php echo $cat['available_positions']; ?> available
                            <small class="text-white-50">
                                (<?php echo $cat['hired_count']; ?>/<?php echo $cat['quantity']; ?> filled)
                            </small>
                        </span>
                    </div>
                </div>
            </div>
            <?php endforeach; ?>
        </div>
    </div>
</div>

<form method="POST" id="editApplicationForm">
    <input type="hidden" name="csrf_token" value="<?php echo generateCSRFToken(); ?>">
    
    <!-- Position Selection -->
    <div class="card mb-3">
        <div class="card-header">Position Selection</div>
        <div class="card-body">
            <div class="row">
                <div class="col-md-6">
                    <label class="form-label">Position Applied *</label>
                    <select name="category_id" class="form-control" required onchange="updatePositionTitle()">
                        <option value="">Select position</option>
                        <?php foreach ($available_categories as $cat): ?>
                            <option value="<?php echo $cat['id']; ?>" 
                                    data-position="<?php echo htmlspecialchars($cat['position_title']); ?>"
                                    data-available="<?php echo $cat['available_positions']; ?>"
                                    <?php echo $cat['id'] == $application['category_id'] ? 'selected' : ''; ?>>
                                <?php echo htmlspecialchars($cat['position_title']); ?> 
                                (<?php echo $cat['available_positions']; ?> available)
                            </option>
                        <?php endforeach; ?>
                    </select>
                    <input type="hidden" name="position_applied" id="position_applied" value="<?php echo htmlspecialchars($application['position_applied']); ?>">
                </div>
                <div class="col-md-6">
                    <label class="form-label">Expected Salary</label>
                    <input type="number" name="expected_salary" class="form-control" step="0.01"
                           value="<?php echo $application['expected_salary']; ?>">
                </div>
            </div>
        </div>
    </div>
    
    <!-- Personal Information -->
    <div class="card mb-3">
        <div class="card-header">Personal Information</div>
        <div class="card-body">
            <div class="row mb-3">
                <div class="col-md-6">
                    <label class="form-label">Full Name (English) *</label>
                    <input type="text" name="full_name_english" class="form-control" required
                           value="<?php echo htmlspecialchars($application['full_name_english']); ?>">
                </div>
                <div class="col-md-6">
                    <label class="form-label">Full Name (Sinhala)</label>
                    <input type="text" name="full_name_sinhala" class="form-control"
                           value="<?php echo htmlspecialchars($application['full_name_sinhala']); ?>">
                </div>
            </div>
            
            <div class="row mb-3">
                <div class="col-md-4">
                    <label class="form-label">NIC Number *</label>
                    <input type="text" name="nic_number" class="form-control" required
                           value="<?php echo htmlspecialchars($application['nic_number']); ?>">
                    <small class="text-muted">This will be checked for duplicates</small>
                </div>
                <div class="col-md-4">
                    <label class="form-label">Date of Birth *</label>
                    <input type="date" name="date_of_birth" class="form-control" required
                           value="<?php echo $application['date_of_birth']; ?>">
                </div>
                <div class="col-md-4">
                    <label class="form-label">Gender *</label>
                    <select name="gender" class="form-control" required>
                        <option value="">Select gender</option>
                        <option value="male" <?php echo $application['gender'] === 'male' ? 'selected' : ''; ?>>Male</option>
                        <option value="female" <?php echo $application['gender'] === 'female' ? 'selected' : ''; ?>>Female</option>
                    </select>
                </div>
            </div>
            
            <div class="row mb-3">
                <div class="col-md-3">
                    <label class="form-label">Marital Status *</label>
                    <select name="marital_status" class="form-control" required>
                        <option value="">Select status</option>
                        <option value="single" <?php echo $application['marital_status'] === 'single' ? 'selected' : ''; ?>>Single</option>
                        <option value="married" <?php echo $application['marital_status'] === 'married' ? 'selected' : ''; ?>>Married</option>
                        <option value="divorced" <?php echo $application['marital_status'] === 'divorced' ? 'selected' : ''; ?>>Divorced</option>
                        <option value="widowed" <?php echo $application['marital_status'] === 'widowed' ? 'selected' : ''; ?>>Widowed</option>
                    </select>
                </div>
                <div class="col-md-3">
                    <label class="form-label">Nationality</label>
                    <input type="text" name="nationality" class="form-control" 
                           value="<?php echo htmlspecialchars($application['nationality']); ?>">
                </div>
                <div class="col-md-3">
                    <label class="form-label">Religion</label>
                    <input type="text" name="religion" class="form-control"
                           value="<?php echo htmlspecialchars($application['religion']); ?>">
                </div>
                <div class="col-md-3">
                    <label class="form-label">Blood Group</label>
                    <select name="blood_group" class="form-control">
                        <option value="">Select blood group</option>
                        <option value="A+" <?php echo $application['blood_group'] === 'A+' ? 'selected' : ''; ?>>A+</option>
                        <option value="A-" <?php echo $application['blood_group'] === 'A-' ? 'selected' : ''; ?>>A-</option>
                        <option value="B+" <?php echo $application['blood_group'] === 'B+' ? 'selected' : ''; ?>>B+</option>
                        <option value="B-" <?php echo $application['blood_group'] === 'B-' ? 'selected' : ''; ?>>B-</option>
                        <option value="AB+" <?php echo $application['blood_group'] === 'AB+' ? 'selected' : ''; ?>>AB+</option>
                        <option value="AB-" <?php echo $application['blood_group'] === 'AB-' ? 'selected' : ''; ?>>AB-</option>
                        <option value="O+" <?php echo $application['blood_group'] === 'O+' ? 'selected' : ''; ?>>O+</option>
                        <option value="O-" <?php echo $application['blood_group'] === 'O-' ? 'selected' : ''; ?>>O-</option>
                    </select>
                </div>
            </div>
            
            <div class="row mb-3">
                <div class="col-md-4">
                    <label class="form-label">District</label>
                    <input type="text" name="district" class="form-control" 
                           value="<?php echo htmlspecialchars($application['district']); ?>">
                </div>
                <div class="col-md-4">
                    <label class="form-label">Nearest Police Station</label>
                    <input type="text" name="police_station" class="form-control" 
                           value="<?php echo htmlspecialchars($application['police_station']); ?>">
                </div>
                <div class="col-md-4">
                    <label class="form-label">Medical Condition</label>
                    <textarea name="medical_condition" class="form-control" rows="2" 
                              placeholder="Any medical conditions or disabilities"><?php echo htmlspecialchars($application['medical_condition']); ?></textarea>
                </div>
            </div>
            
            <div class="row mb-3">
                <div class="col-md-4">
                    <label class="form-label">Driving License Number</label>
                    <input type="text" name="driving_license_no" class="form-control"
                           value="<?php echo htmlspecialchars($application['driving_license_no']); ?>">
                </div>
                <div class="col-md-4">
                    <label class="form-label">License Type</label>
                    <select name="driving_license_type" class="form-control">
                        <option value="none" <?php echo $application['driving_license_type'] === 'none' ? 'selected' : ''; ?>>No License</option>
                        <option value="light_vehicle" <?php echo $application['driving_license_type'] === 'light_vehicle' ? 'selected' : ''; ?>>Light Vehicle</option>
                        <option value="heavy_vehicle" <?php echo $application['driving_license_type'] === 'heavy_vehicle' ? 'selected' : ''; ?>>Heavy Vehicle</option>
                        <option value="both" <?php echo $application['driving_license_type'] === 'both' ? 'selected' : ''; ?>>Both</option>
                    </select>
                </div>
                <div class="col-md-4">
                    <label class="form-label">Language Skills</label>
                    <textarea name="language_skills" class="form-control" rows="2" 
                              placeholder="e.g., Sinhala (Native), English (Good)"><?php echo htmlspecialchars($application['language_skills']); ?></textarea>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Contact Information -->
    <div class="card mb-3">
        <div class="card-header">Contact Information</div>
        <div class="card-body">
            <div class="row mb-3">
                <div class="col-md-6">
                    <label class="form-label">Permanent Address *</label>
                    <textarea name="permanent_address" class="form-control" rows="2" required><?php echo htmlspecialchars($application['permanent_address']); ?></textarea>
                </div>
                <div class="col-md-6">
                    <label class="form-label">Current Address</label>
                    <textarea name="current_address" class="form-control" rows="2"><?php echo htmlspecialchars($application['current_address']); ?></textarea>
                </div>
            </div>
            
            <div class="row mb-3">
                <div class="col-md-4">
                    <label class="form-label">Mobile Phone *</label>
                    <input type="tel" name="phone_mobile" class="form-control" required
                           value="<?php echo htmlspecialchars($application['phone_mobile']); ?>">
                </div>
                <div class="col-md-4">
                    <label class="form-label">Home Phone</label>
                    <input type="tel" name="phone_home" class="form-control"
                           value="<?php echo htmlspecialchars($application['phone_home']); ?>">
                </div>
                <div class="col-md-4">
                    <label class="form-label">Email</label>
                    <input type="email" name="email" class="form-control"
                           value="<?php echo htmlspecialchars($application['email']); ?>">
                </div>
            </div>
            
            <div class="row mb-3">
                <div class="col-md-4">
                    <label class="form-label">Emergency Contact Name</label>
                    <input type="text" name="emergency_contact_name" class="form-control"
                           value="<?php echo htmlspecialchars($application['emergency_contact_name']); ?>">
                </div>
                <div class="col-md-4">
                    <label class="form-label">Emergency Contact Phone</label>
                    <input type="tel" name="emergency_contact_phone" class="form-control"
                           value="<?php echo htmlspecialchars($application['emergency_contact_phone']); ?>">
                </div>
                <div class="col-md-4">
                    <label class="form-label">Relationship</label>
                    <input type="text" name="emergency_contact_relationship" class="form-control"
                           value="<?php echo htmlspecialchars($application['emergency_contact_relationship']); ?>">
                </div>
            </div>
        </div>
    </div>
    
    <!-- Employment Information -->
    <div class="card mb-3">
        <div class="card-header">Employment Information</div>
        <div class="card-body">
            <div class="row mb-3">
                <div class="col-md-4">
                    <label class="form-label">Available Start Date</label>
                    <input type="date" name="available_start_date" class="form-control"
                           value="<?php echo $application['available_start_date']; ?>">
                </div>
                <div class="col-md-4">
                    <label class="form-label">Transportation</label>
                    <select name="transportation_method" class="form-control">
                        <option value="">Select method</option>
                        <option value="own_vehicle" <?php echo $application['transportation_method'] === 'own_vehicle' ? 'selected' : ''; ?>>Own Vehicle</option>
                        <option value="public_transport" <?php echo $application['transportation_method'] === 'public_transport' ? 'selected' : ''; ?>>Public Transport</option>
                        <option value="company_transport" <?php echo $application['transportation_method'] === 'company_transport' ? 'selected' : ''; ?>>Company Transport</option>
                        <option value="other" <?php echo $application['transportation_method'] === 'other' ? 'selected' : ''; ?>>Other</option>
                    </select>
                </div>
            </div>
            
            <div class="form-check mb-3">
                <input type="checkbox" name="currently_employed" class="form-check-input" id="currentlyEmployed" 
                       onchange="toggleCurrentEmployment()" <?php echo $application['currently_employed'] ? 'checked' : ''; ?>>
                <label class="form-check-label" for="currentlyEmployed">Currently Employed</label>
            </div>
            
            <div id="currentEmploymentFields" style="display: <?php echo $application['currently_employed'] ? 'block' : 'none'; ?>;">
                <h6>Current Employment Details</h6>
                <div class="row mb-3">
                    <div class="col-md-6">
                        <label class="form-label">Current Employer</label>
                        <input type="text" name="current_employer" class="form-control"
                               value="<?php echo htmlspecialchars($application['current_employer']); ?>">
                    </div>
                    <div class="col-md-6">
                        <label class="form-label">Current Position</label>
                        <input type="text" name="current_position" class="form-control"
                               value="<?php echo htmlspecialchars($application['current_position']); ?>">
                    </div>
                </div>
                <div class="row mb-3">
                    <div class="col-md-4">
                        <label class="form-label">Current Salary</label>
                        <input type="number" name="current_salary" class="form-control" step="0.01"
                               value="<?php echo $application['current_salary']; ?>">
                    </div>
                    <div class="col-md-4">
                        <label class="form-label">Employment Start Date</label>
                        <input type="date" name="employment_start_date" class="form-control"
                               value="<?php echo $application['employment_start_date']; ?>">
                    </div>
                    <div class="col-md-4">
                        <label class="form-label">Reason for Leaving</label>
                        <input type="text" name="reason_for_leaving" class="form-control"
                               value="<?php echo htmlspecialchars($application['reason_for_leaving']); ?>">
                    </div>
                </div>
            </div>
            
            <!-- Previous Employment History -->
            <h6>Previous Employment History</h6>
            
            <!-- Previous Job 1 -->
            <div class="border rounded p-3 mb-3">
                <h6 class="text-primary">Previous Employment 1</h6>
                <div class="row mb-2">
                    <div class="col-md-4">
                        <label class="form-label">Employer Name</label>
                        <input type="text" name="prev1_employer" class="form-control"
                               value="<?php echo htmlspecialchars($application['prev1_employer']); ?>">
                    </div>
                    <div class="col-md-4">
                        <label class="form-label">Position</label>
                        <input type="text" name="prev1_position" class="form-control"
                               value="<?php echo htmlspecialchars($application['prev1_position']); ?>">
                    </div>
                    <div class="col-md-4">
                        <label class="form-label">Duration</label>
                        <input type="text" name="prev1_duration" class="form-control" placeholder="e.g., 2 years"
                               value="<?php echo htmlspecialchars($application['prev1_duration']); ?>">
                    </div>
                </div>
                <div class="row mb-2">
                    <div class="col-md-4">
                        <label class="form-label">Salary</label>
                        <input type="number" name="prev1_salary" class="form-control" step="0.01"
                               value="<?php echo $application['prev1_salary']; ?>">
                    </div>
                    <div class="col-md-8">
                        <label class="form-label">Reason for Leaving</label>
                        <input type="text" name="prev1_reason_leaving" class="form-control"
                               value="<?php echo htmlspecialchars($application['prev1_reason_leaving']); ?>">
                    </div>
                </div>
            </div>
            
            <!-- Previous Job 2 -->
            <div class="border rounded p-3 mb-3">
                <h6 class="text-primary">Previous Employment 2</h6>
                <div class="row mb-2">
                    <div class="col-md-4">
                        <label class="form-label">Employer Name</label>
                        <input type="text" name="prev2_employer" class="form-control"
                               value="<?php echo htmlspecialchars($application['prev2_employer']); ?>">
                    </div>
                    <div class="col-md-4">
                        <label class="form-label">Position</label>
                        <input type="text" name="prev2_position" class="form-control"
                               value="<?php echo htmlspecialchars($application['prev2_position']); ?>">
                    </div>
                    <div class="col-md-4">
                        <label class="form-label">Duration</label>
                        <input type="text" name="prev2_duration" class="form-control" placeholder="e.g., 1.5 years"
                               value="<?php echo htmlspecialchars($application['prev2_duration']); ?>">
                    </div>
                </div>
                <div class="row mb-2">
                    <div class="col-md-4">
                        <label class="form-label">Salary</label>
                        <input type="number" name="prev2_salary" class="form-control" step="0.01"
                               value="<?php echo $application['prev2_salary']; ?>">
                    </div>
                    <div class="col-md-8">
                        <label class="form-label">Reason for Leaving</label>
                        <input type="text" name="prev2_reason_leaving" class="form-control"
                               value="<?php echo htmlspecialchars($application['prev2_reason_leaving']); ?>">
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Education -->
    <div class="card mb-3">
        <div class="card-header">Education Background</div>
        <div class="card-body">
            <!-- Elementary Education -->
            <div class="row mb-3">
                <div class="col-md-6">
                    <label class="form-label">Elementary School</label>
                    <input type="text" name="elementary_school" class="form-control"
                           value="<?php echo htmlspecialchars($application['elementary_school']); ?>">
                </div>
                <div class="col-md-6">
                    <label class="form-label">Elementary Year Completed</label>
                    <input type="number" name="elementary_year_completed" class="form-control" min="1950" max="<?php echo date('Y'); ?>"
                           value="<?php echo $application['elementary_year_completed']; ?>">
                </div>
            </div>
            
            <!-- High School -->
            <div class="row mb-3">
                <div class="col-md-6">
                    <label class="form-label">High School</label>
                    <input type="text" name="high_school" class="form-control"
                           value="<?php echo htmlspecialchars($application['high_school']); ?>">
                </div>
                <div class="col-md-6">
                    <label class="form-label">High School Year Completed</label>
                    <input type="number" name="high_school_year" class="form-control" min="1950" max="<?php echo date('Y'); ?>"
                           value="<?php echo $application['high_school_year_completed']; ?>">
                </div>
            </div>
            
            <div class="row mb-3">
                <div class="col-md-6">
                    <label class="form-label">O/L Results</label>
                    <input type="text" name="ol_results" class="form-control" placeholder="e.g., 6 passes including Maths"
                           value="<?php echo htmlspecialchars($application['ol_results']); ?>">
                </div>
                <div class="col-md-6">
                    <label class="form-label">A/L Results</label>
                    <input type="text" name="al_results" class="form-control" placeholder="e.g., 2 passes in Science stream"
                           value="<?php echo htmlspecialchars($application['al_results']); ?>">
                </div>
            </div>
            
            <!-- Higher Education -->
            <div class="row mb-3">
                <div class="col-md-4">
                    <label class="form-label">College/University</label>
                    <input type="text" name="college_university" class="form-control"
                           value="<?php echo htmlspecialchars($application['college_university']); ?>">
                </div>
                <div class="col-md-4">
                    <label class="form-label">Degree/Diploma</label>
                    <input type="text" name="degree_diploma" class="form-control"
                           value="<?php echo htmlspecialchars($application['degree_diploma']); ?>">
                </div>
                <div class="col-md-4">
                    <label class="form-label">Graduation Year</label>
                    <input type="number" name="graduation_year" class="form-control" min="1950" max="<?php echo date('Y'); ?>"
                           value="<?php echo $application['graduation_year']; ?>">
                </div>
            </div>
            
            <div class="row mb-3">
                <div class="col-md-4">
                    <label class="form-label">GPA/Class</label>
                    <input type="text" name="gpa_class" class="form-control" placeholder="e.g., First Class, 3.5 GPA"
                           value="<?php echo htmlspecialchars($application['gpa_class']); ?>">
                </div>
                <div class="col-md-4">
                    <label class="form-label">Certifications</label>
                    <textarea name="certifications" class="form-control" rows="2" placeholder="Professional certifications"><?php echo htmlspecialchars($application['certifications']); ?></textarea>
                </div>
                <div class="col-md-4">
                    <label class="form-label">Training Courses</label>
                    <textarea name="training_courses" class="form-control" rows="2" placeholder="Additional training courses"><?php echo htmlspecialchars($application['training_courses']); ?></textarea>
                </div>
            </div>
            
            <div class="row mb-3">
                <div class="col-md-12">
                    <label class="form-label">Other Education</label>
                    <textarea name="other_education" class="form-control" rows="2" placeholder="Any other educational qualifications"><?php echo htmlspecialchars($application['other_education']); ?></textarea>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Banking Information -->
    <div class="card mb-3">
        <div class="card-header">Banking Information</div>
        <div class="card-body">
            <div class="row mb-3">
                <div class="col-md-6">
                    <label class="form-label">Bank Name</label>
                    <input type="text" name="bank_name" class="form-control"
                           value="<?php echo htmlspecialchars($application['bank_name']); ?>">
                </div>
                <div class="col-md-6">
                    <label class="form-label">Branch</label>
                    <input type="text" name="bank_branch" class="form-control"
                           value="<?php echo htmlspecialchars($application['bank_branch']); ?>">
                </div>
            </div>
            
            <div class="row mb-3">
                <div class="col-md-6">
                    <label class="form-label">Account Number</label>
                    <input type="text" name="account_number" class="form-control"
                           value="<?php echo htmlspecialchars($application['account_number']); ?>">
                </div>
                <div class="col-md-6">
                    <label class="form-label">Account Holder Name</label>
                    <input type="text" name="account_holder_name" class="form-control"
                           value="<?php echo htmlspecialchars($application['account_holder_name']); ?>">
                </div>
            </div>
        </div>
    </div>
    
    <!-- Skills & References -->
    <div class="card mb-3">
        <div class="card-header">Skills & References</div>
        <div class="card-body">
            <div class="row mb-3">
                <div class="col-md-4">
                    <label class="form-label">Special Skills</label>
                    <textarea name="special_skills" class="form-control" rows="3" 
                              placeholder="e.g., Welding, Heavy machinery operation"><?php echo htmlspecialchars($application['special_skills']); ?></textarea>
                </div>
                <div class="col-md-4">
                    <label class="form-label">Computer Skills</label>
                    <textarea name="computer_skills" class="form-control" rows="3" 
                              placeholder="e.g., MS Office, AutoCAD"><?php echo htmlspecialchars($application['computer_skills']); ?></textarea>
                </div>
                <div class="col-md-4">
                    <label class="form-label">Hobbies</label>
                    <textarea name="hobbies" class="form-control" rows="3" 
                              placeholder="Personal interests and hobbies"><?php echo htmlspecialchars($application['hobbies']); ?></textarea>
                </div>
            </div>
            
            <div class="row mb-3">
                <div class="col-md-12">
                    <label class="form-label">Employee References</label>
                    <textarea name="employee_references" class="form-control" rows="3" 
                              placeholder="Names and contact details of previous supervisors or colleagues who can provide references"><?php echo htmlspecialchars($application['employee_references']); ?></textarea>
                </div>
            </div>
        </div>
    </div>

    <!-- Save Changes -->
    <div class="card mb-3">
        <div class="card-body">
            <div class="d-flex justify-content-between align-items-center">
                <div>
                    <h6 class="mb-1">Save Changes</h6>
                    <small class="text-muted">Review your changes before saving. All fields marked with * are required.</small>
                </div>
                <div class="btn-group">
                    <a href="view_application.php?id=<?php echo $application_id; ?>" class="btn btn-outline-secondary">
                        <i class="fas fa-times"></i> Cancel
                    </a>
                    <button type="submit" class="btn btn-primary" id="saveBtn">
                        <i class="fas fa-save"></i> Save Changes
                    </button>
                </div>
            </div>
        </div>
    </div>
</form>

<!-- Change Summary Modal -->
<div class="modal fade" id="changesModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Confirm Changes</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <p>You have made changes to the following sections:</p>
                <ul id="changesList"></ul>
                <p><strong>Are you sure you want to save these changes?</strong></p>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <button type="button" class="btn btn-primary" onclick="submitForm()">Save Changes</button>
            </div>
        </div>
    </div>
</div>

<script>
function updatePositionTitle() {
    const categorySelect = document.querySelector('select[name="category_id"]');
    const positionInput = document.getElementById('position_applied');
    
    if (categorySelect.value) {
        const selectedOption = categorySelect.options[categorySelect.selectedIndex];
        positionInput.value = selectedOption.dataset.position || selectedOption.textContent.split('(')[0].trim();
    } else {
        positionInput.value = '';
    }
}

function toggleCurrentEmployment() {
    const checkbox = document.getElementById('currentlyEmployed');
    const fields = document.getElementById('currentEmploymentFields');
    fields.style.display = checkbox.checked ? 'block' : 'none';
}

// Store original form values for change detection
let originalFormData = new FormData(document.getElementById('editApplicationForm'));
let originalValues = {};
for (let [key, value] of originalFormData.entries()) {
    originalValues[key] = value;
}

// Track changes
function detectChanges() {
    const currentForm = new FormData(document.getElementById('editApplicationForm'));
    const changes = [];
    
    for (let [key, value] of currentForm.entries()) {
        if (originalValues[key] !== value) {
            // Get field label
            const field = document.querySelector(`[name="${key}"]`);
            const label = field ? (field.closest('.form-group')?.querySelector('label')?.textContent || key) : key;
            changes.push(label.replace('*', '').trim());
        }
    }
    
    return changes;
}

// Form submission with change confirmation
document.getElementById('editApplicationForm').addEventListener('submit', function(e) {
    e.preventDefault();
    
    const changes = detectChanges();
    
    if (changes.length > 0) {
        // Show changes modal
        const changesList = document.getElementById('changesList');
        changesList.innerHTML = '';
        changes.forEach(change => {
            const li = document.createElement('li');
            li.textContent = change;
            changesList.appendChild(li);
        });
        
        const modal = new bootstrap.Modal(document.getElementById('changesModal'));
        modal.show();
    } else {
        // No changes detected
        alert('No changes detected.');
    }
});

function submitForm() {
    const modal = bootstrap.Modal.getInstance(document.getElementById('changesModal'));
    modal.hide();
    
    // Show loading state
    const saveBtn = document.getElementById('saveBtn');
    saveBtn.disabled = true;
    saveBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Saving...';
    
    // Submit form
    document.getElementById('editApplicationForm').submit();
}

// Auto-save functionality (optional)
let autoSaveTimer;
function enableAutoSave() {
    clearTimeout(autoSaveTimer);
    autoSaveTimer = setTimeout(function() {
        const formData = new FormData(document.getElementById('editApplicationForm'));
        formData.append('auto_save', '1');
        
        fetch('auto_save_application.php', {
            method: 'POST',
            body: formData
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                // Show auto-save indicator
                const indicator = document.createElement('div');
                indicator.className = 'alert alert-info alert-sm position-fixed';
                indicator.style.cssText = 'top: 20px; right: 20px; z-index: 9999; padding: 8px 12px;';
                indicator.innerHTML = '<i class="fas fa-check me-2"></i>Draft saved automatically';
                document.body.appendChild(indicator);
                
                setTimeout(() => indicator.remove(), 3000);
            }
        })
        .catch(error => console.log('Auto-save failed:', error));
    }, 30000); // Auto-save every 30 seconds
}

// Enable auto-save on form changes
document.getElementById('editApplicationForm').addEventListener('input', enableAutoSave);
document.getElementById('editApplicationForm').addEventListener('change', enableAutoSave);

// Warn before leaving with unsaved changes
window.addEventListener('beforeunload', function(e) {
    const changes = detectChanges();
    if (changes.length > 0) {
        e.preventDefault();
        e.returnValue = 'You have unsaved changes. Are you sure you want to leave?';
        return 'You have unsaved changes. Are you sure you want to leave?';
    }
});

// Form validation
document.addEventListener('DOMContentLoaded', function() {
    // NIC validation
    const nicInput = document.querySelector('input[name="nic_number"]');
    if (nicInput) {
        nicInput.addEventListener('blur', function() {
            const nic = this.value.trim();
            if (nic && !/^([0-9]{9}[xXvV]|[0-9]{12})$/.test(nic)) {
                this.setCustomValidity('Invalid NIC format');
            } else {
                this.setCustomValidity('');
            }
        });
    }
    
    // Email validation
    const emailInput = document.querySelector('input[name="email"]');
    if (emailInput) {
        emailInput.addEventListener('blur', function() {
            const email = this.value.trim();
            if (email && !/^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(email)) {
                this.setCustomValidity('Invalid email format');
            } else {
                this.setCustomValidity('');
            }
        });
    }
    
    // Phone validation
    const phoneInput = document.querySelector('input[name="phone_mobile"]');
    if (phoneInput) {
        phoneInput.addEventListener('blur', function() {
            const phone = this.value.trim();
            if (phone && !/^[+]?[0-9\s\-\(\)]{7,15}$/.test(phone)) {
                this.setCustomValidity('Invalid phone number format');
            } else {
                this.setCustomValidity('');
            }
        });
    }
});
</script>

<style>
.form-control:focus {
    border-color: #80bdff;
    box-shadow: 0 0 0 0.2rem rgba(0, 123, 255, 0.25);
}

.card-header {
    font-weight: 600;
}

.alert-sm {
    padding: 0.375rem 0.75rem;
    font-size: 0.875rem;
}

/* Highlight changed fields */
.form-control.changed {
    border-left: 3px solid #ffc107;
}

.form-control.changed:focus {
    border-left: 3px solid #ffc107;
}

/* Loading state */
.btn:disabled {
    opacity: 0.6;
    cursor: not-allowed;
}

/* Responsive improvements */
@media (max-width: 768px) {
    .btn-group {
        flex-direction: column;
        gap: 0.5rem;
    }
    
    .btn-group .btn {
        width: 100%;
    }
}
</style>

<?php require_once '../includes/footer.php'; ?>