<?php
/**
 * Get Application Details (AJAX)
 * Employee Recruitment Management System
 * RR Construction (Pvt) Ltd
 */

// Define access constant
define('HRM_ACCESS', true);

// Start session
session_start();

// Include required files
require_once '../config/constants.php';
require_once '../config/database.php';
require_once '../includes/functions.php';

// Set content type
header('Content-Type: text/html; charset=UTF-8');

// Check authentication
if (!isLoggedIn()) {
    echo '<div class="alert alert-danger">Access denied. Please log in.</div>';
    exit;
}

$currentUser = getCurrentUser();
if (!$currentUser) {
    echo '<div class="alert alert-danger">User session invalid.</div>';
    exit;
}

// Get application ID
$application_id = (int)($_GET['id'] ?? 0);

if (!$application_id) {
    echo '<div class="alert alert-danger">Invalid application ID.</div>';
    exit;
}

try {
    // Get application details with request and project info
    $sql = "SELECT ja.*, er.request_number, er.site_project, 
                   u.full_name as submitted_by_name, u.email as submitter_email
            FROM job_applications ja
            LEFT JOIN employee_requests er ON ja.request_id = er.id
            LEFT JOIN users u ON ja.submitted_by = u.id
            WHERE ja.id = ?";
    
    $application = fetchOne($sql, [$application_id]);
    
    if (!$application) {
        echo '<div class="alert alert-danger">Application not found.</div>';
        exit;
    }
    
    // Check permissions
    $has_access = false;
    
    if ($currentUser['role'] === ROLE_SUPER_ADMIN) {
        $has_access = true;
    } elseif ($currentUser['role'] === ROLE_SITE_ADMIN) {
        $has_access = ($application['submitted_by'] == $currentUser['id']);
    } elseif ($currentUser['role'] === ROLE_PROJECT_MANAGER) {
        $has_access = ($application['site_project'] === $currentUser['site_project']);
    } elseif (in_array($currentUser['role'], [ROLE_GENERAL_MANAGER, ROLE_HR_MANAGER, ROLE_ACCOUNTANT])) {
        $has_access = true;
    }
    
    if (!$has_access) {
        echo '<div class="alert alert-danger">Access denied. You do not have permission to view this application.</div>';
        exit;
    }
    
    // Get approval history for this application
    $approval_sql = "SELECT aal.*, u.full_name as approved_by_name
                     FROM application_approval_logs aal
                     LEFT JOIN users u ON aal.approved_by = u.id
                     WHERE aal.application_id = ?
                     ORDER BY aal.approved_at ASC";
    $approvals = fetchAll($approval_sql, [$application_id]);
    
    // Get uploaded documents for this application
    $documents_sql = "SELECT ed.*, u.full_name as uploaded_by_name
                      FROM employee_documents ed
                      LEFT JOIN users u ON ed.uploaded_by = u.id
                      WHERE ed.application_id = ?
                      ORDER BY ed.uploaded_at DESC";
    $documents = fetchAll($documents_sql, [$application_id]);
    
    ?>
    
    <!-- Application Header -->
    <div class="row mb-3">
        <div class="col-md-8">
            <h5 class="text-primary mb-2">
                <i class="fas fa-user me-2"></i>
                <?php echo htmlspecialchars($application['application_number']); ?>
            </h5>
            <h6 class="text-dark"><?php echo htmlspecialchars($application['full_name_english']); ?></h6>
            <?php if ($application['full_name_sinhala']): ?>
            <small class="text-muted"><?php echo htmlspecialchars($application['full_name_sinhala']); ?></small>
            <?php endif; ?>
        </div>
        <div class="col-md-4 text-end">
            <span class="badge bg-<?php echo getStatusColor($application['status']); ?> p-2 mb-2">
                <?php echo getStatusName($application['status'], 'application'); ?>
            </span>
            <br>
            <small class="text-muted">
                Request: <?php echo htmlspecialchars($application['request_number']); ?>
            </small>
        </div>
    </div>
    
    <!-- Personal Information -->
    <div class="row mb-4">
        <div class="col-md-6">
            <h6 class="text-secondary mb-2">Personal Information</h6>
            <table class="table table-sm table-borderless">
                <tr>
                    <td width="120"><strong>NIC:</strong></td>
                    <td><?php echo htmlspecialchars($application['nic_number']); ?></td>
                </tr>
                <tr>
                    <td><strong>Age:</strong></td>
                    <td><?php echo $application['age']; ?> years</td>
                </tr>
                <tr>
                    <td><strong>Gender:</strong></td>
                    <td><?php echo ucfirst($application['gender']); ?></td>
                </tr>
                <tr>
                    <td><strong>Marital Status:</strong></td>
                    <td><?php echo ucfirst($application['marital_status']); ?></td>
                </tr>
                <tr>
                    <td><strong>Nationality:</strong></td>
                    <td><?php echo htmlspecialchars($application['nationality']); ?></td>
                </tr>
                <?php if ($application['religion']): ?>
                <tr>
                    <td><strong>Religion:</strong></td>
                    <td><?php echo htmlspecialchars($application['religion']); ?></td>
                </tr>
                <?php endif; ?>
            </table>
        </div>
        <div class="col-md-6">
            <h6 class="text-secondary mb-2">Contact Information</h6>
            <table class="table table-sm table-borderless">
                <tr>
                    <td width="120"><strong>Mobile:</strong></td>
                    <td><?php echo htmlspecialchars($application['phone_mobile']); ?></td>
                </tr>
                <?php if ($application['phone_home']): ?>
                <tr>
                    <td><strong>Home Phone:</strong></td>
                    <td><?php echo htmlspecialchars($application['phone_home']); ?></td>
                </tr>
                <?php endif; ?>
                <?php if ($application['email']): ?>
                <tr>
                    <td><strong>Email:</strong></td>
                    <td><?php echo htmlspecialchars($application['email']); ?></td>
                </tr>
                <?php endif; ?>
                <tr>
                    <td><strong>Address:</strong></td>
                    <td><?php echo nl2br(htmlspecialchars($application['permanent_address'])); ?></td>
                </tr>
            </table>
        </div>
    </div>
    
    <!-- Employment Information -->
    <div class="row mb-4">
        <div class="col-md-6">
            <h6 class="text-secondary mb-2">Position Applied</h6>
            <table class="table table-sm table-borderless">
                <tr>
                    <td width="140"><strong>Position:</strong></td>
                    <td><?php echo htmlspecialchars($application['position_applied']); ?></td>
                </tr>
                <tr>
                    <td><strong>Project:</strong></td>
                    <td><?php echo htmlspecialchars($application['site_project']); ?></td>
                </tr>
                <?php if ($application['expected_salary']): ?>
                <tr>
                    <td><strong>Expected Salary:</strong></td>
                    <td><?php echo formatCurrencyAmount($application['expected_salary']); ?></td>
                </tr>
                <?php endif; ?>
                <?php if ($application['available_start_date']): ?>
                <tr>
                    <td><strong>Available From:</strong></td>
                    <td><?php echo formatDisplayDate($application['available_start_date']); ?></td>
                </tr>
                <?php endif; ?>
                <?php if ($application['transportation_method']): ?>
                <tr>
                    <td><strong>Transportation:</strong></td>
                    <td><?php echo ucwords(str_replace('_', ' ', $application['transportation_method'])); ?></td>
                </tr>
                <?php endif; ?>
            </table>
        </div>
        <div class="col-md-6">
            <h6 class="text-secondary mb-2">Current Employment</h6>
            <?php if ($application['currently_employed']): ?>
            <table class="table table-sm table-borderless">
                <tr>
                    <td width="140"><strong>Employer:</strong></td>
                    <td><?php echo htmlspecialchars($application['current_employer']); ?></td>
                </tr>
                <tr>
                    <td><strong>Position:</strong></td>
                    <td><?php echo htmlspecialchars($application['current_position']); ?></td>
                </tr>
                <?php if ($application['current_salary']): ?>
                <tr>
                    <td><strong>Current Salary:</strong></td>
                    <td><?php echo formatCurrencyAmount($application['current_salary']); ?></td>
                </tr>
                <?php endif; ?>
                <?php if ($application['employment_start_date']): ?>
                <tr>
                    <td><strong>Start Date:</strong></td>
                    <td><?php echo formatDisplayDate($application['employment_start_date']); ?></td>
                </tr>
                <?php endif; ?>
                <?php if ($application['reason_for_leaving']): ?>
                <tr>
                    <td><strong>Reason for Leaving:</strong></td>
                    <td><?php echo htmlspecialchars($application['reason_for_leaving']); ?></td>
                </tr>
                <?php endif; ?>
            </table>
            <?php else: ?>
            <p class="text-muted">Not currently employed</p>
            <?php endif; ?>
        </div>
    </div>
    
    <!-- Education -->
    <div class="mb-4">
        <h6 class="text-secondary mb-2">Education</h6>
        <div class="row">
            <div class="col-md-6">
                <?php if ($application['high_school']): ?>
                <div class="mb-2">
                    <strong>High School:</strong> <?php echo htmlspecialchars($application['high_school']); ?>
                    <?php if ($application['high_school_year_completed']): ?>
                        (<?php echo $application['high_school_year_completed']; ?>)
                    <?php endif; ?>
                </div>
                <?php endif; ?>
                
                <?php if ($application['ol_results']): ?>
                <div class="mb-2">
                    <strong>O/L Results:</strong> <?php echo htmlspecialchars($application['ol_results']); ?>
                </div>
                <?php endif; ?>
                
                <?php if ($application['al_results']): ?>
                <div class="mb-2">
                    <strong>A/L Results:</strong> <?php echo htmlspecialchars($application['al_results']); ?>
                </div>
                <?php endif; ?>
            </div>
            <div class="col-md-6">
                <?php if ($application['college_university']): ?>
                <div class="mb-2">
                    <strong>Higher Education:</strong> <?php echo htmlspecialchars($application['college_university']); ?>
                </div>
                <?php endif; ?>
                
                <?php if ($application['degree_diploma']): ?>
                <div class="mb-2">
                    <strong>Degree/Diploma:</strong> <?php echo htmlspecialchars($application['degree_diploma']); ?>
                    <?php if ($application['graduation_year']): ?>
                        (<?php echo $application['graduation_year']; ?>)
                    <?php endif; ?>
                </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
    
    <!-- Skills and Other Information -->
    <div class="mb-4">
        <h6 class="text-secondary mb-2">Skills & Qualifications</h6>
        <div class="row">
            <div class="col-md-4">
                <?php if ($application['special_skills']): ?>
                <div class="mb-2">
                    <strong>Special Skills:</strong><br>
                    <small><?php echo nl2br(htmlspecialchars($application['special_skills'])); ?></small>
                </div>
                <?php endif; ?>
            </div>
            <div class="col-md-4">
                <?php if ($application['computer_skills']): ?>
                <div class="mb-2">
                    <strong>Computer Skills:</strong><br>
                    <small><?php echo nl2br(htmlspecialchars($application['computer_skills'])); ?></small>
                </div>
                <?php endif; ?>
            </div>
            <div class="col-md-4">
                <?php if ($application['language_skills']): ?>
                <div class="mb-2">
                    <strong>Language Skills:</strong><br>
                    <small><?php echo nl2br(htmlspecialchars($application['language_skills'])); ?></small>
                </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
    
    <!-- Banking Information -->
    <?php if ($application['bank_name'] || $application['account_number']): ?>
    <div class="mb-4">
        <h6 class="text-secondary mb-2">Banking Information</h6>
        <div class="row">
            <div class="col-md-6">
                <?php if ($application['bank_name']): ?>
                <div><strong>Bank:</strong> <?php echo htmlspecialchars($application['bank_name']); ?></div>
                <?php endif; ?>
                <?php if ($application['bank_branch']): ?>
                <div><strong>Branch:</strong> <?php echo htmlspecialchars($application['bank_branch']); ?></div>
                <?php endif; ?>
            </div>
            <div class="col-md-6">
                <?php if ($application['account_number']): ?>
                <div><strong>Account Number:</strong> <?php echo htmlspecialchars($application['account_number']); ?></div>
                <?php endif; ?>
                <?php if ($application['account_holder_name']): ?>
                <div><strong>Account Holder:</strong> <?php echo htmlspecialchars($application['account_holder_name']); ?></div>
                <?php endif; ?>
            </div>
        </div>
    </div>
    <?php endif; ?>
    
    <!-- Emergency Contact -->
    <?php if ($application['emergency_contact_name']): ?>
    <div class="mb-4">
        <h6 class="text-secondary mb-2">Emergency Contact</h6>
        <div class="row">
            <div class="col-md-4">
                <strong>Name:</strong> <?php echo htmlspecialchars($application['emergency_contact_name']); ?>
            </div>
            <div class="col-md-4">
                <strong>Phone:</strong> <?php echo htmlspecialchars($application['emergency_contact_phone']); ?>
            </div>
            <div class="col-md-4">
                <strong>Relationship:</strong> <?php echo htmlspecialchars($application['emergency_contact_relationship']); ?>
            </div>
        </div>
    </div>
    <?php endif; ?>
    
    <!-- Documents -->
    <?php if (!empty($documents)): ?>
    <div class="mb-4">
        <h6 class="text-secondary mb-2">Uploaded Documents (<?php echo count($documents); ?>)</h6>
        <div class="table-responsive">
            <table class="table table-sm">
                <thead class="bg-light">
                    <tr>
                        <th>Document</th>
                        <th>Type</th>
                        <th>Size</th>
                        <th>Uploaded</th>
                        <th>Action</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($documents as $doc): ?>
                    <tr>
                        <td>
                            <i class="<?php echo getFileIcon($doc['original_filename']); ?> me-2"></i>
                            <?php echo htmlspecialchars($doc['document_name']); ?>
                        </td>
                        <td>
                            <span class="badge bg-secondary">
                                <?php echo DOCUMENT_TYPES[$doc['document_type']] ?? $doc['document_type']; ?>
                            </span>
                        </td>
                        <td><?php echo formatFileSize($doc['file_size']); ?></td>
                        <td><?php echo formatDisplayDate($doc['uploaded_at']); ?></td>
                        <td>
                            <a href="../documents/download.php?id=<?php echo $doc['id']; ?>" 
                               class="btn btn-outline-primary btn-sm" target="_blank">
                                <i class="fas fa-download"></i>
                            </a>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    </div>
    <?php endif; ?>
    
    <!-- Approval History -->
    <?php if (!empty($approvals)): ?>
    <div class="mb-4">
        <h6 class="text-secondary mb-2">Approval History</h6>
        <div class="table-responsive">
            <table class="table table-sm">
                <thead class="bg-light">
                    <tr>
                        <th>Level</th>
                        <th>Approved By</th>
                        <th>Action</th>
                        <th>Comments</th>
                        <th>Salary</th>
                        <th>Date</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($approvals as $approval): ?>
                    <tr>
                        <td><?php echo getRoleName($approval['approval_level']); ?></td>
                        <td><?php echo htmlspecialchars($approval['approved_by_name']); ?></td>
                        <td>
                            <span class="badge <?php echo $approval['action'] === 'approved' ? 'bg-success' : 'bg-danger'; ?>">
                                <?php echo ucfirst($approval['action']); ?>
                            </span>
                        </td>
                        <td><?php echo htmlspecialchars($approval['comments'] ?: 'No comments'); ?></td>
                        <td>
                            <?php echo $approval['salary_approved'] ? formatCurrencyAmount($approval['salary_approved']) : '-'; ?>
                        </td>
                        <td><?php echo formatDisplayDateTime($approval['approved_at']); ?></td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    </div>
    <?php endif; ?>
    
    <!-- Application Summary -->
    <div class="mb-4">
        <h6 class="text-secondary mb-2">Application Summary</h6>
        <div class="row">
            <div class="col-md-6">
                <table class="table table-sm table-borderless">
                    <tr>
                        <td width="140"><strong>Submitted By:</strong></td>
                        <td><?php echo htmlspecialchars($application['submitted_by_name']); ?></td>
                    </tr>
                    <tr>
                        <td><strong>Submitted Date:</strong></td>
                        <td><?php echo formatDisplayDateTime($application['submitted_at']); ?></td>
                    </tr>
                    <tr>
                        <td><strong>Last Updated:</strong></td>
                        <td><?php echo formatDisplayDateTime($application['updated_at']); ?></td>
                    </tr>
                </table>
            </div>
            <div class="col-md-6">
                <?php if ($application['medical_check_status']): ?>
                <div class="mb-2">
                    <strong>Medical Check:</strong> 
                    <span class="badge bg-<?php echo $application['medical_check_status'] === 'passed' ? 'success' : ($application['medical_check_status'] === 'failed' ? 'danger' : 'warning'); ?>">
                        <?php echo ucfirst($application['medical_check_status']); ?>
                    </span>
                </div>
                <?php endif; ?>
                
                <?php if ($application['background_check_status']): ?>
                <div class="mb-2">
                    <strong>Background Check:</strong> 
                    <span class="badge bg-<?php echo $application['background_check_status'] === 'cleared' ? 'success' : ($application['background_check_status'] === 'issues' ? 'danger' : 'warning'); ?>">
                        <?php echo ucfirst($application['background_check_status']); ?>
                    </span>
                </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
    
    <!-- Rejection Reason (if applicable) -->
    <?php if ($application['status'] === 'rejected' && $application['rejection_reason']): ?>
    <div class="alert alert-danger">
        <h6><i class="fas fa-exclamation-triangle me-2"></i>Rejection Reason</h6>
        <p class="mb-0"><?php echo htmlspecialchars($application['rejection_reason']); ?></p>
    </div>
    <?php endif; ?>
    
    <?php
    
} catch (Exception $e) {
    error_log("Error fetching application details: " . $e->getMessage());
    echo '<div class="alert alert-danger">Error loading application details. Please try again.</div>';
}
?>