<?php
/**
 * Document Management
 * Employee Recruitment Management System
 * RR Construction (Pvt) Ltd
 */

$page_title = 'Manage Documents';
$active_menu = 'documents';

require_once '../includes/header.php';
requirePermission(ROLE_SITE_ADMIN);

$currentUser = getCurrentUser();
$errors = [];
$success = false;

// Get application ID from URL
$application_id = (int)($_GET['app_id'] ?? 0);

// Verify application exists and belongs to current user
$application = null;
if ($application_id) {
    try {
        $sql = "SELECT ja.*, er.request_number, er.site_project 
                FROM job_applications ja 
                JOIN employee_requests er ON ja.request_id = er.id 
                WHERE ja.id = ? AND ja.submitted_by = ?";
        $application = fetchOne($sql, [$application_id, $currentUser['id']]);
    } catch (Exception $e) {
        error_log("Error fetching application: " . $e->getMessage());
    }
}

if (!$application) {
    setFlashMessage('Application not found or access denied.', 'danger');
    header('Location: ../employees/view_applications.php');
    exit;
}

// Handle document deletion
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['delete_document'])) {
    if (verifyCSRFToken($_POST['csrf_token'] ?? '')) {
        $document_id = (int)$_POST['document_id'];
        
        try {
            // Get document details
            $document = fetchOne("SELECT * FROM employee_documents WHERE id = ? AND application_id = ?", 
                               [$document_id, $application_id]);
            
            if ($document) {
                beginTransaction();
                
                // Delete file from filesystem
                if (file_exists($document['file_path'])) {
                    unlink($document['file_path']);
                }
                
                // Delete record from database
                deleteRecord('employee_documents', 'id = ?', [$document_id]);
                
                // Log activity
                logActivity($currentUser['id'], 'delete', 'employee_documents', $document_id);
                
                commit();
                setFlashMessage('Document deleted successfully.', 'success');
            } else {
                $errors[] = 'Document not found.';
            }
        } catch (Exception $e) {
            rollback();
            error_log("Error deleting document: " . $e->getMessage());
            $errors[] = 'Failed to delete document.';
        }
    } else {
        $errors[] = 'Invalid security token.';
    }
}

// Get existing documents for this application
$existing_documents = [];
try {
    $sql = "SELECT ed.*, u.full_name as uploaded_by_name 
            FROM employee_documents ed 
            JOIN users u ON ed.uploaded_by = u.id 
            WHERE ed.application_id = ? 
            ORDER BY ed.document_type, ed.uploaded_at DESC";
    $existing_documents = fetchAll($sql, [$application_id]);
} catch (Exception $e) {
    error_log("Error fetching documents: " . $e->getMessage());
}

// Group documents by type
$documents_by_type = [];
foreach ($existing_documents as $doc) {
    $documents_by_type[$doc['document_type']][] = $doc;
}

// Define document types with descriptions
$document_types = [
    'nic_copy' => ['name' => 'NIC Copy', 'required' => true, 'icon' => 'fa-id-card'],
    'cv' => ['name' => 'CV/Resume', 'required' => true, 'icon' => 'fa-file-alt'],
    'certificates' => ['name' => 'Educational Certificates', 'required' => true, 'icon' => 'fa-certificate'],
    'photo' => ['name' => 'Passport Photo', 'required' => true, 'icon' => 'fa-image'],
    'birth_certificate' => ['name' => 'Birth Certificate', 'required' => false, 'icon' => 'fa-baby'],
    'bank_details' => ['name' => 'Bank Details', 'required' => false, 'icon' => 'fa-university'],
    'medical_report' => ['name' => 'Medical Report', 'required' => false, 'icon' => 'fa-heartbeat'],
    'police_report' => ['name' => 'Police Report', 'required' => false, 'icon' => 'fa-shield-alt'],
    'other' => ['name' => 'Other Documents', 'required' => false, 'icon' => 'fa-file']
];
?>

<div class="row mb-4">
    <div class="col-12">
        <div class="d-flex justify-content-between align-items-center">
            <div>
                <h4 class="mb-0">Document Management</h4>
                <small class="text-muted">
                    Application: <?php echo htmlspecialchars($application['application_number']); ?> - 
                    <?php echo htmlspecialchars($application['full_name_english']); ?>
                </small>
            </div>
            <div>
                <a href="upload.php?app_id=<?php echo $application_id; ?>" class="btn btn-primary">
                    <i class="fas fa-upload"></i> Upload Documents
                </a>
            </div>
        </div>
    </div>
</div>

<!-- Application Summary -->
<div class="card mb-3">
    <div class="card-header">Application Summary</div>
    <div class="card-body">
        <div class="row">
            <div class="col-md-8">
                <table class="table table-sm table-borderless">
                    <tr>
                        <td width="150"><strong>Applicant:</strong></td>
                        <td><?php echo htmlspecialchars($application['full_name_english']); ?></td>
                        <td width="120"><strong>Position:</strong></td>
                        <td><?php echo htmlspecialchars($application['position_applied']); ?></td>
                    </tr>
                    <tr>
                        <td><strong>NIC Number:</strong></td>
                        <td><?php echo htmlspecialchars($application['nic_number']); ?></td>
                        <td><strong>Request:</strong></td>
                        <td><?php echo htmlspecialchars($application['request_number']); ?></td>
                    </tr>
                    <tr>
                        <td><strong>Project:</strong></td>
                        <td><?php echo htmlspecialchars($application['site_project']); ?></td>
                        <td><strong>Status:</strong></td>
                        <td><?php echo getStatusBadge($application['status'], 'application'); ?></td>
                    </tr>
                </table>
            </div>
            <div class="col-md-4">
                <div class="text-center">
                    <h5>Document Progress</h5>
                    <?php 
                    $required_types = array_filter($document_types, fn($type) => $type['required']);
                    $uploaded_required = count(array_intersect(array_keys($required_types), array_keys($documents_by_type)));
                    $total_required = count($required_types);
                    $progress = $total_required > 0 ? ($uploaded_required / $total_required) * 100 : 0;
                    ?>
                    <div class="progress mb-2" style="height: 20px;">
                        <div class="progress-bar <?php echo $progress == 100 ? 'bg-success' : 'bg-warning'; ?>" 
                             style="width: <?php echo $progress; ?>%">
                            <?php echo round($progress); ?>%
                        </div>
                    </div>
                    <small class="text-muted">
                        <?php echo $uploaded_required; ?> of <?php echo $total_required; ?> required documents
                    </small>
                </div>
            </div>
        </div>
    </div>
</div>

<?php if (!empty($errors)): ?>
<div class="alert alert-danger">
    <ul class="mb-0">
        <?php foreach ($errors as $error): ?>
            <li><?php echo htmlspecialchars($error); ?></li>
        <?php endforeach; ?>
    </ul>
</div>
<?php endif; ?>

<!-- Document Categories -->
<div class="row">
    <?php foreach ($document_types as $type => $info): ?>
    <div class="col-md-6 mb-3">
        <div class="card h-100">
            <div class="card-header d-flex justify-content-between align-items-center">
                <div>
                    <i class="fas <?php echo $info['icon']; ?> me-2"></i>
                    <?php echo $info['name']; ?>
                    <?php if ($info['required']): ?>
                        <span class="badge bg-danger ms-2">Required</span>
                    <?php endif; ?>
                </div>
                <div>
                    <?php if (isset($documents_by_type[$type])): ?>
                        <span class="badge bg-success"><?php echo count($documents_by_type[$type]); ?> uploaded</span>
                    <?php else: ?>
                        <span class="badge bg-secondary">Not uploaded</span>
                    <?php endif; ?>
                </div>
            </div>
            <div class="card-body">
                <?php if (isset($documents_by_type[$type]) && !empty($documents_by_type[$type])): ?>
                    <div class="list-group list-group-flush">
                        <?php foreach ($documents_by_type[$type] as $doc): ?>
                        <div class="list-group-item px-0">
                            <div class="d-flex justify-content-between align-items-start">
                                <div class="flex-grow-1">
                                    <h6 class="mb-1"><?php echo htmlspecialchars($doc['document_name']); ?></h6>
                                    <p class="mb-1">
                                        <small class="text-muted">
                                            <?php echo htmlspecialchars($doc['original_filename']); ?> 
                                            (<?php echo formatFileSize($doc['file_size']); ?>)
                                        </small>
                                    </p>
                                    <small class="text-muted">
                                        Uploaded <?php echo formatDisplayDate($doc['uploaded_at']); ?> 
                                        by <?php echo htmlspecialchars($doc['uploaded_by_name']); ?>
                                    </small>
                                </div>
                                <div class="btn-group btn-group-sm ms-2">
                                    <a href="../documents/download.php?id=<?php echo $doc['id']; ?>" 
                                       class="btn btn-outline-primary" target="_blank" title="Download">
                                        <i class="fas fa-download"></i>
                                    </a>
                                    <a href="../documents/download.php?id=<?php echo $doc['id']; ?>&view=1" 
                                       class="btn btn-outline-info" target="_blank" title="View">
                                        <i class="fas fa-eye"></i>
                                    </a>
                                    <button type="button" class="btn btn-outline-danger" 
                                            onclick="deleteDocument(<?php echo $doc['id']; ?>)" title="Delete">
                                        <i class="fas fa-trash"></i>
                                    </button>
                                </div>
                            </div>
                        </div>
                        <?php endforeach; ?>
                    </div>
                <?php else: ?>
                    <div class="text-center py-3">
                        <i class="fas fa-folder-open fa-2x text-muted mb-2"></i>
                        <p class="text-muted mb-2">No documents uploaded</p>
                        <a href="upload.php?app_id=<?php echo $application_id; ?>" 
                           class="btn btn-sm btn-outline-primary">
                            <i class="fas fa-upload"></i> Upload <?php echo $info['name']; ?>
                        </a>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
    <?php endforeach; ?>
</div>

<!-- Document Statistics -->
<div class="card mt-3">
    <div class="card-header">Document Statistics</div>
    <div class="card-body">
        <div class="row">
            <div class="col-md-3 text-center">
                <h4 class="text-primary"><?php echo count($existing_documents); ?></h4>
                <small class="text-muted">Total Documents</small>
            </div>
            <div class="col-md-3 text-center">
                <h4 class="text-success"><?php echo $uploaded_required; ?></h4>
                <small class="text-muted">Required Documents</small>
            </div>
            <div class="col-md-3 text-center">
                <h4 class="text-info">
                    <?php echo formatFileSize(array_sum(array_column($existing_documents, 'file_size'))); ?>
                </h4>
                <small class="text-muted">Total Size</small>
            </div>
            <div class="col-md-3 text-center">
                <h4 class="<?php echo $progress == 100 ? 'text-success' : 'text-warning'; ?>">
                    <?php echo round($progress); ?>%
                </h4>
                <small class="text-muted">Completion</small>
            </div>
        </div>
    </div>
</div>

<!-- Action Buttons -->
<div class="text-center mt-4">
    <a href="../employees/view_applications.php" class="btn btn-outline-secondary">
        <i class="fas fa-arrow-left"></i> Back to Applications
    </a>
    <a href="upload.php?app_id=<?php echo $application_id; ?>" class="btn btn-primary">
        <i class="fas fa-upload"></i> Upload More Documents
    </a>
    <?php if (!empty($existing_documents)): ?>
    <button type="button" class="btn btn-info" onclick="downloadAllDocuments()">
        <i class="fas fa-download"></i> Download All
    </button>
    <?php endif; ?>
</div>

<script>
function deleteDocument(documentId) {
    if (confirm('Are you sure you want to delete this document? This action cannot be undone.')) {
        const form = document.createElement('form');
        form.method = 'POST';
        form.action = '';
        
        const csrfInput = document.createElement('input');
        csrfInput.type = 'hidden';
        csrfInput.name = 'csrf_token';
        csrfInput.value = '<?php echo generateCSRFToken(); ?>';
        
        const deleteInput = document.createElement('input');
        deleteInput.type = 'hidden';
        deleteInput.name = 'delete_document';
        deleteInput.value = '1';
        
        const idInput = document.createElement('input');
        idInput.type = 'hidden';
        idInput.name = 'document_id';
        idInput.value = documentId;
        
        form.appendChild(csrfInput);
        form.appendChild(deleteInput);
        form.appendChild(idInput);
        
        document.body.appendChild(form);
        form.submit();
    }
}

function downloadAllDocuments() {
    window.open('../documents/download_all.php?app_id=<?php echo $application_id; ?>', '_blank');
}
</script>

<?php require_once '../includes/footer.php'; ?>