<?php
/**
 * Print Application Details
 * Employee Recruitment Management System
 * RR Construction (Pvt) Ltd
 */

// Define access constant
define('HRM_ACCESS', true);

// Start session if not already started
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Include required files
require_once '../config/constants.php';
require_once '../config/database.php';
require_once '../includes/functions.php';

// Check authentication
if (!isLoggedIn()) {
    header('Location: ../auth/login.php');
    exit;
}

$currentUser = getCurrentUser();
$application_id = (int)($_GET['id'] ?? 0);

// Get application details
$application = null;
if ($application_id) {
    try {
        $sql = "SELECT ja.*, er.request_number, er.site_project, er.request_date, er.urgent_requirement,
                       ec.position_title, ec.category, ec.quantity, ec.salary_range_min, ec.salary_range_max,
                       u.full_name as submitted_by_name
                FROM job_applications ja
                JOIN employee_requests er ON ja.request_id = er.id
                JOIN employee_categories ec ON ja.category_id = ec.id
                JOIN users u ON ja.submitted_by = u.id
                WHERE ja.id = ?";
        
        // Add role-based access control
        $params = [$application_id];
        
        switch ($currentUser['role']) {
            case ROLE_SUPER_ADMIN:
                // Super admin can view all applications
                break;
                
            case ROLE_SITE_ADMIN:
                // Site admin can only view their own applications
                $sql .= " AND ja.submitted_by = ?";
                $params[] = $currentUser['id'];
                break;
                
            case ROLE_PROJECT_MANAGER:
                // Project manager can view applications from their project
                $sql .= " AND er.site_project = ?";
                $params[] = $currentUser['site_project'];
                break;
                
            case ROLE_GENERAL_MANAGER:
            case ROLE_HR_MANAGER:
            case ROLE_ACCOUNTANT:
                // These roles can view all applications
                break;
                
            default:
                // No access for other roles
                $sql .= " AND 1=0";
                break;
        }
        
        $application = fetchOne($sql, $params);
    } catch (Exception $e) {
        error_log("Error fetching application: " . $e->getMessage());
    }
}

if (!$application) {
    die('Application not found or access denied.');
}

// Get application documents
$documents = [];
try {
    $sql = "SELECT ed.*, u.full_name as uploaded_by_name
            FROM employee_documents ed
            LEFT JOIN users u ON ed.uploaded_by = u.id
            WHERE ed.application_id = ?
            ORDER BY ed.document_type, ed.uploaded_at DESC";
    $documents = fetchAll($sql, [$application_id]);
} catch (Exception $e) {
    error_log("Error fetching documents: " . $e->getMessage());
}

// Calculate age properly
$age = calculateAge($application['date_of_birth']);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Job Application - <?php echo htmlspecialchars($application['application_number']); ?></title>
    <style>
        @page {
            size: A4;
            margin: 10mm;
        }
        
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: 'Arial', 'Helvetica', sans-serif;
            font-size: 12px;
            line-height: 1.4;
            color: #333;
            background: white;
        }
        
        .header {
            text-align: center;
            margin-bottom: 20px;
            border-bottom: 3px solid #2c5aa0;
            padding-bottom: 15px;
        }
        
        .company-logo {
            font-size: 18px;
            font-weight: bold;
            color: #2c5aa0;
            margin-bottom: 5px;
            text-transform: uppercase;
            letter-spacing: 1px;
        }
        
        .company-details {
            font-size: 11px;
            color: #666;
            margin-bottom: 8px;
        }
        
        .form-title {
            font-size: 16px;
            font-weight: bold;
            color: #2c5aa0;
            margin-top: 10px;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        
        .section {
            margin-bottom: 15px;
            page-break-inside: avoid;
        }
        
        .section-title {
            background: linear-gradient(135deg, #2c5aa0, #1e3d6f);
            color: white;
            padding: 6px 10px;
            font-weight: bold;
            font-size: 13px;
            margin-bottom: 8px;
            text-transform: uppercase;
            letter-spacing: 0.5px;
            border-radius: 4px;
        }
        
        .field-group {
            margin-bottom: 8px;
        }
        
        .field-row {
            display: flex;
            margin-bottom: 6px;
            gap: 12px;
            align-items: center;
        }
        
        .field {
            display: flex;
            align-items: center;
            gap: 8px;
        }
        
        .field-label {
            font-weight: 600;
            color: #2c5aa0;
            font-size: 11px;
            white-space: nowrap;
            min-width: fit-content;
        }
        
        .field-value {
            border: 1px solid #ddd;
            border-radius: 3px;
            padding: 4px 8px;
            background: #f9f9f9;
            font-size: 12px;
            min-height: 28px;
            display: flex;
            align-items: center;
        }
        
        .field-value.empty {
            background: white;
            border-style: dashed;
            color: #999;
        }
        
        .w-full { flex: 1; }
        .w-half { flex: 0 0 48%; }
        .w-third { flex: 0 0 32%; }
        .w-quarter { flex: 0 0 23%; }
        .w-auto { flex: 0 0 auto; }
        
        .table-container {
            margin: 8px 0;
        }
        
        .table {
            width: 100%;
            border-collapse: collapse;
            border: 2px solid #2c5aa0;
            background: white;
        }
        
        .table th {
            background: linear-gradient(135deg, #2c5aa0, #1e3d6f);
            color: white;
            padding: 6px 6px;
            text-align: left;
            font-weight: bold;
            font-size: 11px;
            border-right: 1px solid #fff;
        }
        
        .table th:last-child {
            border-right: none;
        }
        
        .table td {
            padding: 8px 6px;
            border: 1px solid #ddd;
            font-size: 11px;
            vertical-align: top;
            background: #fafafa;
        }
        
        .table tr:nth-child(even) td {
            background: #f5f5f5;
        }
        
        .table tr:hover td {
            background: #e8f4f8;
        }
        
        .checkbox-group {
            display: flex;
            align-items: center;
            gap: 15px;
        }
        
        .checkbox-item {
            display: flex;
            align-items: center;
            gap: 5px;
        }
        
        .checkbox {
            width: 16px;
            height: 16px;
            border: 2px solid #2c5aa0;
            border-radius: 3px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 12px;
            font-weight: bold;
            color: #2c5aa0;
            background: white;
        }
        
        .checkbox.checked {
            background: #2c5aa0;
            color: white;
        }
        
        .textarea-field {
            border: 1px solid #ddd;
            border-radius: 4px;
            padding: 8px;
            background: #f9f9f9;
            min-height: 60px;
            font-size: 12px;
            font-family: inherit;
            line-height: 1.4;
            width: 100%;
        }
        
        .info-badge {
            background: #e3f2fd;
            border: 1px solid #2196f3;
            border-radius: 4px;
            padding: 6px 10px;
            margin: 6px 0;
            font-size: 11px;
            color: #1976d2;
        }
        
        .signature-section {
            margin-top: 20px;
            border-top: 2px solid #2c5aa0;
            padding-top: 15px;
        }
        
        .signature-row {
            display: flex;
            justify-content: space-between;
            gap: 20px;
            margin-bottom: 15px;
        }
        
        .signature-block {
            flex: 1;
            text-align: center;
        }
        
        .signature-label {
            font-weight: bold;
            color: #2c5aa0;
            margin-bottom: 8px;
            font-size: 11px;
        }
        
        .signature-line {
            border-bottom: 2px solid #333;
            height: 50px;
            margin-bottom: 5px;
            position: relative;
        }
        
        .signature-text {
            font-size: 10px;
            color: #666;
        }
        
        .office-section {
            background: linear-gradient(135deg, #f5f5f5, #e8e8e8);
            border: 2px solid #2c5aa0;
            border-radius: 6px;
            padding: 20px;
            margin-top: 30px;
        }
        
        .office-title {
            text-align: center;
            font-weight: bold;
            font-size: 14px;
            color: #2c5aa0;
            margin-bottom: 20px;
            text-transform: uppercase;
        }
        
        .office-grid {
            display: grid;
            grid-template-columns: repeat(3, 1fr);
            gap: 20px;
        }
        
        .office-column {
            text-align: center;
            background: white;
            border-radius: 4px;
            padding: 15px;
            border: 1px solid #ddd;
        }
        
        .office-role {
            font-weight: bold;
            color: #2c5aa0;
            margin-bottom: 12px;
            font-size: 12px;
        }
        
        .office-field {
            margin-bottom: 4px;
            display: flex;
            align-items: center;
            gap: 5px;
        }
        
        .office-field-label {
            font-size: 9px;
            color: #666;
            min-width: 35px;
        }
        
        .office-field-box {
            border: 1px solid #ccc;
            height: 18px;
            border-radius: 2px;
            background: #f9f9f9;
            flex: 1;
        }
        
        .office-signature {
            border: 1px solid #ccc;
            height: 30px;
            margin-top: 8px;
            border-radius: 3px;
            background: white;
            display: flex;
            align-items: center;
            justify-content: center;
            color: #999;
            font-size: 9px;
        }
        
        .declaration {
            background: #f8f9fa;
            border-left: 4px solid #2c5aa0;
            padding: 10px;
            margin: 15px 0;
            font-size: 11px;
            line-height: 1.5;
            text-align: justify;
        }
        
        @media print {
            body {
                -webkit-print-color-adjust: exact;
                print-color-adjust: exact;
            }
            
            .no-print {
                display: none !important;
            }
            
            .section {
                page-break-inside: avoid;
            }
            
            .office-section {
                page-break-inside: avoid;
            }
        }
        
        /* Responsive adjustments for smaller content */
        @media screen and (max-width: 768px) {
            .field-row {
                flex-direction: column;
                gap: 8px;
            }
            
            .office-grid {
                grid-template-columns: 1fr;
            }
        }
    </style>
</head>
<body>
    <!-- Print Controls -->
    <div class="no-print" style="text-align: center; margin-bottom: 20px; padding: 15px; background: #f8f9fa; border-radius: 6px;">
        <button onclick="window.print()" style="padding: 12px 24px; font-size: 14px; background: #2c5aa0; color: white; border: none; border-radius: 5px; cursor: pointer; margin-right: 10px; box-shadow: 0 2px 4px rgba(0,0,0,0.1);">
            🖨️ Print Application
        </button>
        <button onclick="window.close()" style="padding: 12px 24px; font-size: 14px; background: #6c757d; color: white; border: none; border-radius: 5px; cursor: pointer; box-shadow: 0 2px 4px rgba(0,0,0,0.1);">
            ✕ Close
        </button>
    </div>

    <!-- Header -->
    <div class="header">
        <div class="company-logo">RR Construction (Pvt) Ltd.</div>
        <div class="company-details">
            <strong>Address:</strong> No 626c, Samurdhi Mw, Cheenagahawela, Heiyanthuduwa, Sapugaskanda<br>
            <strong>Phone:</strong> 011-2433427 | <strong>Fax:</strong> 011-2430161
        </div>
        <div class="form-title">Job Application Form</div>
        <div class="info-badge">
            Application ID: <?php echo htmlspecialchars($application['application_number']); ?> | 
            Date Submitted: <?php echo formatDisplayDate($application['submitted_date']); ?>
        </div>
    </div>

    <!-- Personal Information -->
    <div class="section">
        <div class="section-title">👤 Personal Information</div>
        
        <div class="field-group">
            <div class="field-row">
                <div class="field w-full">
                    <span class="field-label">Full Name:</span>
                    <div class="field-value w-full"><?php echo htmlspecialchars($application['full_name_english']); ?></div>
                </div>
            </div>
            
            <div class="field-row">
                <div class="field w-full">
                    <span class="field-label">Name with Initials:</span>
                    <div class="field-value w-full"><?php echo htmlspecialchars($application['full_name_sinhala'] ?: $application['full_name_english']); ?></div>
                </div>
            </div>
            
            <div class="field-row">
                <div class="field w-full">
                    <span class="field-label">Address:</span>
                    <div class="field-value w-full"><?php echo nl2br(htmlspecialchars($application['permanent_address'])); ?></div>
                </div>
            </div>
            
            <div class="field-row">
                <div class="field w-third">
                    <span class="field-label">NIC Number:</span>
                    <div class="field-value w-full"><?php echo htmlspecialchars($application['nic_number']); ?></div>
                </div>
                <div class="field w-third">
                    <span class="field-label">Date of Birth:</span>
                    <div class="field-value w-full"><?php echo formatDisplayDate($application['date_of_birth']); ?></div>
                </div>
                <div class="field w-third">
                    <span class="field-label">Age:</span>
                    <div class="field-value w-full"><?php echo $age; ?> years</div>
                </div>
            </div>
            
            <div class="field-row">
                <div class="field w-quarter">
                    <span class="field-label">Gender:</span>
                    <div class="field-value w-full"><?php echo ucfirst($application['gender']); ?></div>
                </div>
                <div class="field w-quarter">
                    <span class="field-label">Marital Status:</span>
                    <div class="field-value w-full"><?php echo ucfirst($application['marital_status']); ?></div>
                </div>
                <div class="field w-quarter">
                    <span class="field-label">Nationality:</span>
                    <div class="field-value w-full"><?php echo htmlspecialchars($application['nationality']); ?></div>
                </div>
                <div class="field w-quarter">
                    <span class="field-label">Religion:</span>
                    <div class="field-value w-full"><?php echo htmlspecialchars($application['religion'] ?: 'Buddhist'); ?></div>
                </div>
            </div>
            
            <div class="field-row">
                <div class="field w-half">
                    <span class="field-label">Mobile Phone:</span>
                    <div class="field-value w-full"><?php echo htmlspecialchars($application['phone_mobile']); ?></div>
                </div>
                <div class="field w-half">
                    <span class="field-label">Email Address:</span>
                    <div class="field-value w-full"><?php echo htmlspecialchars($application['email'] ?: 'Not provided'); ?></div>
                </div>
            </div>
            
            <div class="field-row">
                <div class="field w-third">
                    <span class="field-label">District:</span>
                    <div class="field-value w-full"><?php echo htmlspecialchars($application['district'] ?? 'Gampaha'); ?></div>
                </div>
                <div class="field w-third">
                    <span class="field-label">Blood Group:</span>
                    <div class="field-value w-full"><?php echo htmlspecialchars($application['blood_group'] ?: 'Not specified'); ?></div>
                </div>
                <div class="field w-third">
                    <span class="field-label">Medical Condition:</span>
                    <div class="field-value w-full"><?php echo htmlspecialchars($application['medical_condition'] ?: 'Good Health'); ?></div>
                </div>
            </div>
            
            <div class="field-row">
                <div class="field w-third">
                    <span class="field-label">Driving License:</span>
                    <div class="field-value w-full"><?php echo htmlspecialchars($application['driving_license_no'] ?: 'Not provided'); ?></div>
                </div>
                <div class="field w-auto">
                    <span class="field-label">License Type:</span>
                    <div class="checkbox-group">
                        <div class="checkbox-item">
                            <div class="checkbox <?php echo (isset($application['driving_license_type']) && in_array($application['driving_license_type'], ['heavy_vehicle', 'both'])) ? 'checked' : ''; ?>">
                                <?php echo (isset($application['driving_license_type']) && in_array($application['driving_license_type'], ['heavy_vehicle', 'both'])) ? '✓' : ''; ?>
                            </div>
                            <span>Heavy Vehicle</span>
                        </div>
                        <div class="checkbox-item">
                            <div class="checkbox <?php echo (isset($application['driving_license_type']) && in_array($application['driving_license_type'], ['light_vehicle', 'both'])) ? 'checked' : ''; ?>">
                                <?php echo (isset($application['driving_license_type']) && in_array($application['driving_license_type'], ['light_vehicle', 'both'])) ? '✓' : ''; ?>
                            </div>
                            <span>Light Vehicle</span>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Employment Information -->
    <div class="section">
        <div class="section-title">💼 Employment Information</div>
        
        <div class="field-group">
            <div class="field-row">
                <div class="field w-auto">
                    <span class="field-label">Currently Employed:</span>
                    <div class="checkbox-group">
                        <div class="checkbox-item">
                            <div class="checkbox <?php echo $application['currently_employed'] ? 'checked' : ''; ?>">
                                <?php echo $application['currently_employed'] ? '✓' : ''; ?>
                            </div>
                            <span>Yes</span>
                        </div>
                        <div class="checkbox-item">
                            <div class="checkbox <?php echo !$application['currently_employed'] ? 'checked' : ''; ?>">
                                <?php echo !$application['currently_employed'] ? '✓' : ''; ?>
                            </div>
                            <span>No</span>
                        </div>
                    </div>
                </div>
            </div>
            
            <div class="field-row">
                <div class="field w-full">
                    <span class="field-label">Position Applied For:</span>
                    <div class="field-value w-full" style="font-weight: bold; background: #e3f2fd; border-color: #2196f3;">
                        <?php echo htmlspecialchars($application['position_applied']); ?>
                    </div>
                </div>
            </div>
        </div>

        <?php if ($application['currently_employed']): ?>
        <div class="table-container">
            <table class="table">
                <thead>
                    <tr>
                        <th>Current Employer</th>
                        <th>Position</th>
                        <th>Employment Start</th>
                        <th>Current Salary</th>
                        <th>Reason for Leaving</th>
                    </tr>
                </thead>
                <tbody>
                    <tr>
                        <td><?php echo htmlspecialchars($application['current_employer']); ?></td>
                        <td><?php echo htmlspecialchars($application['current_position']); ?></td>
                        <td><?php echo $application['employment_start_date'] ? formatDisplayDate($application['employment_start_date']) : '-'; ?></td>
                        <td><?php echo $application['current_salary'] ? 'Rs. ' . number_format($application['current_salary'], 2) : '-'; ?></td>
                        <td><?php echo htmlspecialchars($application['reason_for_leaving'] ?: 'Career advancement'); ?></td>
                    </tr>
                </tbody>
            </table>
        </div>
        <?php endif; ?>
    </div>

    <!-- Educational Qualifications -->
    <div class="section">
        <div class="section-title">🎓 Educational Qualifications</div>
        
        <div class="table-container">
            <table class="table">
                <thead>
                    <tr>
                        <th style="width: 30%;">Institution / School</th>
                        <th style="width: 15%;">Year Completed</th>
                        <th style="width: 30%;">Qualification</th>
                        <th style="width: 25%;">Specialization / Results</th>
                    </tr>
                </thead>
                <tbody>
                    <tr>
                        <td>
                            <strong>Primary Education</strong><br>
                            <?php echo htmlspecialchars($application['elementary_school'] ?: 'Not specified'); ?>
                        </td>
                        <td><?php echo $application['elementary_year_completed'] ?: '-'; ?></td>
                        <td>Primary Education Certificate</td>
                        <td>-</td>
                    </tr>
                    <tr>
                        <td>
                            <strong>Secondary Education</strong><br>
                            <?php echo htmlspecialchars($application['high_school'] ?: 'Not specified'); ?>
                        </td>
                        <td><?php echo $application['high_school_year_completed'] ?: '-'; ?></td>
                        <td>
                            O/L Results: <?php echo htmlspecialchars($application['ol_results'] ?: 'Not provided'); ?><br>
                            A/L Results: <?php echo htmlspecialchars($application['al_results'] ?: 'Not provided'); ?>
                        </td>
                        <td>-</td>
                    </tr>
                    <?php if ($application['college_university']): ?>
                    <tr>
                        <td>
                            <strong>Higher Education</strong><br>
                            <?php echo htmlspecialchars($application['college_university']); ?>
                        </td>
                        <td><?php echo $application['graduation_year'] ?: '-'; ?></td>
                        <td>
                            <?php echo htmlspecialchars($application['degree_diploma']); ?>
                            <?php if ($application['gpa_class']): ?>
                            <br><strong>Class:</strong> <?php echo htmlspecialchars($application['gpa_class']); ?>
                            <?php endif; ?>
                        </td>
                        <td><?php echo htmlspecialchars($application['certifications'] ?: '-'); ?></td>
                    </tr>
                    <?php endif; ?>
                    <?php if ($application['other_education'] || $application['training_courses']): ?>
                    <tr>
                        <td>
                            <strong>Other Education/Training</strong><br>
                            <?php echo htmlspecialchars($application['other_education'] ?: 'Various institutions'); ?>
                        </td>
                        <td>-</td>
                        <td><?php echo htmlspecialchars($application['training_courses'] ?: 'Professional courses'); ?></td>
                        <td>-</td>
                    </tr>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
    </div>

    <!-- Banking Information -->
    <div class="section">
        <div class="section-title">🏦 Banking Information</div>
        
        <div class="field-group">
            <div class="field-row">
                <div class="field w-third">
                    <span class="field-label">Bank Name:</span>
                    <div class="field-value w-full"><?php echo htmlspecialchars($application['bank_name'] ?: 'Not provided'); ?></div>
                </div>
                <div class="field w-third">
                    <span class="field-label">Branch:</span>
                    <div class="field-value w-full"><?php echo htmlspecialchars($application['bank_branch'] ?: 'Not provided'); ?></div>
                </div>
                <div class="field w-third">
                    <span class="field-label">Account Number:</span>
                    <div class="field-value w-full"><?php echo htmlspecialchars($application['account_number'] ?: 'Not provided'); ?></div>
                </div>
            </div>
        </div>
    </div>

    <!-- Skills and Qualifications -->
    <div class="section">
        <div class="section-title">🛠️ Skills & Additional Information</div>
        
        <div class="field-group">
            <div class="field-row">
                <div class="field w-full">
                    <span class="field-label">Special Skills & Qualifications:</span>
                </div>
            </div>
            <div class="textarea-field">
                <?php 
                $skills_text = [];
                if ($application['special_skills']) $skills_text[] = "🔧 Special Skills: " . $application['special_skills'];
                if ($application['computer_skills']) $skills_text[] = "💻 Computer Skills: " . $application['computer_skills'];
                if ($application['language_skills']) $skills_text[] = "🗣️ Language Skills: " . $application['language_skills'];
                if ($application['hobbies']) $skills_text[] = "🎯 Hobbies & Interests: " . $application['hobbies'];
                if ($application['training_courses']) $skills_text[] = "📚 Training & Courses: " . $application['training_courses'];
                
                echo htmlspecialchars(implode("\n\n", $skills_text) ?: 'No additional skills specified.');
                ?>
            </div>
            
            <?php if (!empty($application['employee_references'])): ?>
            <div class="field-row">
                <div class="field w-full">
                    <span class="field-label">Employee References:</span>
                </div>
            </div>
            <div class="textarea-field">
                <?php echo nl2br(htmlspecialchars($application['employee_references'])); ?>
            </div>
            <?php endif; ?>
        </div>
    </div>

    <!-- Declaration -->
    <div class="declaration">
        <strong>Declaration:</strong> I hereby certify that the information provided in this application is true and complete to the best of my knowledge. I understand that any false information may result in my disqualification from employment consideration or dismissal if employed. I authorize investigation of all statements contained in this application and also authorize former employers and persons named as references to provide information concerning my previous employment and character. I understand that employment may be subject to verification of the information provided.
    </div>

    <!-- Signature Section -->
    <div class="signature-section">
        <div class="signature-row">
            <div class="signature-block">
                <div class="signature-label">Applicant Signature</div>
                <div class="signature-line"></div>
                <div class="signature-text">Signature</div>
            </div>
            <div class="signature-block">
                <div class="signature-label">Date</div>
                <div class="signature-line"><?php echo date('Y-m-d'); ?></div>
                <div class="signature-text">Date Signed</div>
            </div>
        </div>
    </div>

    <!-- Office Use Only Section -->
    <div class="office-section">
        <div class="office-title">Office use only.</div>
        
        <div class="office-grid">
            <div class="office-column">
                <div class="office-role">Recruited Officer</div>
                <div class="office-info-box">
                    <!-- Blank box for writing -->
                </div>
                <div class="office-signature">Signature</div>
            </div>
            
            <div class="office-column">
                <div class="office-role">Admin Officer</div>
                <div class="office-info-box">
                    <!-- Blank box for writing -->
                </div>
                <div class="office-signature">Signature</div>
            </div>
            
            <div class="office-column">
                <div class="office-role">Project Manager</div>
                <div class="office-info-box">
                    <!-- Blank box for writing -->
                </div>
                <div class="office-signature">Signature</div>
            </div>
        </div>
    </div>

    <!-- Footer -->
    <div style="margin-top: 30px; text-align: center; font-size: 10px; color: #666; border-top: 1px solid #ddd; padding-top: 15px;">
        <p><strong>RR Construction (Pvt) Ltd.</strong> - Human Resources Department</p>
        <p>This document is confidential and intended solely for internal use.</p>
        <p>Generated on: <?php echo date('F j, Y \a\t g:i A'); ?></p>
    </div>
</body>
</html>