<?php
/**
 * View Application Details - Enhanced Version
 * Employee Recruitment Management System
 * RR Construction (Pvt) Ltd
 */

$page_title = 'Application Details';
$active_menu = 'applications';

require_once '../includes/header.php';
requirePermission(ROLE_SITE_ADMIN);

$currentUser = getCurrentUser();
$application_id = (int)($_GET['id'] ?? 0);

// Get application details
$application = null;
if ($application_id) {
    try {
        $sql = "SELECT ja.*, er.request_number, er.site_project, er.request_date, er.urgent_requirement,
                       ec.position_title, ec.category, ec.quantity, ec.salary_range_min, ec.salary_range_max,
                       u.full_name as submitted_by_name
                FROM job_applications ja
                JOIN employee_requests er ON ja.request_id = er.id
                JOIN employee_categories ec ON ja.category_id = ec.id
                JOIN users u ON ja.submitted_by = u.id
                WHERE ja.id = ?";
        
        // Add role-based access control
        $params = [$application_id];
        
        switch ($currentUser['role']) {
            case ROLE_SUPER_ADMIN:
                // Super admin can view all applications
                break;
                
            case ROLE_SITE_ADMIN:
                // Site admin can only view their own applications
                $sql .= " AND ja.submitted_by = ?";
                $params[] = $currentUser['id'];
                break;
                
            case ROLE_PROJECT_MANAGER:
                // Project manager can view applications from their project
                $sql .= " AND er.site_project = ?";
                $params[] = $currentUser['site_project'];
                break;
                
            case ROLE_GENERAL_MANAGER:
            case ROLE_HR_MANAGER:
            case ROLE_ACCOUNTANT:
                // These roles can view all applications
                break;
                
            default:
                // No access for other roles
                $sql .= " AND 1=0"; // This will return no results
                break;
        }
        
        $application = fetchOne($sql, $params);
    } catch (Exception $e) {
        error_log("Error fetching application: " . $e->getMessage());
    }
}

if (!$application) {
    setFlashMessage('Application not found or access denied.', 'danger');
    header('Location: view_applications.php');
    exit;
}

// Get application documents
$documents = [];
try {
    $sql = "SELECT ed.*, u.full_name as uploaded_by_name
            FROM employee_documents ed
            LEFT JOIN users u ON ed.uploaded_by = u.id
            WHERE ed.application_id = ?
            ORDER BY ed.document_type, ed.uploaded_at DESC";
    $documents = fetchAll($sql, [$application_id]);
} catch (Exception $e) {
    error_log("Error fetching documents: " . $e->getMessage());
}

// Get approval history
$approval_history = [];
try {
    $sql = "SELECT aal.*, u.full_name as approver_name
            FROM application_approval_logs aal
            JOIN users u ON aal.approved_by = u.id
            WHERE aal.application_id = ?
            ORDER BY aal.approved_at ASC";
    $approval_history = fetchAll($sql, [$application_id]);
} catch (Exception $e) {
    error_log("Error fetching approval history: " . $e->getMessage());
}

// Document types mapping
$document_types = [
    'nic_copy' => 'NIC Copy',
    'cv' => 'CV/Resume',
    'certificates' => 'Educational Certificates',
    'photo' => 'Passport Photo',
    'birth_certificate' => 'Birth Certificate',
    'bank_details' => 'Bank Details',
    'medical_report' => 'Medical Report',
    'police_report' => 'Police Report',
    'other' => 'Other Documents'
];

// Calculate completion percentage
$required_fields = [
    'full_name_english', 'nic_number', 'date_of_birth', 'gender', 'marital_status',
    'permanent_address', 'phone_mobile', 'position_applied'
];

$completed_fields = 0;
foreach ($required_fields as $field) {
    if (!empty($application[$field])) {
        $completed_fields++;
    }
}
$completion_percentage = round(($completed_fields / count($required_fields)) * 100);

// Required documents check
$required_docs = ['nic_copy', 'cv', 'certificates', 'photo'];
$uploaded_doc_types = array_column($documents, 'document_type');
$missing_docs = array_diff($required_docs, $uploaded_doc_types);

// Helper function to display field value or dash if empty
function displayField($value, $format = 'text') {
    if (empty($value) || $value === '0000-00-00' || $value === '0000-00-00 00:00:00') {
        return '<span class="text-muted">-</span>';
    }
    
    switch ($format) {
        case 'date':
            return formatDisplayDate($value);
        case 'datetime':
            return formatDisplayDateTime($value);
        case 'currency':
            return formatCurrencyAmount($value);
        case 'boolean':
            return $value ? '<span class="text-success">Yes</span>' : '<span class="text-muted">No</span>';
        case 'capitalize':
            return ucfirst(str_replace('_', ' ', $value));
        default:
            return htmlspecialchars($value);
    }
}
?>

<div class="row mb-4">
    <div class="col-12">
        <div class="d-flex justify-content-between align-items-center">
            <div>
                <h4 class="mb-0">Application Details</h4>
                <small class="text-muted">
                    <?php echo htmlspecialchars($application['application_number']); ?> - 
                    <?php echo htmlspecialchars($application['full_name_english']); ?>
                </small>
            </div>
            <div class="btn-group">
                <a href="view_applications.php" class="btn btn-outline-secondary">
                    <i class="fas fa-arrow-left"></i> Back to Applications
                </a>
                <?php if ($currentUser['role'] === ROLE_SITE_ADMIN): ?>
                <a href="../documents/upload.php?app_id=<?php echo $application_id; ?>" class="btn btn-primary">
                    <i class="fas fa-upload"></i> Upload Documents
                </a>
                <?php else: ?>
                <a href="../documents/view.php?app_id=<?php echo $application_id; ?>" class="btn btn-primary">
                    <i class="fas fa-folder"></i> View Documents
                </a>
                <?php endif; ?>
                <a href="print_application.php?id=<?php echo $application_id; ?>" class="btn btn-info" target="_blank">
                    <i class="fas fa-print"></i> Print
                </a>
                <?php 
                // Show approval button if user can approve this application
                $can_approve = false;
                if ($currentUser['role'] === ROLE_PROJECT_MANAGER && $application['status'] === 'submitted') {
                    $can_approve = true;
                } elseif ($currentUser['role'] === ROLE_HR_MANAGER && $application['status'] === 'approved_by_pm') {
                    $can_approve = true;
                } elseif ($currentUser['role'] === ROLE_ACCOUNTANT && $application['status'] === 'approved_by_hr') {
                    $can_approve = true;
                }
                
                if ($can_approve): ?>
                <a href="../employees/approve_application.php?id=<?php echo $application_id; ?>" class="btn btn-success">
                    <i class="fas fa-check"></i> Review & Approve
                </a>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<!-- Application Status and Progress -->
<div class="row mb-4">
    <div class="col-md-8">
        <div class="card">
            <div class="card-header">Application Status</div>
            <div class="card-body">
                <div class="row">
                    <div class="col-md-6">
                        <h5>Current Status</h5>
                        <?php echo getStatusBadge($application['status'], 'application'); ?>
                        <p class="mt-2 mb-0">
                            <small class="text-muted">
                                Submitted on <?php echo formatDisplayDateTime($application['submitted_at']); ?>
                            </small>
                        </p>
                    </div>
                    <div class="col-md-6">
                        <h5>Application Progress</h5>
                        <div class="progress mb-2" style="height: 20px;">
                            <div class="progress-bar bg-info" style="width: <?php echo $completion_percentage; ?>%">
                                <?php echo $completion_percentage; ?>% Complete
                            </div>
                        </div>
                        <small class="text-muted">
                            <?php echo $completed_fields; ?> of <?php echo count($required_fields); ?> required fields completed
                        </small>
                    </div>
                </div>
            </div>
        </div>
    </div>
    <div class="col-md-4">
        <div class="card">
            <div class="card-header">Quick Actions</div>
            <div class="card-body">
                <div class="d-grid gap-2">
                    <?php if ($application['status'] === 'submitted' && $currentUser['role'] === ROLE_SITE_ADMIN): ?>
                    <a href="edit_application.php?id=<?php echo $application_id; ?>" class="btn btn-warning btn-sm">
                        <i class="fas fa-edit"></i> Edit Application
                    </a>
                    <?php endif; ?>
                    
                    <?php if ($currentUser['role'] === ROLE_SITE_ADMIN): ?>
                    <a href="../documents/manage.php?app_id=<?php echo $application_id; ?>" class="btn btn-primary btn-sm">
                        <i class="fas fa-folder"></i> Manage Documents
                    </a>
                    <?php else: ?>
                    <a href="../documents/view.php?app_id=<?php echo $application_id; ?>" class="btn btn-primary btn-sm">
                        <i class="fas fa-folder"></i> View Documents
                    </a>
                    <?php endif; ?>
                    
                    <a href="application_timeline.php?id=<?php echo $application_id; ?>" class="btn btn-info btn-sm">
                        <i class="fas fa-history"></i> View Timeline
                    </a>
                    
                    <?php if ($can_approve): ?>
                    <a href="../employees/approve_application.php?id=<?php echo $application_id; ?>" class="btn btn-success btn-sm">
                        <i class="fas fa-check"></i> Review & Approve
                    </a>
                    <?php endif; ?>
                    
                    <?php if (in_array($application['status'], ['submitted', 'under_review']) && $currentUser['role'] === ROLE_SITE_ADMIN): ?>
                    <button class="btn btn-outline-danger btn-sm" onclick="withdrawApplication()">
                        <i class="fas fa-times"></i> Withdraw Application
                    </button>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Personal Information -->
<div class="card mb-4">
    <div class="card-header">
        <i class="fas fa-user me-2"></i>Personal Information
    </div>
    <div class="card-body">
        <div class="row">
            <div class="col-md-6">
                <table class="table table-borderless table-sm">
                    <tr>
                        <td width="180"><strong>Full Name (English):</strong></td>
                        <td><?php echo displayField($application['full_name_english']); ?></td>
                    </tr>
                    <tr>
                        <td><strong>Full Name (Sinhala):</strong></td>
                        <td><?php echo displayField($application['full_name_sinhala']); ?></td>
                    </tr>
                    <tr>
                        <td><strong>NIC Number:</strong></td>
                        <td><strong class="text-primary"><?php echo displayField($application['nic_number']); ?></strong></td>
                    </tr>
                    <tr>
                        <td><strong>Date of Birth:</strong></td>
                        <td><?php echo displayField($application['date_of_birth'], 'date'); ?> 
                            <?php if ($application['age']): ?>
                                <span class="text-muted">(Age: <?php echo $application['age']; ?>)</span>
                            <?php endif; ?>
                        </td>
                    </tr>
                    <tr>
                        <td><strong>Gender:</strong></td>
                        <td><?php echo displayField($application['gender'], 'capitalize'); ?></td>
                    </tr>
                    <tr>
                        <td><strong>Marital Status:</strong></td>
                        <td><?php echo displayField($application['marital_status'], 'capitalize'); ?></td>
                    </tr>
                    <tr>
                        <td><strong>Nationality:</strong></td>
                        <td><?php echo displayField($application['nationality']); ?></td>
                    </tr>
                </table>
            </div>
            <div class="col-md-6">
                <table class="table table-borderless table-sm">
                    <tr>
                        <td width="150"><strong>Religion:</strong></td>
                        <td><?php echo displayField($application['religion']); ?></td>
                    </tr>
                    <tr>
                        <td><strong>Blood Group:</strong></td>
                        <td><?php echo displayField($application['blood_group']); ?></td>
                    </tr>
                    <tr>
                        <td><strong>District:</strong></td>
                        <td><?php echo displayField($application['district']); ?></td>
                    </tr>
                    <tr>
                        <td><strong>Police Station:</strong></td>
                        <td><?php echo displayField($application['police_station']); ?></td>
                    </tr>
                    <tr>
                        <td><strong>Medical Condition:</strong></td>
                        <td><?php echo displayField($application['medical_condition']); ?></td>
                    </tr>
                    <tr>
                        <td><strong>Transportation:</strong></td>
                        <td><?php echo displayField($application['transportation_method'], 'capitalize'); ?></td>
                    </tr>
                    <tr>
                        <td><strong>Driving License:</strong></td>
                        <td>
                            <?php if (!empty($application['driving_license_no'])): ?>
                                <?php echo displayField($application['driving_license_no']); ?>
                                <br><small class="text-muted">Type: <?php echo displayField($application['driving_license_type'], 'capitalize'); ?></small>
                            <?php else: ?>
                                <span class="text-muted">No License</span>
                            <?php endif; ?>
                        </td>
                    </tr>
                </table>
            </div>
        </div>
    </div>
</div>

<!-- Contact Information -->
<div class="card mb-4">
    <div class="card-header">
        <i class="fas fa-address-book me-2"></i>Contact & Address Information
    </div>
    <div class="card-body">
        <div class="row">
            <div class="col-md-6">
                <h6 class="text-primary">Address Information</h6>
                <table class="table table-borderless table-sm">
                    <tr>
                        <td width="140"><strong>Permanent Address:</strong></td>
                        <td><?php echo nl2br(displayField($application['permanent_address'])); ?></td>
                    </tr>
                    <?php if (!empty($application['current_address'])): ?>
                    <tr>
                        <td><strong>Current Address:</strong></td>
                        <td><?php echo nl2br(displayField($application['current_address'])); ?></td>
                    </tr>
                    <?php endif; ?>
                </table>
                
                <h6 class="text-primary mt-3">Contact Information</h6>
                <table class="table table-borderless table-sm">
                    <tr>
                        <td width="140"><strong>Mobile Phone:</strong></td>
                        <td><strong class="text-success"><?php echo displayField($application['phone_mobile']); ?></strong></td>
                    </tr>
                    <tr>
                        <td><strong>Home Phone:</strong></td>
                        <td><?php echo displayField($application['phone_home']); ?></td>
                    </tr>
                    <tr>
                        <td><strong>Email:</strong></td>
                        <td><?php echo displayField($application['email']); ?></td>
                    </tr>
                </table>
            </div>
            <div class="col-md-6">
                <?php if (!empty($application['emergency_contact_name'])): ?>
                <h6 class="text-primary">Emergency Contact</h6>
                <table class="table table-borderless table-sm">
                    <tr>
                        <td width="120"><strong>Name:</strong></td>
                        <td><?php echo displayField($application['emergency_contact_name']); ?></td>
                    </tr>
                    <tr>
                        <td><strong>Phone:</strong></td>
                        <td><?php echo displayField($application['emergency_contact_phone']); ?></td>
                    </tr>
                    <tr>
                        <td><strong>Relationship:</strong></td>
                        <td><?php echo displayField($application['emergency_contact_relationship']); ?></td>
                    </tr>
                </table>
                <?php endif; ?>
                
                <h6 class="text-primary mt-3">Skills & Languages</h6>
                <table class="table table-borderless table-sm">
                    <tr>
                        <td width="120"><strong>Language Skills:</strong></td>
                        <td><?php echo nl2br(displayField($application['language_skills'])); ?></td>
                    </tr>
                    <tr>
                        <td><strong>Hobbies:</strong></td>
                        <td><?php echo nl2br(displayField($application['hobbies'])); ?></td>
                    </tr>
                </table>
            </div>
        </div>
    </div>
</div>

<!-- Employment Information -->
<div class="card mb-4">
    <div class="card-header">
        <i class="fas fa-briefcase me-2"></i>Employment Information
    </div>
    <div class="card-body">
        <div class="row">
            <div class="col-md-6">
                <h6 class="text-primary">Position Details</h6>
                <table class="table table-borderless table-sm">
                    <tr>
                        <td width="150"><strong>Position Applied:</strong></td>
                        <td><strong class="text-primary"><?php echo displayField($application['position_applied']); ?></strong></td>
                    </tr>
                    <tr>
                        <td><strong>Category:</strong></td>
                        <td><?php echo displayField($application['category']); ?></td>
                    </tr>
                    <tr>
                        <td><strong>Request Number:</strong></td>
                        <td><?php echo displayField($application['request_number']); ?></td>
                    </tr>
                    <tr>
                        <td><strong>Site/Project:</strong></td>
                        <td><?php echo displayField($application['site_project']); ?></td>
                    </tr>
                    <tr>
                        <td><strong>Expected Salary:</strong></td>
                        <td><?php echo displayField($application['expected_salary'], 'currency'); ?></td>
                    </tr>
                    <tr>
                        <td><strong>Available Start Date:</strong></td>
                        <td><?php echo displayField($application['available_start_date'], 'date'); ?></td>
                    </tr>
                    <?php if ($currentUser['role'] !== ROLE_SITE_ADMIN): ?>
                    <tr>
                        <td><strong>Submitted by:</strong></td>
                        <td><?php echo displayField($application['submitted_by_name']); ?></td>
                    </tr>
                    <?php endif; ?>
                </table>
            </div>
            <div class="col-md-6">
                <?php if ($application['currently_employed']): ?>
                <h6 class="text-primary">Current Employment</h6>
                <table class="table table-borderless table-sm">
                    <tr>
                        <td width="150"><strong>Currently Employed:</strong></td>
                        <td><?php echo displayField($application['currently_employed'], 'boolean'); ?></td>
                    </tr>
                    <tr>
                        <td><strong>Current Employer:</strong></td>
                        <td><?php echo displayField($application['current_employer']); ?></td>
                    </tr>
                    <tr>
                        <td><strong>Current Position:</strong></td>
                        <td><?php echo displayField($application['current_position']); ?></td>
                    </tr>
                    <tr>
                        <td><strong>Current Salary:</strong></td>
                        <td><?php echo displayField($application['current_salary'], 'currency'); ?></td>
                    </tr>
                    <tr>
                        <td><strong>Employment Start:</strong></td>
                        <td><?php echo displayField($application['employment_start_date'], 'date'); ?></td>
                    </tr>
                    <tr>
                        <td><strong>Reason for Leaving:</strong></td>
                        <td><?php echo displayField($application['reason_for_leaving']); ?></td>
                    </tr>
                </table>
                <?php else: ?>
                <div class="alert alert-info">
                    <i class="fas fa-info-circle me-2"></i>Currently not employed
                </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<!-- Previous Employment History -->
<?php if (!empty($application['prev1_employer']) || !empty($application['prev2_employer']) || !empty($application['prev3_employer'])): ?>
<div class="card mb-4">
    <div class="card-header">
        <i class="fas fa-history me-2"></i>Previous Employment History
    </div>
    <div class="card-body">
        <?php for ($i = 1; $i <= 3; $i++): ?>
            <?php 
            $employer = $application["prev{$i}_employer"];
            $position = $application["prev{$i}_position"];
            $duration = $application["prev{$i}_duration"];
            $salary = $application["prev{$i}_salary"];
            $reason = $application["prev{$i}_reason_leaving"];
            $start_date = $application["prev{$i}_start_date"] ?? '';
            $end_date = $application["prev{$i}_end_date"] ?? '';
            ?>
            <?php if (!empty($employer)): ?>
            <div class="border rounded p-3 mb-3">
                <h6 class="text-primary">Previous Employment <?php echo $i; ?></h6>
                <div class="row">
                    <div class="col-md-6">
                        <table class="table table-borderless table-sm">
                            <tr>
                                <td width="120"><strong>Employer:</strong></td>
                                <td><?php echo displayField($employer); ?></td>
                            </tr>
                            <tr>
                                <td><strong>Position:</strong></td>
                                <td><?php echo displayField($position); ?></td>
                            </tr>
                            <tr>
                                <td><strong>Duration:</strong></td>
                                <td><?php echo displayField($duration); ?></td>
                            </tr>
                        </table>
                    </div>
                    <div class="col-md-6">
                        <table class="table table-borderless table-sm">
                            <tr>
                                <td width="120"><strong>Salary:</strong></td>
                                <td><?php echo displayField($salary, 'currency'); ?></td>
                            </tr>
                            <?php if (!empty($start_date)): ?>
                            <tr>
                                <td><strong>Start Date:</strong></td>
                                <td><?php echo displayField($start_date, 'date'); ?></td>
                            </tr>
                            <?php endif; ?>
                            <?php if (!empty($end_date)): ?>
                            <tr>
                                <td><strong>End Date:</strong></td>
                                <td><?php echo displayField($end_date, 'date'); ?></td>
                            </tr>
                            <?php endif; ?>
                        </table>
                    </div>
                </div>
                <?php if (!empty($reason)): ?>
                <div class="row">
                    <div class="col-12">
                        <strong>Reason for Leaving:</strong> <?php echo displayField($reason); ?>
                    </div>
                </div>
                <?php endif; ?>
            </div>
            <?php endif; ?>
        <?php endfor; ?>
    </div>
</div>
<?php endif; ?>

<!-- Education Background -->
<div class="card mb-4">
    <div class="card-header">
        <i class="fas fa-graduation-cap me-2"></i>Education Background
    </div>
    <div class="card-body">
        <div class="row">
            <div class="col-md-6">
                <h6 class="text-primary">School Education</h6>
                <table class="table table-borderless table-sm">
                    <tr>
                        <td width="150"><strong>Elementary School:</strong></td>
                        <td><?php echo displayField($application['elementary_school']); ?></td>
                    </tr>
                    <tr>
                        <td><strong>Year Completed:</strong></td>
                        <td><?php echo displayField($application['elementary_year_completed']); ?></td>
                    </tr>
                    <tr>
                        <td><strong>High School:</strong></td>
                        <td><?php echo displayField($application['high_school']); ?></td>
                    </tr>
                    <tr>
                        <td><strong>Year Completed:</strong></td>
                        <td><?php echo displayField($application['high_school_year_completed']); ?></td>
                    </tr>
                    <tr>
                        <td><strong>O/L Results:</strong></td>
                        <td><?php echo displayField($application['ol_results']); ?></td>
                    </tr>
                    <tr>
                        <td><strong>A/L Results:</strong></td>
                        <td><?php echo displayField($application['al_results']); ?></td>
                    </tr>
                </table>
            </div>
            <div class="col-md-6">
                <h6 class="text-primary">Higher Education</h6>
                <table class="table table-borderless table-sm">
                    <tr>
                        <td width="150"><strong>College/University:</strong></td>
                        <td><?php echo displayField($application['college_university']); ?></td>
                    </tr>
                    <tr>
                        <td><strong>Degree/Diploma:</strong></td>
                        <td><?php echo displayField($application['degree_diploma']); ?></td>
                    </tr>
                    <tr>
                        <td><strong>Graduation Year:</strong></td>
                        <td><?php echo displayField($application['graduation_year']); ?></td>
                    </tr>
                    <tr>
                        <td><strong>GPA/Class:</strong></td>
                        <td><?php echo displayField($application['gpa_class']); ?></td>
                    </tr>
                </table>
                
                <?php if (!empty($application['other_education']) || !empty($application['certifications']) || !empty($application['training_courses'])): ?>
                <h6 class="text-primary mt-3">Additional Education</h6>
                <table class="table table-borderless table-sm">
                    <?php if (!empty($application['other_education'])): ?>
                    <tr>
                        <td width="150"><strong>Other Education:</strong></td>
                        <td><?php echo nl2br(displayField($application['other_education'])); ?></td>
                    </tr>
                    <?php endif; ?>
                    <?php if (!empty($application['certifications'])): ?>
                    <tr>
                        <td><strong>Certifications:</strong></td>
                        <td><?php echo nl2br(displayField($application['certifications'])); ?></td>
                    </tr>
                    <?php endif; ?>
                    <?php if (!empty($application['training_courses'])): ?>
                    <tr>
                        <td><strong>Training Courses:</strong></td>
                        <td><?php echo nl2br(displayField($application['training_courses'])); ?></td>
                    </tr>
                    <?php endif; ?>
                </table>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<!-- Skills & Banking -->
<div class="row mb-4">
    <div class="col-md-6">
        <div class="card h-100">
            <div class="card-header">
                <i class="fas fa-tools me-2"></i>Skills & Abilities
            </div>
            <div class="card-body">
                <?php if (!empty($application['special_skills'])): ?>
                <h6 class="text-primary">Special Skills</h6>
                <p class="mb-3"><?php echo nl2br(displayField($application['special_skills'])); ?></p>
                <?php endif; ?>
                
                <?php if (!empty($application['computer_skills'])): ?>
                <h6 class="text-primary">Computer Skills</h6>
                <p class="mb-3"><?php echo nl2br(displayField($application['computer_skills'])); ?></p>
                <?php endif; ?>
                
                <?php if (!empty($application['employee_references'])): ?>
                <h6 class="text-primary">Employee References</h6>
                <p class="mb-0"><?php echo nl2br(displayField($application['employee_references'])); ?></p>
                <?php endif; ?>
                
                <?php if (empty($application['special_skills']) && empty($application['computer_skills']) && empty($application['employee_references'])): ?>
                <p class="text-muted">No skills information provided</p>
                <?php endif; ?>
            </div>
        </div>
    </div>
    <div class="col-md-6">
        <div class="card h-100">
            <div class="card-header">
                <i class="fas fa-university me-2"></i>Banking Information
            </div>
            <div class="card-body">
                <?php if (!empty($application['bank_name'])): ?>
                <table class="table table-borderless table-sm">
                    <tr>
                        <td width="140"><strong>Bank Name:</strong></td>
                        <td><?php echo displayField($application['bank_name']); ?></td>
                    </tr>
                    <tr>
                        <td><strong>Branch:</strong></td>
                        <td><?php echo displayField($application['bank_branch']); ?></td>
                    </tr>
                    <tr>
                        <td><strong>Account Number:</strong></td>
                        <td><strong class="text-primary"><?php echo displayField($application['account_number']); ?></strong></td>
                    </tr>
                    <tr>
                        <td><strong>Account Holder:</strong></td>
                        <td><?php echo displayField($application['account_holder_name']); ?></td>
                    </tr>
                </table>
                <?php else: ?>
                <p class="text-muted">Banking information not provided</p>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<!-- Medical & Background Checks -->
<?php if (!empty($application['medical_check_status']) || !empty($application['background_check_status'])): ?>
<div class="card mb-4">
    <div class="card-header">
        <i class="fas fa-user-shield me-2"></i>Medical & Background Checks
    </div>
    <div class="card-body">
        <div class="row">
            <div class="col-md-6">
                <?php if (!empty($application['medical_check_status'])): ?>
                <h6 class="text-primary">Medical Check</h6>
                <p>
                    Status: 
                    <?php 
                    $medical_status = $application['medical_check_status'];
                    $status_class = $medical_status === 'passed' ? 'success' : ($medical_status === 'failed' ? 'danger' : 'warning');
                    ?>
                    <span class="badge bg-<?php echo $status_class; ?>"><?php echo ucfirst($medical_status); ?></span>
                </p>
                <?php endif; ?>
            </div>
            <div class="col-md-6">
                <?php if (!empty($application['background_check_status'])): ?>
                <h6 class="text-primary">Background Check</h6>
                <p>
                    Status: 
                    <?php 
                    $bg_status = $application['background_check_status'];
                    $status_class = $bg_status === 'cleared' ? 'success' : ($bg_status === 'issues' ? 'danger' : 'warning');
                    ?>
                    <span class="badge bg-<?php echo $status_class; ?>"><?php echo ucfirst($bg_status); ?></span>
                </p>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>
<?php endif; ?>

<!-- Documents -->
<div class="card mb-4">
    <div class="card-header d-flex justify-content-between align-items-center">
        <span><i class="fas fa-folder me-2"></i>Uploaded Documents (<?php echo count($documents); ?>)</span>
        <?php if ($currentUser['role'] === ROLE_SITE_ADMIN): ?>
        <a href="../documents/manage.php?app_id=<?php echo $application_id; ?>" class="btn btn-sm btn-outline-primary">
            <i class="fas fa-folder"></i> Manage Documents
        </a>
        <?php else: ?>
        <a href="../documents/view.php?app_id=<?php echo $application_id; ?>" class="btn btn-sm btn-outline-primary">
            <i class="fas fa-folder"></i> View Documents
        </a>
        <?php endif; ?>
    </div>
    <div class="card-body">
        <?php if (empty($documents)): ?>
        <div class="text-center py-3">
            <i class="fas fa-folder-open fa-2x text-muted mb-3"></i>
            <p class="text-muted">No documents uploaded yet</p>
            <?php if ($currentUser['role'] === ROLE_SITE_ADMIN): ?>
            <a href="../documents/upload.php?app_id=<?php echo $application_id; ?>" class="btn btn-primary">
                <i class="fas fa-upload"></i> Upload Documents
            </a>
            <?php endif; ?>
        </div>
        <?php else: ?>
        <div class="row">
            <?php foreach ($documents as $doc): ?>
            <div class="col-md-6 mb-3">
                <div class="border rounded p-3">
                    <div class="d-flex justify-content-between align-items-start">
                        <div class="flex-grow-1">
                            <h6 class="mb-1">
                                <i class="fas fa-file me-2"></i>
                                <?php echo htmlspecialchars($doc['document_name']); ?>
                            </h6>
                            <p class="mb-1">
                                <span class="badge bg-secondary">
                                    <?php echo $document_types[$doc['document_type']] ?? ucfirst($doc['document_type']); ?>
                                </span>
                            </p>
                            <small class="text-muted">
                                <?php echo htmlspecialchars($doc['original_filename']); ?>
                                (<?php echo formatFileSize($doc['file_size']); ?>)
                                <br>
                                Uploaded: <?php echo formatDisplayDateTime($doc['uploaded_at']); ?>
                                <?php if (!empty($doc['uploaded_by_name'])): ?>
                                    by <?php echo htmlspecialchars($doc['uploaded_by_name']); ?>
                                <?php endif; ?>
                            </small>
                        </div>
                        <div class="btn-group btn-group-sm">
                            <a href="../documents/download.php?id=<?php echo $doc['id']; ?>" 
                               class="btn btn-outline-primary" target="_blank" title="Download">
                                <i class="fas fa-download"></i>
                            </a>
                            <a href="../documents/download.php?id=<?php echo $doc['id']; ?>&view=1" 
                               class="btn btn-outline-info" target="_blank" title="View">
                                <i class="fas fa-eye"></i>
                            </a>
                        </div>
                    </div>
                </div>
            </div>
            <?php endforeach; ?>
        </div>
        
        <!-- Document Status -->
        <?php if (!empty($missing_docs)): ?>
        <div class="alert alert-warning mt-3">
            <i class="fas fa-exclamation-triangle me-2"></i>
            <strong>Missing Required Documents:</strong>
            <?php foreach ($missing_docs as $doc_type): ?>
                <span class="badge bg-warning text-dark ms-1"><?php echo $document_types[$doc_type]; ?></span>
            <?php endforeach; ?>
        </div>
        <?php else: ?>
        <div class="alert alert-success mt-3">
            <i class="fas fa-check-circle me-2"></i>
            All required documents have been uploaded.
        </div>
        <?php endif; ?>
        <?php endif; ?>
    </div>
</div>

<!-- Position & Salary Information (for HR/Accountant) -->
<?php if (in_array($currentUser['role'], [ROLE_HR_MANAGER, ROLE_ACCOUNTANT, ROLE_SUPER_ADMIN])): ?>
<div class="card mb-4">
    <div class="card-header">
        <i class="fas fa-chart-line me-2"></i>Position & Salary Analysis
    </div>
    <div class="card-body">
        <div class="row">
            <div class="col-md-6">
                <h6 class="text-primary">Position Requirements</h6>
                <table class="table table-borderless table-sm">
                    <tr>
                        <td width="150"><strong>Quantity Required:</strong></td>
                        <td><?php echo displayField($application['quantity']); ?></td>
                    </tr>
                    <?php if (!empty($application['salary_range_min']) || !empty($application['salary_range_max'])): ?>
                    <tr>
                        <td><strong>Salary Range:</strong></td>
                        <td>
                            <?php echo displayField($application['salary_range_min'], 'currency'); ?> - 
                            <?php echo displayField($application['salary_range_max'], 'currency'); ?>
                        </td>
                    </tr>
                    <?php endif; ?>
                    <tr>
                        <td><strong>Expected Salary:</strong></td>
                        <td><?php echo displayField($application['expected_salary'], 'currency'); ?></td>
                    </tr>
                </table>
            </div>
            <div class="col-md-6">
                <h6 class="text-primary">Request Information</h6>
                <table class="table table-borderless table-sm">
                    <tr>
                        <td width="150"><strong>Request Date:</strong></td>
                        <td><?php echo displayField($application['request_date'], 'date'); ?></td>
                    </tr>
                    <tr>
                        <td><strong>Urgent Request:</strong></td>
                        <td><?php echo displayField($application['urgent_requirement'], 'boolean'); ?></td>
                    </tr>
                    <tr>
                        <td><strong>Application Date:</strong></td>
                        <td><?php echo displayField($application['submitted_at'], 'datetime'); ?></td>
                    </tr>
                </table>
            </div>
        </div>
    </div>
</div>
<?php endif; ?>

<!-- Approval History -->
<?php if (!empty($approval_history)): ?>
<div class="card mb-4">
    <div class="card-header">
        <i class="fas fa-clipboard-check me-2"></i>Approval History
    </div>
    <div class="card-body">
        <div class="timeline">
            <?php foreach ($approval_history as $approval): ?>
            <div class="timeline-item">
                <div class="timeline-marker bg-<?php echo $approval['action'] === 'approved' ? 'success' : ($approval['action'] === 'rejected' ? 'danger' : 'warning'); ?>">
                    <i class="fas fa-<?php echo $approval['action'] === 'approved' ? 'check' : ($approval['action'] === 'rejected' ? 'times' : 'clock'); ?>"></i>
                </div>
                <div class="timeline-content">
                    <h6 class="mb-1">
                        <?php echo ucfirst($approval['action']); ?> by <?php echo ucfirst(str_replace('_', ' ', $approval['approval_level'])); ?>
                    </h6>
                    <p class="mb-1">
                        <strong><?php echo htmlspecialchars($approval['approver_name']); ?></strong>
                        <?php if (!empty($approval['comments'])): ?>
                        <br><em>"<?php echo htmlspecialchars($approval['comments']); ?>"</em>
                        <?php endif; ?>
                        <?php if (!empty($approval['salary_approved'])): ?>
                        <br><strong>Approved Salary:</strong> <?php echo formatCurrencyAmount($approval['salary_approved']); ?>
                        <?php endif; ?>
                        <?php if (!empty($approval['start_date_approved'])): ?>
                        <br><strong>Approved Start Date:</strong> <?php echo formatDisplayDate($approval['start_date_approved']); ?>
                        <?php endif; ?>
                    </p>
                    <small class="text-muted">
                        <?php echo formatDisplayDateTime($approval['approved_at']); ?>
                    </small>
                </div>
            </div>
            <?php endforeach; ?>
        </div>
    </div>
</div>
<?php endif; ?>

<!-- Application Summary (for quick overview) -->
<div class="card mb-4">
    <div class="card-header">
        <i class="fas fa-info-circle me-2"></i>Application Summary
    </div>
    <div class="card-body">
        <div class="row">
            <div class="col-md-3">
                <div class="text-center">
                    <h5 class="text-primary"><?php echo htmlspecialchars($application['application_number']); ?></h5>
                    <small class="text-muted">Application Number</small>
                </div>
            </div>
            <div class="col-md-3">
                <div class="text-center">
                    <h5 class="text-success"><?php echo count($documents); ?></h5>
                    <small class="text-muted">Documents Uploaded</small>
                </div>
            </div>
            <div class="col-md-3">
                <div class="text-center">
                    <h5 class="text-info"><?php echo $completion_percentage; ?>%</h5>
                    <small class="text-muted">Profile Complete</small>
                </div>
            </div>
            <div class="col-md-3">
                <div class="text-center">
                    <h5 class="text-warning"><?php echo count($approval_history); ?></h5>
                    <small class="text-muted">Approval Steps</small>
                </div>
            </div>
        </div>
        
        <hr class="my-3">
        
        <div class="row text-center">
            <div class="col-md-4">
                <strong>Contact:</strong> <?php echo htmlspecialchars($application['phone_mobile']); ?>
                <?php if (!empty($application['email'])): ?>
                    <br><small class="text-muted"><?php echo htmlspecialchars($application['email']); ?></small>
                <?php endif; ?>
            </div>
            <div class="col-md-4">
                <strong>Experience:</strong> 
                <?php 
                $exp_count = 0;
                for ($i = 1; $i <= 3; $i++) {
                    if (!empty($application["prev{$i}_employer"])) $exp_count++;
                }
                echo $exp_count > 0 ? "{$exp_count} Previous Job(s)" : "No Previous Employment";
                ?>
            </div>
            <div class="col-md-4">
                <strong>Education:</strong> 
                <?php if (!empty($application['college_university'])): ?>
                    University Level
                <?php elseif (!empty($application['al_results'])): ?>
                    A/L Completed
                <?php elseif (!empty($application['ol_results'])): ?>
                    O/L Completed
                <?php else: ?>
                    School Level
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<!-- Rejection Information (if applicable) -->
<?php if ($application['status'] === 'rejected' && !empty($application['rejection_reason'])): ?>
<div class="card mb-4 border-danger">
    <div class="card-header bg-danger text-white">
        <i class="fas fa-times-circle me-2"></i>Rejection Information
    </div>
    <div class="card-body">
        <p class="mb-0"><strong>Reason for Rejection:</strong></p>
        <p class="text-danger"><?php echo nl2br(htmlspecialchars($application['rejection_reason'])); ?></p>
    </div>
</div>
<?php endif; ?>

<script>
function withdrawApplication() {
    if (confirm('Are you sure you want to withdraw this application? This action cannot be undone.')) {
        const form = document.createElement('form');
        form.method = 'POST';
        form.action = 'withdraw_application.php';
        
        const csrfInput = document.createElement('input');
        csrfInput.type = 'hidden';
        csrfInput.name = 'csrf_token';
        csrfInput.value = '<?php echo generateCSRFToken(); ?>';
        
        const idInput = document.createElement('input');
        idInput.type = 'hidden';
        idInput.name = 'application_id';
        idInput.value = '<?php echo $application_id; ?>';
        
        form.appendChild(csrfInput);
        form.appendChild(idInput);
        
        document.body.appendChild(form);
        form.submit();
    }
}

// Print specific sections
function printSection(sectionId) {
    const section = document.getElementById(sectionId);
    if (section) {
        const printWindow = window.open('', '_blank');
        printWindow.document.write(`
            <html>
                <head>
                    <title>Application Details - <?php echo htmlspecialchars($application['application_number']); ?></title>
                    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
                    <style>
                        body { font-size: 12px; margin: 20px; }
                        .timeline-marker { width: 30px; height: 30px; }
                        @media print { .no-print { display: none !important; } }
                    </style>
                </head>
                <body>
                    <div class="container-fluid">
                        <h3 class="mb-3">Application Details</h3>
                        <h5 class="text-muted mb-4"><?php echo htmlspecialchars($application['application_number']); ?> - <?php echo htmlspecialchars($application['full_name_english']); ?></h5>
                        ${section.innerHTML}
                    </div>
                </body>
            </html>
        `);
        printWindow.document.close();
        printWindow.print();
    }
}

// Auto-refresh for real-time updates (every 5 minutes)
setInterval(function() {
    // Check for application status updates
    fetch('check_application_status.php?id=<?php echo $application_id; ?>')
        .then(response => response.json())
        .then(data => {
            if (data.status !== '<?php echo $application['status']; ?>') {
                // Status has changed, show notification
                if (confirm('Application status has been updated. Would you like to refresh the page to see the latest changes?')) {
                    location.reload();
                }
            }
        })
        .catch(error => console.log('Status check failed:', error));
}, 300000); // 5 minutes

// Smooth scrolling for anchor links
document.querySelectorAll('a[href^="#"]').forEach(anchor => {
    anchor.addEventListener('click', function (e) {
        e.preventDefault();
        const target = document.querySelector(this.getAttribute('href'));
        if (target) {
            target.scrollIntoView({
                behavior: 'smooth',
                block: 'start'
            });
        }
    });
});
</script>

<style>
.timeline {
    position: relative;
    padding: 0;
    list-style: none;
}

.timeline::before {
    content: '';
    position: absolute;
    left: 20px;
    top: 0;
    bottom: 0;
    width: 2px;
    background: #dee2e6;
}

.timeline-item {
    position: relative;
    margin-bottom: 20px;
    padding-left: 60px;
}

.timeline-marker {
    position: absolute;
    left: 0;
    top: 0;
    width: 40px;
    height: 40px;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    color: white;
    font-size: 14px;
}

.timeline-content {
    background: #f8f9fa;
    padding: 15px;
    border-radius: 8px;
    border-left: 3px solid #dee2e6;
}

.alert-sm {
    padding: 0.375rem 0.75rem;
    font-size: 0.875rem;
}

/* Print styles */
@media print {
    .btn, .no-print {
        display: none !important;
    }
    
    .card {
        border: 1px solid #dee2e6 !important;
        break-inside: avoid;
    }
    
    .timeline-marker {
        width: 30px !important;
        height: 30px !important;
        font-size: 12px !important;
    }
    
    .timeline-item {
        padding-left: 50px !important;
    }
}

/* Enhanced table styling for better readability */
.table-borderless td {
    padding: 0.5rem 0.75rem;
    border-bottom: 1px solid #f8f9fa;
}

.table-borderless tr:last-child td {
    border-bottom: none;
}

/* Status badge enhancements */
.badge {
    font-size: 0.75em;
    padding: 0.35em 0.65em;
}

/* Card header icons */
.card-header i {
    margin-right: 0.5rem;
    width: 16px;
    text-align: center;
}

/* Responsive adjustments */
@media (max-width: 768px) {
    .timeline-item {
        padding-left: 40px;
    }
    
    .timeline-marker {
        width: 30px;
        height: 30px;
        font-size: 12px;
    }
    
    .timeline::before {
        left: 15px;
    }
}
</style>

<?php require_once '../includes/footer.php'; ?>