<?php
/**
 * View Applications with Document Management and Request Filtering
 * Employee Recruitment Management System
 * RR Construction (Pvt) Ltd
 */

$page_title = 'Job Applications';
$active_menu = 'applications';

require_once '../includes/header.php';
requirePermission(ROLE_SITE_ADMIN);

$currentUser = getCurrentUser();

// Get filter parameters
$request_id = (int)($_GET['request_id'] ?? 0);
$status_filter = $_GET['status'] ?? '';
$search = $_GET['search'] ?? '';
$page = max(1, (int)($_GET['page'] ?? 1));
$records_per_page = 20;
$offset = ($page - 1) * $records_per_page;

// Get request details if filtering by request
$request_details = null;
if ($request_id > 0) {
    try {
        $sql = "SELECT er.*, u.full_name as requested_by_name
                FROM employee_requests er
                JOIN users u ON er.requested_by = u.id
                WHERE er.id = ?";
        
        // Add role-based access control for request
        $params = [$request_id];
        
        switch ($currentUser['role']) {
            case ROLE_SUPER_ADMIN:
                // Super admin can view all requests
                break;
                
            case ROLE_SITE_ADMIN:
                $sql .= " AND er.requested_by = ?";
                $params[] = $currentUser['id'];
                break;
                
            case ROLE_PROJECT_MANAGER:
                $sql .= " AND er.site_project = ?";
                $params[] = $currentUser['site_project'];
                break;
                
            case ROLE_GENERAL_MANAGER:
            case ROLE_HR_MANAGER:
            case ROLE_ACCOUNTANT:
                // These roles can view all requests
                break;
                
            default:
                $sql .= " AND 1=0"; // No access
                break;
        }
        
        $request_details = fetchOne($sql, $params);
        
        if (!$request_details) {
            setFlashMessage('Request not found or access denied.', 'danger');
            header('Location: view_requests.php');
            exit;
        }
    } catch (Exception $e) {
        error_log("Error fetching request details: " . $e->getMessage());
        setFlashMessage('Error loading request details', 'danger');
        header('Location: view_requests.php');
        exit;
    }
}

// Build application query conditions
$where_conditions = ['1=1'];
$params = [];

// Filter by specific request if provided
if ($request_id > 0) {
    $where_conditions[] = 'ja.request_id = ?';
    $params[] = $request_id;
}

// Role-based filtering (if not filtering by specific request)
if ($request_id == 0) {
    switch ($currentUser['role']) {
        case ROLE_SUPER_ADMIN:
            // Super admin can see all applications
            break;
            
        case ROLE_SITE_ADMIN:
            $where_conditions[] = 'ja.submitted_by = ?';
            $params[] = $currentUser['id'];
            break;
            
        case ROLE_PROJECT_MANAGER:
            $where_conditions[] = 'er.site_project = ?';
            $params[] = $currentUser['site_project'];
            break;
            
        case ROLE_GENERAL_MANAGER:
        case ROLE_HR_MANAGER:
        case ROLE_ACCOUNTANT:
            // These roles can see all applications
            break;
            
        default:
            $where_conditions[] = '1=0'; // No access
            break;
    }
}

// Status filter
if (!empty($status_filter)) {
    $where_conditions[] = 'ja.status = ?';
    $params[] = $status_filter;
}

// Search filter
if (!empty($search)) {
    $where_conditions[] = '(ja.application_number LIKE ? OR ja.full_name_english LIKE ? OR ja.nic_number LIKE ? OR ec.position_title LIKE ?)';
    $params[] = "%$search%";
    $params[] = "%$search%";
    $params[] = "%$search%";
    $params[] = "%$search%";
}

$where_clause = implode(' AND ', $where_conditions);

// Get total count
$count_sql = "SELECT COUNT(*) as total 
              FROM job_applications ja
              JOIN employee_requests er ON ja.request_id = er.id
              JOIN employee_categories ec ON ja.category_id = ec.id
              WHERE $where_clause";
$total_records = fetchOne($count_sql, $params)['total'];

// Get applications
$applications = [];
try {
    $sql = "SELECT ja.*, er.request_number, er.site_project, ec.position_title, ec.category,
                   u.full_name as submitted_by_name,
                   COUNT(ed.id) as document_count,
                   CASE 
                       WHEN COUNT(CASE WHEN ed.document_type = 'nic_copy' THEN 1 END) > 0 AND
                            COUNT(CASE WHEN ed.document_type = 'cv' THEN 1 END) > 0 AND
                            COUNT(CASE WHEN ed.document_type = 'certificates' THEN 1 END) > 0 AND
                            COUNT(CASE WHEN ed.document_type = 'photo' THEN 1 END) > 0
                       THEN 'Complete'
                       ELSE 'Incomplete'
                   END as document_status
            FROM job_applications ja
            JOIN employee_requests er ON ja.request_id = er.id
            JOIN employee_categories ec ON ja.category_id = ec.id
            JOIN users u ON ja.submitted_by = u.id
            LEFT JOIN employee_documents ed ON ja.id = ed.application_id
            WHERE $where_clause
            GROUP BY ja.id
            ORDER BY ja.submitted_at DESC
            LIMIT $records_per_page OFFSET $offset";
    
    $applications = fetchAll($sql, $params);
} catch (Exception $e) {
    error_log("Error fetching applications: " . $e->getMessage());
    setFlashMessage('Error loading applications', 'danger');
}

// Pagination data
$total_pages = ceil($total_records / $records_per_page);

// Build page title and description based on context
if ($request_details) {
    $page_title = 'Applications for ' . $request_details['request_number'];
    $page_description = 'Job applications for request: ' . htmlspecialchars($request_details['site_project']);
} else {
    switch ($currentUser['role']) {
        case ROLE_SITE_ADMIN:
            $page_title = 'My Applications';
            $page_description = 'View and manage your job applications';
            break;
        case ROLE_PROJECT_MANAGER:
            $page_title = 'Project Applications';
            $page_description = 'Applications for your project: ' . htmlspecialchars($currentUser['site_project'] ?? 'All Projects');
            break;
        case ROLE_GENERAL_MANAGER:
        case ROLE_HR_MANAGER:
        case ROLE_ACCOUNTANT:
            $page_title = 'All Applications';
            $page_description = 'Review and manage all job applications';
            break;
        default:
            $page_title = 'Applications';
            $page_description = 'View applications';
    }
}
?>

<div class="row mb-4">
    <div class="col-12">
        <div class="d-flex justify-content-between align-items-center">
            <div>
                <h4 class="mb-0"><?php echo $page_title; ?></h4>
                <small class="text-muted"><?php echo $page_description; ?></small>
                <?php if ($request_details): ?>
                <div class="mt-2">
                    <span class="badge bg-primary"><?php echo htmlspecialchars($request_details['request_number']); ?></span>
                    <span class="badge bg-info"><?php echo htmlspecialchars($request_details['site_project']); ?></span>
                    <span class="badge bg-secondary"><?php echo getStatusName($request_details['status']); ?></span>
                </div>
                <?php endif; ?>
            </div>
            <div class="btn-group">
                <?php if ($request_details): ?>
                <a href="../requests/view_requests.php" class="btn btn-outline-secondary">
                    <i class="fas fa-arrow-left"></i> Back to Requests
                </a>
                <button type="button" class="btn btn-outline-info" onclick="viewRequestDetails()">
                    <i class="fas fa-info-circle"></i> Request Details
                </button>
                <?php endif; ?>
                <?php if ($currentUser['role'] === ROLE_SITE_ADMIN): ?>
                <a href="application_form.php<?php echo $request_id ? '?request_id=' . $request_id : ''; ?>" class="btn btn-primary">
                    <i class="fas fa-plus"></i> New Application
                </a>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<!-- Filters -->
<div class="card mb-3">
    <div class="card-body">
        <form method="GET" class="row g-3">
            <?php if ($request_id > 0): ?>
            <input type="hidden" name="request_id" value="<?php echo $request_id; ?>">
            <?php endif; ?>
            
            <div class="col-md-3">
                <label class="form-label">Status</label>
                <select name="status" class="form-control">
                    <option value="">All Status</option>
                    <?php foreach (APPLICATION_STATUSES as $key => $value): ?>
                        <option value="<?php echo $key; ?>" <?php echo $status_filter === $key ? 'selected' : ''; ?>>
                            <?php echo $value; ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
            <div class="col-md-6">
                <label class="form-label">Search</label>
                <input type="text" name="search" class="form-control" 
                       placeholder="Application number, name, NIC, or position" 
                       value="<?php echo htmlspecialchars($search); ?>">
            </div>
            <div class="col-md-3">
                <label class="form-label">&nbsp;</label>
                <div>
                    <button type="submit" class="btn btn-outline-primary btn-sm">
                        <i class="fas fa-search"></i> Filter
                    </button>
                    <a href="view_applications.php<?php echo $request_id ? '?request_id=' . $request_id : ''; ?>" class="btn btn-outline-secondary btn-sm">
                        <i class="fas fa-times"></i> Clear
                    </a>
                </div>
            </div>
        </form>
    </div>
</div>

<!-- Statistics Cards -->
<div class="row mb-4">
    <?php
    $stats = [
        'total' => count($applications),
        'submitted' => count(array_filter($applications, fn($app) => $app['status'] === 'submitted')),
        'approved' => count(array_filter($applications, fn($app) => in_array($app['status'], ['approved_by_pm', 'approved_by_hr', 'approved_by_accountant']))),
        'hired' => count(array_filter($applications, fn($app) => $app['status'] === 'hired')),
        'complete_docs' => count(array_filter($applications, fn($app) => $app['document_status'] === 'Complete'))
    ];
    ?>
    
    <div class="col-md-2">
        <div class="card bg-primary text-white">
            <div class="card-body text-center">
                <h3 class="mb-1"><?php echo $total_records; ?></h3>
                <small>Total Applications</small>
            </div>
        </div>
    </div>
    <div class="col-md-2">
        <div class="card bg-info text-white">
            <div class="card-body text-center">
                <h3 class="mb-1"><?php echo $stats['submitted']; ?></h3>
                <small>Submitted</small>
            </div>
        </div>
    </div>
    <div class="col-md-2">
        <div class="card bg-warning text-white">
            <div class="card-body text-center">
                <h3 class="mb-1"><?php echo $stats['approved']; ?></h3>
                <small>In Process</small>
            </div>
        </div>
    </div>
    <div class="col-md-2">
        <div class="card bg-success text-white">
            <div class="card-body text-center">
                <h3 class="mb-1"><?php echo $stats['hired']; ?></h3>
                <small>Hired</small>
            </div>
        </div>
    </div>
    <div class="col-md-2">
        <div class="card bg-secondary text-white">
            <div class="card-body text-center">
                <h3 class="mb-1"><?php echo $stats['complete_docs']; ?></h3>
                <small>Complete Docs</small>
            </div>
        </div>
    </div>
</div>

<!-- Request-specific position summary -->
<?php if ($request_details): ?>
<div class="card mb-3">
    <div class="card-header">
        <i class="fas fa-chart-pie me-2"></i>Position Summary for <?php echo htmlspecialchars($request_details['request_number']); ?>
    </div>
    <div class="card-body">
        <?php
        // Get position breakdown for this request
        try {
            $position_sql = "SELECT ec.id, ec.position_title, ec.category, ec.quantity,
                                    COUNT(ja.id) as total_applications,
                                    COUNT(CASE WHEN ja.status = 'submitted' THEN 1 END) as submitted_apps,
                                    COUNT(CASE WHEN ja.status IN ('approved_by_pm', 'approved_by_hr', 'approved_by_accountant') THEN 1 END) as pending_apps,
                                    COUNT(CASE WHEN ja.status = 'hired' THEN 1 END) as hired_apps,
                                    COUNT(CASE WHEN ja.status = 'rejected' THEN 1 END) as rejected_apps,
                                    (ec.quantity - COUNT(CASE WHEN ja.status = 'hired' THEN 1 END)) as remaining_positions
                             FROM employee_categories ec
                             LEFT JOIN job_applications ja ON ec.id = ja.category_id
                             WHERE ec.request_id = ?
                             GROUP BY ec.id
                             ORDER BY ec.position_title";
            $positions = fetchAll($position_sql, [$request_id]);
        } catch (Exception $e) {
            $positions = [];
            error_log("Error fetching position summary: " . $e->getMessage());
        }
        ?>
        
        <?php if (!empty($positions)): ?>
        <div class="row">
            <?php foreach ($positions as $position): ?>
            <div class="col-md-6 mb-3">
                <div class="border rounded p-3">
                    <h6 class="text-primary mb-2"><?php echo htmlspecialchars($position['position_title']); ?></h6>
                    <div class="row text-center">
                        <div class="col-3">
                            <strong><?php echo $position['quantity']; ?></strong>
                            <br><small class="text-muted">Needed</small>
                        </div>
                        <div class="col-3">
                            <strong><?php echo $position['total_applications']; ?></strong>
                            <br><small class="text-muted">Applied</small>
                        </div>
                        <div class="col-3">
                            <strong class="text-success"><?php echo $position['hired_apps']; ?></strong>
                            <br><small class="text-muted">Hired</small>
                        </div>
                        <div class="col-3">
                            <strong class="text-warning"><?php echo $position['remaining_positions']; ?></strong>
                            <br><small class="text-muted">Remaining</small>
                        </div>
                    </div>
                    <?php if ($position['total_applications'] > 0): ?>
                    <div class="mt-2">
                        <?php if ($position['submitted_apps'] > 0): ?>
                        <span class="badge bg-info"><?php echo $position['submitted_apps']; ?> New</span>
                        <?php endif; ?>
                        <?php if ($position['pending_apps'] > 0): ?>
                        <span class="badge bg-warning"><?php echo $position['pending_apps']; ?> Pending</span>
                        <?php endif; ?>
                        <?php if ($position['rejected_apps'] > 0): ?>
                        <span class="badge bg-danger"><?php echo $position['rejected_apps']; ?> Rejected</span>
                        <?php endif; ?>
                    </div>
                    <?php endif; ?>
                </div>
            </div>
            <?php endforeach; ?>
        </div>
        <?php else: ?>
        <p class="text-muted">No positions defined for this request.</p>
        <?php endif; ?>
    </div>
</div>
<?php endif; ?>

<?php if (empty($applications)): ?>
<div class="card">
    <div class="card-body text-center py-5">
        <i class="fas fa-file-alt fa-3x text-muted mb-3"></i>
        <?php if ($request_details): ?>
        <h5>No Applications Found for This Request</h5>
        <p class="text-muted">
            No job applications have been submitted for request <?php echo htmlspecialchars($request_details['request_number']); ?> yet.
        </p>
        <?php if ($currentUser['role'] === ROLE_SITE_ADMIN && in_array($request_details['status'], ['approved_by_pm', 'approved_by_gm', 'approved_final'])): ?>
        <a href="application_form.php?request_id=<?php echo $request_id; ?>" class="btn btn-primary">
            <i class="fas fa-plus"></i> Submit First Application for This Request
        </a>
        <?php endif; ?>
        <?php elseif ($currentUser['role'] === ROLE_SITE_ADMIN): ?>
        <h5>No Applications Found</h5>
        <p class="text-muted">You haven't submitted any job applications yet.</p>
        <a href="application_form.php" class="btn btn-primary">
            <i class="fas fa-plus"></i> Submit Your First Application
        </a>
        <?php else: ?>
        <h5>No Applications Available</h5>
        <p class="text-muted">
            <?php 
            switch ($currentUser['role']) {
                case ROLE_PROJECT_MANAGER:
                    echo 'No applications found for your project.';
                    break;
                case ROLE_GENERAL_MANAGER:
                case ROLE_HR_MANAGER:
                case ROLE_ACCOUNTANT:
                    echo 'No applications available for review at this time.';
                    break;
                default:
                    echo 'No applications found.';
            }
            ?>
        </p>
        <?php endif; ?>
    </div>
</div>
<?php else: ?>

<div class="card">
    <div class="card-header">
        <div class="d-flex justify-content-between align-items-center">
            <span>
                Applications (<?php echo $total_records; ?>)
                <?php if ($request_details): ?>
                - <?php echo htmlspecialchars($request_details['request_number']); ?>
                <?php endif; ?>
            </span>
            <div class="btn-group btn-group-sm">
                <button type="button" class="btn btn-outline-secondary" onclick="printApplications()">
                    <i class="fas fa-print"></i> Print
                </button>
                <button type="button" class="btn btn-outline-secondary" onclick="exportApplications()">
                    <i class="fas fa-download"></i> Export
                </button>
            </div>
        </div>
    </div>
    <div class="card-body p-0">
        <div class="table-responsive">
            <table class="table table-hover mb-0" id="applicationsTable">
                <thead class="bg-light">
                    <tr>
                        <th>Application Details</th>
                        <th>Position</th>
                        <?php if (!$request_details): ?>
                        <th>Project</th>
                        <?php endif; ?>
                        <th>Status</th>
                        <th>Documents</th>
                        <th>Submitted</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($applications as $app): ?>
                    <tr>
                        <td>
                            <div>
                                <strong><?php echo htmlspecialchars($app['application_number']); ?></strong><br>
                                <small class="text-muted">
                                    <?php echo htmlspecialchars($app['full_name_english']); ?>
                                    <br>NIC: <?php echo htmlspecialchars($app['nic_number']); ?>
                                    <?php if ($currentUser['role'] !== ROLE_SITE_ADMIN): ?>
                                    <br>Submitted by: <?php echo htmlspecialchars($app['submitted_by_name']); ?>
                                    <?php endif; ?>
                                </small>
                            </div>
                        </td>
                        <td>
                            <div>
                                <?php echo htmlspecialchars($app['position_title']); ?><br>
                                <small class="text-muted">
                                    <?php echo htmlspecialchars($app['category']); ?>
                                    <?php if (!$request_details): ?>
                                    <br>Req: <?php echo htmlspecialchars($app['request_number']); ?>
                                    <?php endif; ?>
                                </small>
                            </div>
                        </td>
                        <?php if (!$request_details): ?>
                        <td>
                            <small><?php echo htmlspecialchars($app['site_project']); ?></small>
                        </td>
                        <?php endif; ?>
                        <td>
                            <?php echo getStatusBadge($app['status'], 'application'); ?>
                        </td>
                        <td>
                            <div class="d-flex align-items-center">
                                <span class="badge <?php echo $app['document_status'] === 'Complete' ? 'bg-success' : 'bg-warning'; ?> me-2">
                                    <?php echo $app['document_count']; ?> files
                                </span>
                                <small class="text-<?php echo $app['document_status'] === 'Complete' ? 'success' : 'warning'; ?>">
                                    <?php echo $app['document_status']; ?>
                                </small>
                            </div>
                        </td>
                        <td>
                            <div>
                                <?php echo formatDisplayDate($app['submitted_at']); ?><br>
                                <small class="text-muted">
                                    <?php echo getRelativeTime($app['submitted_at']); ?>
                                </small>
                            </div>
                        </td>
                        <td>
                            <div class="btn-group btn-group-sm">
                                <button type="button" class="btn btn-outline-primary dropdown-toggle" 
                                        data-bs-toggle="dropdown" aria-expanded="false">
                                    <i class="fas fa-cog"></i>
                                </button>
                                <ul class="dropdown-menu">
                                    <li>
                                        <a class="dropdown-item" href="view_application.php?id=<?php echo $app['id']; ?>">
                                            <i class="fas fa-eye me-2"></i>View Details
                                        </a>
                                    </li>
                                    <?php if ($currentUser['role'] === ROLE_SITE_ADMIN): ?>
                                    <li>
                                        <a class="dropdown-item" href="../documents/upload.php?app_id=<?php echo $app['id']; ?>">
                                            <i class="fas fa-upload me-2"></i>Upload Documents
                                        </a>
                                    </li>
                                    <li>
                                        <a class="dropdown-item" href="../documents/upload.php?app_id=<?php echo $app['id']; ?>">
                                            <i class="fas fa-folder me-2"></i>Manage Documents
                                        </a>
                                    </li>
                                    <?php else: ?>
                                    <li>
                                        <a class="dropdown-item" href="../documents/view.php?app_id=<?php echo $app['id']; ?>">
                                            <i class="fas fa-folder me-2"></i>View Documents
                                        </a>
                                    </li>
                                    <?php endif; ?>
                                    <li><hr class="dropdown-divider"></li>
                                    <?php if ($app['status'] === 'submitted' && $currentUser['role'] === ROLE_SITE_ADMIN): ?>
                                    <li>
                                        <a class="dropdown-item" href="edit_application.php?id=<?php echo $app['id']; ?>">
                                            <i class="fas fa-edit me-2"></i>Edit Application
                                        </a>
                                    </li>
                                    <?php endif; ?>
                                    <li>
                                        <a class="dropdown-item" href="print_application.php?id=<?php echo $app['id']; ?>" target="_blank">
                                            <i class="fas fa-print me-2"></i>Print Application
                                        </a>
                                    </li>
                                    <?php 
                                    // Show approval option based on role and status
                                    $can_approve = false;
                                    $approval_url = '';
                                    
                                    if ($currentUser['role'] === ROLE_PROJECT_MANAGER && $app['status'] === 'submitted') {
                                        $can_approve = true;
                                        $approval_url = '../employees/approve_application.php?id=' . $app['id'];
                                    } elseif ($currentUser['role'] === ROLE_HR_MANAGER && $app['status'] === 'approved_by_pm') {
                                        $can_approve = true;
                                        $approval_url = '../employees/approve_application.php?id=' . $app['id'];
                                    } elseif ($currentUser['role'] === ROLE_ACCOUNTANT && $app['status'] === 'approved_by_hr') {
                                        $can_approve = true;
                                        $approval_url = '../employees/approve_application.php?id=' . $app['id'];
                                    }
                                    
                                    if ($can_approve): ?>
                                    <li><hr class="dropdown-divider"></li>
                                    <li>
                                        <a class="dropdown-item text-success" href="<?php echo $approval_url; ?>">
                                            <i class="fas fa-check me-2"></i>Review & Approve
                                        </a>
                                    </li>
                                    <?php endif; ?>
                                    <?php if (in_array($app['status'], ['submitted', 'under_review']) && $currentUser['role'] === ROLE_SITE_ADMIN): ?>
                                    <li><hr class="dropdown-divider"></li>
                                    <li>
                                        <a class="dropdown-item text-danger" href="#" 
                                           onclick="withdrawApplication(<?php echo $app['id']; ?>)">
                                            <i class="fas fa-times me-2"></i>Withdraw Application
                                        </a>
                                    </li>
                                    <?php endif; ?>
                                </ul>
                            </div>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    </div>
</div>

<!-- Pagination -->
<?php if ($total_pages > 1): ?>
<nav class="mt-3">
    <ul class="pagination justify-content-center">
        <?php if ($page > 1): ?>
        <li class="page-item">
            <a class="page-link" href="?page=<?php echo $page-1; ?>&request_id=<?php echo $request_id; ?>&status=<?php echo $status_filter; ?>&search=<?php echo $search; ?>">
                <i class="fas fa-chevron-left"></i> Previous
            </a>
        </li>
        <?php endif; ?>
        
        <?php for ($i = max(1, $page-2); $i <= min($total_pages, $page+2); $i++): ?>
        <li class="page-item <?php echo $i === $page ? 'active' : ''; ?>">
            <a class="page-link" href="?page=<?php echo $i; ?>&request_id=<?php echo $request_id; ?>&status=<?php echo $status_filter; ?>&search=<?php echo $search; ?>">
                <?php echo $i; ?>
            </a>
        </li>
        <?php endfor; ?>
        
        <?php if ($page < $total_pages): ?>
        <li class="page-item">
            <a class="page-link" href="?page=<?php echo $page+1; ?>&request_id=<?php echo $request_id; ?>&status=<?php echo $status_filter; ?>&search=<?php echo $search; ?>">
                Next <i class="fas fa-chevron-right"></i>
            </a>
        </li>
        <?php endif; ?>
    </ul>
</nav>
<?php endif; ?>

<!-- Document Status Legend -->
<div class="row mt-3">
    <div class="col-12">
        <div class="card">
            <div class="card-header">Document Requirements</div>
            <div class="card-body">
                <div class="row">
                    <div class="col-md-6">
                        <h6>Required Documents:</h6>
                        <ul class="list-unstyled">
                            <li><i class="fas fa-check-circle text-success me-2"></i>NIC Copy</li>
                            <li><i class="fas fa-check-circle text-success me-2"></i>CV/Resume</li>
                            <li><i class="fas fa-check-circle text-success me-2"></i>Educational Certificates</li>
                            <li><i class="fas fa-check-circle text-success me-2"></i>Passport Photo</li>
                        </ul>
                    </div>
                    <div class="col-md-6">
                        <h6>Optional Documents:</h6>
                        <ul class="list-unstyled">
                            <li><i class="fas fa-circle text-muted me-2"></i>Birth Certificate</li>
                            <li><i class="fas fa-circle text-muted me-2"></i>Bank Details</li>
                            <li><i class="fas fa-circle text-muted me-2"></i>Medical Reports</li>
                            <li><i class="fas fa-circle text-muted me-2"></i>Other Certificates</li>
                        </ul>
                    </div>
                </div>
                <div class="alert alert-info mt-3 mb-0">
                    <i class="fas fa-info-circle me-2"></i>
                    <strong>Note:</strong> Applications with complete documents are processed faster. 
                    You can upload documents anytime using the "Upload Documents" button.
                </div>
            </div>
        </div>
    </div>
</div>

<?php endif; ?>

<!-- Request Details Modal -->
<?php if ($request_details): ?>
<div class="modal fade" id="requestDetailsModal" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Request Details - <?php echo htmlspecialchars($request_details['request_number']); ?></h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <div class="row">
                    <div class="col-md-6">
                        <table class="table table-borderless table-sm">
                            <tr>
                                <td><strong>Request Number:</strong></td>
                                <td><?php echo htmlspecialchars($request_details['request_number']); ?></td>
                            </tr>
                            <tr>
                                <td><strong>Project:</strong></td>
                                <td><?php echo htmlspecialchars($request_details['site_project']); ?></td>
                            </tr>
                            <tr>
                                <td><strong>Request Date:</strong></td>
                                <td><?php echo formatDisplayDate($request_details['request_date']); ?></td>
                            </tr>
                            <tr>
                                <td><strong>Status:</strong></td>
                                <td><?php echo getStatusBadge($request_details['status']); ?></td>
                            </tr>
                        </table>
                    </div>
                    <div class="col-md-6">
                        <table class="table table-borderless table-sm">
                            <tr>
                                <td><strong>Requested By:</strong></td>
                                <td><?php echo htmlspecialchars($request_details['requested_by_name']); ?></td>
                            </tr>
                            <tr>
                                <td><strong>Requirement Reason:</strong></td>
                                <td><?php echo REQUIREMENT_REASONS[$request_details['requirement_reason']] ?? $request_details['requirement_reason']; ?></td>
                            </tr>
                            <tr>
                                <td><strong>Total Employees:</strong></td>
                                <td><?php echo $request_details['total_employees_requested']; ?></td>
                            </tr>
                            <tr>
                                <td><strong>Expected Start:</strong></td>
                                <td><?php echo formatDisplayDate($request_details['expected_start_date']); ?></td>
                            </tr>
                        </table>
                    </div>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                <a href="../requests/view_requests.php" class="btn btn-primary">View All Requests</a>
            </div>
        </div>
    </div>
</div>
<?php endif; ?>

<script>
function withdrawApplication(applicationId) {
    if (confirm('Are you sure you want to withdraw this application? This action cannot be undone.')) {
        const form = document.createElement('form');
        form.method = 'POST';
        form.action = 'withdraw_application.php';
        
        const csrfInput = document.createElement('input');
        csrfInput.type = 'hidden';
        csrfInput.name = 'csrf_token';
        csrfInput.value = '<?php echo generateCSRFToken(); ?>';
        
        const idInput = document.createElement('input');
        idInput.type = 'hidden';
        idInput.name = 'application_id';
        idInput.value = applicationId;
        
        form.appendChild(csrfInput);
        form.appendChild(idInput);
        
        document.body.appendChild(form);
        form.submit();
    }
}

function printApplications() {
    const url = 'print_applications.php' + <?php echo $request_id ? "'?request_id=" . $request_id . "'" : "''"; ?>;
    window.open(url, '_blank');
}

function exportApplications() {
    const url = 'export_applications.php?format=excel' + <?php echo $request_id ? "'&request_id=" . $request_id . "'" : "''"; ?>;
    window.location.href = url;
}

function viewRequestDetails() {
    const modal = new bootstrap.Modal(document.getElementById('requestDetailsModal'));
    modal.show();
}

// Auto-submit form when status filter changes
document.addEventListener('DOMContentLoaded', function() {
    const statusFilter = document.querySelector('select[name="status"]');
    if (statusFilter) {
        statusFilter.addEventListener('change', function() {
            this.form.submit();
        });
    }
    
    // Search on enter key
    const searchInput = document.querySelector('input[name="search"]');
    if (searchInput) {
        searchInput.addEventListener('keypress', function(e) {
            if (e.key === 'Enter') {
                this.form.submit();
            }
        });
    }
});

// Initialize DataTable if available
$(document).ready(function() {
    if (typeof $.fn.DataTable !== 'undefined') {
        $('#applicationsTable').DataTable({
            order: [[5, 'desc']], // Sort by submitted date
            pageLength: 10,
            responsive: true,
            language: {
                search: "Search applications:",
                lengthMenu: "Show _MENU_ applications per page",
                info: "Showing _START_ to _END_ of _TOTAL_ applications",
                infoEmpty: "No applications found",
                infoFiltered: "(filtered from _MAX_ total applications)"
            }
        });
    }
});

// Real-time updates for application status
function checkForUpdates() {
    <?php if ($request_id > 0): ?>
    fetch('check_application_updates.php?request_id=<?php echo $request_id; ?>')
    <?php else: ?>
    fetch('check_application_updates.php')
    <?php endif; ?>
        .then(response => response.json())
        .then(data => {
            if (data.hasUpdates) {
                // Show notification banner
                const banner = document.createElement('div');
                banner.className = 'alert alert-info alert-dismissible fade show position-fixed';
                banner.style.cssText = 'top: 20px; right: 20px; z-index: 9999; max-width: 400px;';
                banner.innerHTML = `
                    <i class="fas fa-info-circle me-2"></i>
                    New application updates available.
                    <a href="#" onclick="location.reload()" class="alert-link">Refresh to see changes</a>
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                `;
                document.body.appendChild(banner);
                
                // Auto-remove after 10 seconds
                setTimeout(() => {
                    if (banner.parentNode) {
                        banner.remove();
                    }
                }, 10000);
            }
        })
        .catch(error => console.log('Update check failed:', error));
}

// Check for updates every 30 seconds
setInterval(checkForUpdates, 30000);

// Bulk actions functionality
function initializeBulkActions() {
    const selectAllCheckbox = document.getElementById('selectAll');
    const rowCheckboxes = document.querySelectorAll('.row-checkbox');
    const bulkActionsDiv = document.getElementById('bulkActions');
    
    if (selectAllCheckbox && rowCheckboxes.length > 0) {
        selectAllCheckbox.addEventListener('change', function() {
            rowCheckboxes.forEach(checkbox => {
                checkbox.checked = this.checked;
            });
            toggleBulkActions();
        });
        
        rowCheckboxes.forEach(checkbox => {
            checkbox.addEventListener('change', function() {
                const checkedCount = document.querySelectorAll('.row-checkbox:checked').length;
                selectAllCheckbox.checked = checkedCount === rowCheckboxes.length;
                selectAllCheckbox.indeterminate = checkedCount > 0 && checkedCount < rowCheckboxes.length;
                toggleBulkActions();
            });
        });
    }
}

function toggleBulkActions() {
    const checkedBoxes = document.querySelectorAll('.row-checkbox:checked');
    const bulkActionsDiv = document.getElementById('bulkActions');
    
    if (bulkActionsDiv) {
        if (checkedBoxes.length > 0) {
            bulkActionsDiv.style.display = 'block';
            bulkActionsDiv.querySelector('.selected-count').textContent = checkedBoxes.length;
        } else {
            bulkActionsDiv.style.display = 'none';
        }
    }
}

function performBulkAction(action) {
    const checkedBoxes = document.querySelectorAll('.row-checkbox:checked');
    const applicationIds = Array.from(checkedBoxes).map(cb => cb.value);
    
    if (applicationIds.length === 0) {
        alert('Please select at least one application.');
        return;
    }
    
    let confirmMessage = '';
    switch (action) {
        case 'export':
            exportSelectedApplications(applicationIds);
            return;
        case 'print':
            printSelectedApplications(applicationIds);
            return;
        case 'approve':
            confirmMessage = `Are you sure you want to approve ${applicationIds.length} application(s)?`;
            break;
        case 'reject':
            confirmMessage = `Are you sure you want to reject ${applicationIds.length} application(s)?`;
            break;
        default:
            alert('Invalid action selected.');
            return;
    }
    
    if (confirm(confirmMessage)) {
        // Submit bulk action
        const form = document.createElement('form');
        form.method = 'POST';
        form.action = 'bulk_actions.php';
        
        const csrfInput = document.createElement('input');
        csrfInput.type = 'hidden';
        csrfInput.name = 'csrf_token';
        csrfInput.value = '<?php echo generateCSRFToken(); ?>';
        form.appendChild(csrfInput);
        
        const actionInput = document.createElement('input');
        actionInput.type = 'hidden';
        actionInput.name = 'action';
        actionInput.value = action;
        form.appendChild(actionInput);
        
        applicationIds.forEach(id => {
            const idInput = document.createElement('input');
            idInput.type = 'hidden';
            idInput.name = 'application_ids[]';
            idInput.value = id;
            form.appendChild(idInput);
        });
        
        document.body.appendChild(form);
        form.submit();
    }
}

function exportSelectedApplications(applicationIds) {
    const url = 'export_applications.php?format=excel&ids=' + applicationIds.join(',');
    window.location.href = url;
}

function printSelectedApplications(applicationIds) {
    const url = 'print_applications.php?ids=' + applicationIds.join(',');
    window.open(url, '_blank');
}

// Initialize bulk actions when page loads
document.addEventListener('DOMContentLoaded', function() {
    initializeBulkActions();
});
</script>

<style>
/* Enhanced styling for better UX */
.table th {
    background-color: #f8f9fa;
    border-bottom: 2px solid #dee2e6;
    font-weight: 600;
    position: sticky;
    top: 0;
    z-index: 10;
}

.badge {
    font-size: 0.75em;
    margin: 1px;
}

.btn-group-sm .dropdown-menu {
    font-size: 0.875rem;
    min-width: 180px;
}

.dropdown-item {
    padding: 0.5rem 1rem;
}

.dropdown-item i {
    width: 16px;
    text-align: center;
}

/* Position summary cards */
.position-summary .card {
    transition: transform 0.2s;
}

.position-summary .card:hover {
    transform: translateY(-2px);
    box-shadow: 0 4px 8px rgba(0,0,0,0.1);
}

/* Application status colors */
.status-submitted { border-left: 4px solid #17a2b8; }
.status-approved { border-left: 4px solid #28a745; }
.status-rejected { border-left: 4px solid #dc3545; }
.status-hired { border-left: 4px solid #155724; }

/* Responsive improvements */
@media (max-width: 768px) {
    .table-responsive {
        font-size: 0.875rem;
    }
    
    .btn-group-sm .btn {
        padding: 0.25rem 0.5rem;
    }
    
    .card-body {
        padding: 1rem 0.75rem;
    }
}

/* Filter improvements */
.form-control:focus {
    border-color: #80bdff;
    box-shadow: 0 0 0 0.2rem rgba(0, 123, 255, 0.25);
}

/* Animation for statistics cards */
.card.bg-primary, .card.bg-info, .card.bg-warning, .card.bg-success, .card.bg-secondary {
    transition: transform 0.2s ease-in-out;
}

.card.bg-primary:hover, .card.bg-info:hover, .card.bg-warning:hover, 
.card.bg-success:hover, .card.bg-secondary:hover {
    transform: scale(1.05);
}

/* Document status indicators */
.document-complete {
    background: linear-gradient(45deg, #28a745, #20c997);
}

.document-incomplete {
    background: linear-gradient(45deg, #ffc107, #fd7e14);
}

/* Loading states */
.loading {
    opacity: 0.6;
    pointer-events: none;
    position: relative;
}

.loading::after {
    content: '';
    position: absolute;
    top: 50%;
    left: 50%;
    width: 20px;
    height: 20px;
    margin: -10px 0 0 -10px;
    border: 2px solid #f3f3f3;
    border-top: 2px solid #3498db;
    border-radius: 50%;
    animation: spin 1s linear infinite;
}

@keyframes spin {
    0% { transform: rotate(0deg); }
    100% { transform: rotate(360deg); }
}

/* Bulk actions styling */
#bulkActions {
    display: none;
    position: sticky;
    top: 0;
    z-index: 1000;
    background: #fff;
    border-bottom: 1px solid #dee2e6;
    padding: 1rem;
    margin: -1rem -1rem 1rem -1rem;
}

.bulk-actions-bar {
    display: flex;
    align-items: center;
    gap: 1rem;
}

.selected-info {
    flex-grow: 1;
    font-weight: 500;
    color: #495057;
}

/* Print styles */
@media print {
    .no-print {
        display: none !important;
    }
    
    .card {
        border: 1px solid #000 !important;
        page-break-inside: avoid;
    }
    
    .table {
        font-size: 12px;
    }
    
    .badge {
        border: 1px solid #000;
        color: #000 !important;
        background: transparent !important;
    }
}
</style>

<?php 
// Include DataTables
$include_datatables = true;
require_once '../includes/footer.php'; 
?>