<?php
/**
 * Withdraw Application
 * Employee Recruitment Management System
 * RR Construction (Pvt) Ltd
 */

// Define access constant
define('HRM_ACCESS', true);

// Start session
session_start();

// Include required files
require_once '../config/constants.php';
require_once '../config/database.php';
require_once '../includes/functions.php';

// Check authentication
if (!isLoggedIn()) {
    header('Location: ' . BASE_URL . 'auth/login.php');
    exit;
}

// Require site admin permission
requirePermission(ROLE_SITE_ADMIN);

$currentUser = getCurrentUser();
$errors = [];
$success = false;

// Process withdrawal request
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!verifyCSRFToken($_POST['csrf_token'] ?? '')) {
        setFlashMessage('Invalid security token.', 'danger');
    } else {
        $application_id = (int)($_POST['application_id'] ?? 0);
        $withdrawal_reason = cleanInput($_POST['withdrawal_reason'] ?? '');
        
        if (!$application_id) {
            setFlashMessage('Invalid application ID.', 'danger');
        } else {
            try {
                // Get application details with access control
                $sql = "SELECT ja.*, er.request_number, er.site_project, ec.position_title,
                               u.full_name as submitted_by_name
                        FROM job_applications ja
                        JOIN employee_requests er ON ja.request_id = er.id
                        JOIN employee_categories ec ON ja.category_id = ec.id
                        JOIN users u ON ja.submitted_by = u.id
                        WHERE ja.id = ? AND ja.submitted_by = ?";
                
                $application = fetchOne($sql, [$application_id, $currentUser['id']]);
                
                if (!$application) {
                    setFlashMessage('Application not found or access denied.', 'danger');
                } else {
                    // Check if application can be withdrawn
                    $withdrawable_statuses = ['submitted', 'under_review'];
                    
                    if (!in_array($application['status'], $withdrawable_statuses)) {
                        setFlashMessage('Application cannot be withdrawn. Only submitted or under review applications can be withdrawn.', 'warning');
                    } else {
                        beginTransaction();
                        
                        // Store old status for logging
                        $old_status = $application['status'];
                        
                        // Update application status to rejected with withdrawal reason
                        $update_data = [
                            'status' => 'rejected',
                            'rejection_reason' => 'Withdrawn by applicant' . ($withdrawal_reason ? ': ' . $withdrawal_reason : ''),
                            'updated_at' => date('Y-m-d H:i:s')
                        ];
                        
                        updateRecord('job_applications', $update_data, 'id = ?', [$application_id]);
                        
                        // Log the withdrawal activity
                        logActivity($currentUser['id'], 'withdraw', 'job_applications', $application_id, 
                                   ['status' => $old_status], 
                                   ['status' => 'rejected', 'reason' => 'Application withdrawn by user']);
                        
                        // Optional: Send notification to relevant parties
                        try {
                            // Notify HR/Managers about withdrawal
                            $notification_message = "
                            <h3>Application Withdrawal Notice</h3>
                            <p><strong>Application:</strong> {$application['application_number']}</p>
                            <p><strong>Applicant:</strong> {$application['full_name_english']}</p>
                            <p><strong>Position:</strong> {$application['position_title']}</p>
                            <p><strong>Project:</strong> {$application['site_project']}</p>
                            <p><strong>Withdrawal Date:</strong> " . date('d/m/Y H:i') . "</p>
                            " . ($withdrawal_reason ? "<p><strong>Reason:</strong> {$withdrawal_reason}</p>" : "") . "
                            <p>This application has been withdrawn by the applicant and is no longer active.</p>
                            ";
                            
                            // You can add email notification here if needed
                            // sendNotification($hr_email, 'Application Withdrawn', $notification_message);
                            
                        } catch (Exception $e) {
                            error_log("Notification error during withdrawal: " . $e->getMessage());
                            // Don't fail the withdrawal if notification fails
                        }
                        
                        commit();
                        setFlashMessage(
                            "Application {$application['application_number']} has been successfully withdrawn.", 
                            'success'
                        );
                        $success = true;
                    }
                }
                
            } catch (Exception $e) {
                rollback();
                error_log("Error withdrawing application: " . $e->getMessage());
                setFlashMessage('Error withdrawing application. Please try again.', 'danger');
            }
        }
    }
} else {
    // GET request - show withdrawal form
    $application_id = (int)($_GET['id'] ?? 0);
    
    if ($application_id) {
        try {
            // Get application details for confirmation
            $sql = "SELECT ja.*, er.request_number, er.site_project, ec.position_title,
                           u.full_name as submitted_by_name
                    FROM job_applications ja
                    JOIN employee_requests er ON ja.request_id = er.id
                    JOIN employee_categories ec ON ja.category_id = ec.id
                    JOIN users u ON ja.submitted_by = u.id
                    WHERE ja.id = ? AND ja.submitted_by = ?";
            
            $application = fetchOne($sql, [$application_id, $currentUser['id']]);
            
            if (!$application) {
                setFlashMessage('Application not found or access denied.', 'danger');
                header('Location: view_applications.php');
                exit;
            }
            
            // Check if application can be withdrawn
            $withdrawable_statuses = ['submitted', 'under_review'];
            
            if (!in_array($application['status'], $withdrawable_statuses)) {
                setFlashMessage('Application cannot be withdrawn. Only submitted or under review applications can be withdrawn.', 'warning');
                header('Location: view_application.php?id=' . $application_id);
                exit;
            }
            
        } catch (Exception $e) {
            error_log("Error fetching application for withdrawal: " . $e->getMessage());
            setFlashMessage('Error loading application details.', 'danger');
            header('Location: view_applications.php');
            exit;
        }
    }
}

// Redirect after POST processing
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if ($success) {
        header('Location: view_applications.php');
    } else {
        header('Location: view_application.php?id=' . $application_id);
    }
    exit;
}

// If we reach here, it's a GET request to show the withdrawal form
if (!isset($application) || !$application) {
    setFlashMessage('Invalid application for withdrawal.', 'danger');
    header('Location: view_applications.php');
    exit;
}

// Set page variables for header
$page_title = 'Withdraw Application';
$active_menu = 'applications';

require_once '../includes/header.php';
?>

<div class="row mb-4">
    <div class="col-12">
        <div class="d-flex justify-content-between align-items-center">
            <div>
                <h4 class="mb-0 text-danger">
                    <i class="fas fa-exclamation-triangle me-2"></i>Withdraw Application
                </h4>
                <small class="text-muted">This action cannot be undone</small>
            </div>
            <div class="btn-group">
                <a href="view_application.php?id=<?php echo $application_id; ?>" class="btn btn-outline-secondary">
                    <i class="fas fa-arrow-left"></i> Back to Application
                </a>
                <a href="view_applications.php" class="btn btn-outline-primary">
                    <i class="fas fa-list"></i> All Applications
                </a>
            </div>
        </div>
    </div>
</div>

<!-- Application Details -->
<div class="card mb-4">
    <div class="card-header bg-danger text-white">
        <i class="fas fa-file-alt me-2"></i>Application to be Withdrawn
    </div>
    <div class="card-body">
        <div class="row">
            <div class="col-md-6">
                <table class="table table-sm table-borderless">
                    <tr>
                        <td width="150"><strong>Application Number:</strong></td>
                        <td><?php echo htmlspecialchars($application['application_number']); ?></td>
                    </tr>
                    <tr>
                        <td><strong>Applicant Name:</strong></td>
                        <td><?php echo htmlspecialchars($application['full_name_english']); ?></td>
                    </tr>
                    <tr>
                        <td><strong>NIC Number:</strong></td>
                        <td><?php echo htmlspecialchars($application['nic_number']); ?></td>
                    </tr>
                    <tr>
                        <td><strong>Position Applied:</strong></td>
                        <td><?php echo htmlspecialchars($application['position_title']); ?></td>
                    </tr>
                </table>
            </div>
            <div class="col-md-6">
                <table class="table table-sm table-borderless">
                    <tr>
                        <td width="150"><strong>Request Number:</strong></td>
                        <td><?php echo htmlspecialchars($application['request_number']); ?></td>
                    </tr>
                    <tr>
                        <td><strong>Project:</strong></td>
                        <td><?php echo htmlspecialchars($application['site_project']); ?></td>
                    </tr>
                    <tr>
                        <td><strong>Current Status:</strong></td>
                        <td><?php echo getStatusBadge($application['status'], 'application'); ?></td>
                    </tr>
                    <tr>
                        <td><strong>Submitted Date:</strong></td>
                        <td><?php echo formatDisplayDateTime($application['submitted_at']); ?></td>
                    </tr>
                </table>
            </div>
        </div>
    </div>
</div>

<!-- Warning Notice -->
<div class="card mb-4 border-warning">
    <div class="card-header bg-warning text-dark">
        <i class="fas fa-exclamation-triangle me-2"></i>Important Notice
    </div>
    <div class="card-body">
        <h6 class="text-warning">Please read carefully before proceeding:</h6>
        <ul class="mb-3">
            <li><strong>Permanent Action:</strong> Once withdrawn, this application cannot be reactivated or restored.</li>
            <li><strong>Status Change:</strong> The application status will be changed to "Rejected" with a withdrawal note.</li>
            <li><strong>Position Availability:</strong> If you change your mind, you'll need to submit a completely new application (if positions are still available).</li>
            <li><strong>Notification:</strong> Relevant managers and HR will be notified of this withdrawal.</li>
            <li><strong>Documents:</strong> Any uploaded documents will remain in the system for record-keeping purposes.</li>
        </ul>
        
        <div class="alert alert-info mb-0">
            <i class="fas fa-info-circle me-2"></i>
            <strong>Alternative Options:</strong> If you need to make changes to your application, consider using the 
            <a href="edit_application.php?id=<?php echo $application_id; ?>" class="alert-link">Edit Application</a> 
            feature instead of withdrawing.
        </div>
    </div>
</div>

<!-- Withdrawal Form -->
<div class="card">
    <div class="card-header">
        <i class="fas fa-form me-2"></i>Withdrawal Confirmation
    </div>
    <div class="card-body">
        <form method="POST" id="withdrawalForm">
            <input type="hidden" name="csrf_token" value="<?php echo generateCSRFToken(); ?>">
            <input type="hidden" name="application_id" value="<?php echo $application_id; ?>">
            
            <div class="mb-4">
                <label class="form-label">Reason for Withdrawal (Optional)</label>
                <textarea name="withdrawal_reason" class="form-control" rows="3" 
                          placeholder="Please provide a brief reason for withdrawing this application (e.g., Found another job, Changed mind about position, Personal reasons)"></textarea>
                <small class="text-muted">This information helps us improve our recruitment process.</small>
            </div>
            
            <div class="mb-4">
                <div class="form-check">
                    <input type="checkbox" class="form-check-input" id="confirmWithdrawal" required>
                    <label class="form-check-label" for="confirmWithdrawal">
                        <strong>I understand that this action is permanent and cannot be undone</strong>
                    </label>
                </div>
                <div class="form-check">
                    <input type="checkbox" class="form-check-input" id="confirmNotification" required>
                    <label class="form-check-label" for="confirmNotification">
                        I acknowledge that relevant managers and HR will be notified of this withdrawal
                    </label>
                </div>
            </div>
            
            <hr>
            
            <div class="d-flex justify-content-between align-items-center">
                <div>
                    <h6 class="mb-1 text-danger">Confirm Withdrawal</h6>
                    <small class="text-muted">This action will immediately withdraw your application</small>
                </div>
                <div class="btn-group">
                    <a href="view_application.php?id=<?php echo $application_id; ?>" class="btn btn-outline-secondary">
                        <i class="fas fa-times"></i> Cancel
                    </a>
                    <button type="button" class="btn btn-danger" onclick="confirmWithdrawal()" id="withdrawBtn">
                        <i class="fas fa-exclamation-triangle"></i> Withdraw Application
                    </button>
                </div>
            </div>
        </form>
    </div>
</div>

<!-- Final Confirmation Modal -->
<div class="modal fade" id="finalConfirmModal" tabindex="-1" aria-labelledby="finalConfirmModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header bg-danger text-white">
                <h5 class="modal-title" id="finalConfirmModalLabel">
                    <i class="fas fa-exclamation-triangle me-2"></i>Final Confirmation
                </h5>
                <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <div class="text-center mb-3">
                    <i class="fas fa-exclamation-triangle fa-3x text-danger mb-3"></i>
                    <h5>Are you absolutely sure?</h5>
                    <p class="text-muted">You are about to withdraw application:</p>
                    <p class="mb-0">
                        <strong><?php echo htmlspecialchars($application['application_number']); ?></strong><br>
                        <small class="text-muted"><?php echo htmlspecialchars($application['full_name_english']); ?> - <?php echo htmlspecialchars($application['position_title']); ?></small>
                    </p>
                </div>
                
                <div class="alert alert-danger">
                    <i class="fas fa-exclamation-triangle me-2"></i>
                    <strong>This action is irreversible!</strong> Your application will be permanently withdrawn.
                </div>
                
                <p class="mb-0">Type <strong>WITHDRAW</strong> below to confirm:</p>
                <input type="text" class="form-control mt-2" id="confirmationText" placeholder="Type WITHDRAW here">
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <button type="button" class="btn btn-danger" onclick="finalWithdraw()" id="finalWithdrawBtn" disabled>
                    <i class="fas fa-exclamation-triangle"></i> Yes, Withdraw Application
                </button>
            </div>
        </div>
    </div>
</div>

<script>
function confirmWithdrawal() {
    // Check if all checkboxes are checked
    const confirmWithdrawal = document.getElementById('confirmWithdrawal').checked;
    const confirmNotification = document.getElementById('confirmNotification').checked;
    
    if (!confirmWithdrawal || !confirmNotification) {
        alert('Please confirm all requirements before proceeding.');
        return;
    }
    
    // Show final confirmation modal
    const modal = new bootstrap.Modal(document.getElementById('finalConfirmModal'));
    modal.show();
}

function finalWithdraw() {
    const confirmationText = document.getElementById('confirmationText').value.trim();
    
    if (confirmationText.toUpperCase() !== 'WITHDRAW') {
        alert('Please type WITHDRAW correctly to confirm.');
        return;
    }
    
    // Show loading state
    const withdrawBtn = document.getElementById('finalWithdrawBtn');
    withdrawBtn.disabled = true;
    withdrawBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Processing...';
    
    // Submit the form
    document.getElementById('withdrawalForm').submit();
}

// Enable/disable final withdraw button based on confirmation text
document.getElementById('confirmationText').addEventListener('input', function() {
    const finalBtn = document.getElementById('finalWithdrawBtn');
    finalBtn.disabled = this.value.trim().toUpperCase() !== 'WITHDRAW';
});

// Form validation
document.getElementById('withdrawalForm').addEventListener('submit', function(e) {
    const confirmWithdrawal = document.getElementById('confirmWithdrawal').checked;
    const confirmNotification = document.getElementById('confirmNotification').checked;
    
    if (!confirmWithdrawal || !confirmNotification) {
        e.preventDefault();
        alert('Please confirm all requirements before withdrawing the application.');
        return false;
    }
});

// Prevent accidental back button
window.addEventListener('beforeunload', function(e) {
    // Only show warning if form has been interacted with
    const reason = document.querySelector('textarea[name="withdrawal_reason"]').value;
    const confirmChecked = document.getElementById('confirmWithdrawal').checked;
    
    if (reason || confirmChecked) {
        e.preventDefault();
        e.returnValue = 'Are you sure you want to leave? Your withdrawal process will be cancelled.';
        return 'Are you sure you want to leave? Your withdrawal process will be cancelled.';
    }
});

// Auto-focus on reason field
document.addEventListener('DOMContentLoaded', function() {
    document.querySelector('textarea[name="withdrawal_reason"]').focus();
});
</script>

<style>
.card.border-warning {
    border-width: 2px;
}

.btn-danger:disabled {
    opacity: 0.6;
    cursor: not-allowed;
}

.alert-danger {
    border-left: 4px solid #dc3545;
}

.modal-header.bg-danger .btn-close-white {
    filter: invert(1) grayscale(100%) brightness(200%);
}

/* Form styling */
.form-check-input:checked {
    background-color: #dc3545;
    border-color: #dc3545;
}

.form-check-input:focus {
    border-color: #dc3545;
    box-shadow: 0 0 0 0.25rem rgba(220, 53, 69, 0.25);
}

/* Responsive */
@media (max-width: 768px) {
    .btn-group {
        flex-direction: column;
        gap: 0.5rem;
    }
    
    .btn-group .btn {
        width: 100%;
    }
}
</style>

<?php require_once '../includes/footer.php'; ?>