<?php
/**
 * Common PHP Functions
 * Employee Recruitment Management System
 * RR Construction (Pvt) Ltd
 */

// Prevent direct access
if (!defined('HRM_ACCESS')) {
    die('Direct access not permitted');
}

// =============================================
// AUTHENTICATION FUNCTIONS
// =============================================

/**
 * Check if user is logged in
 */
function isLoggedIn() {
    return isset($_SESSION['user_id']) && !empty($_SESSION['user_id']);
}

/**
 * Get current logged in user details
 */
function getCurrentUser() {
    if (!isLoggedIn()) {
        return null;
    }
    
    try {
        $sql = "SELECT * FROM users WHERE id = ? AND status = 'active'";
        return fetchOne($sql, [$_SESSION['user_id']]);
    } catch (Exception $e) {
        error_log("Error getting current user: " . $e->getMessage());
        return null;
    }
}

/**
 * Login user
 */
function loginUser($username, $password) {
    try {
        $sql = "SELECT * FROM users WHERE username = ? AND status = 'active'";
        $user = fetchOne($sql, [$username]);
        
        if ($user && $password === $user['password']) { // Simple password check (no encryption)
            $_SESSION['user_id'] = $user['id'];
            $_SESSION['username'] = $user['username'];
            $_SESSION['role'] = $user['role'];
            $_SESSION['full_name'] = $user['full_name'];
            $_SESSION['login_time'] = time();
            
            // Log login activity
            logActivity($user['id'], 'login', 'users', $user['id']);
            
            return true;
        }
        
        return false;
    } catch (Exception $e) {
        error_log("Login error: " . $e->getMessage());
        return false;
    }
}

/**
 * Logout user
 */
function logoutUser() {
    if (isLoggedIn()) {
        logActivity($_SESSION['user_id'], 'logout', 'users', $_SESSION['user_id']);
    }
    
    session_destroy();
    session_start();
}

/**
 * Check if user has required permission
 */
function requirePermission($requiredRole) {
    if (!isLoggedIn()) {
        header('Location: ' . BASE_URL . 'auth/login.php');
        exit;
    }
    
    $user = getCurrentUser();
    if (!$user || !hasPermission($user['role'], $requiredRole)) {
        setFlashMessage('Access denied. Insufficient permissions.', 'danger');
        header('Location: ' . getDashboardUrl($user['role']));
        exit;
    }
}

/**
 * Get dashboard URL based on user role
 */
function getDashboardUrl($role) {
    $dashboards = [
        ROLE_SUPER_ADMIN => BASE_URL . 'dashboard/super_admin.php',
        ROLE_SITE_ADMIN => BASE_URL . 'dashboard/site_admin.php',
        ROLE_PROJECT_MANAGER => BASE_URL . 'dashboard/project_manager.php',
        ROLE_GENERAL_MANAGER => BASE_URL . 'dashboard/general_manager.php',
        ROLE_HR_MANAGER => BASE_URL . 'dashboard/hr_manager.php',
        ROLE_ACCOUNTANT => BASE_URL . 'dashboard/accountant.php'
    ];
    
    return $dashboards[$role] ?? BASE_URL . 'dashboard/site_admin.php';
}

// =============================================
// FLASH MESSAGE FUNCTIONS
// =============================================

/**
 * Set flash message
 */
function setFlashMessage($message, $type = 'info') {
    if (!isset($_SESSION['flash_messages'])) {
        $_SESSION['flash_messages'] = [];
    }
    
    $_SESSION['flash_messages'][] = [
        'message' => $message,
        'type' => $type,
        'timestamp' => time()
    ];
}

/**
 * Get and clear flash messages
 */
function getFlashMessages() {
    $messages = $_SESSION['flash_messages'] ?? [];
    unset($_SESSION['flash_messages']);
    return $messages;
}

/**
 * Display flash messages HTML
 */
function displayFlashMessages() {
    $messages = getFlashMessages();
    if (empty($messages)) {
        return;
    }
    
    echo '<div class="alert-container">';
    foreach ($messages as $flash) {
        $icon = getFlashIcon($flash['type']);
        echo '<div class="alert alert-' . htmlspecialchars($flash['type']) . ' alert-dismissible fade show" role="alert">';
        echo '<i class="fas fa-' . $icon . ' me-2"></i>';
        echo htmlspecialchars($flash['message']);
        echo '<button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>';
        echo '</div>';
    }
    echo '</div>';
}

/**
 * Get icon for flash message type
 */
function getFlashIcon($type) {
    $icons = [
        'success' => 'check-circle',
        'info' => 'info-circle',
        'warning' => 'exclamation-triangle',
        'danger' => 'exclamation-circle',
        'error' => 'exclamation-circle'
    ];
    return $icons[$type] ?? 'info-circle';
}

// =============================================
// DATA VALIDATION FUNCTIONS
// =============================================

/**
 * Validate required fields
 */
function validateRequired($data, $requiredFields) {
    $errors = [];
    
    foreach ($requiredFields as $field) {
        if (!isset($data[$field]) || empty(trim($data[$field]))) {
            $errors[$field] = ucfirst(str_replace('_', ' ', $field)) . ' is required';
        }
    }
    
    return $errors;
}

/**
 * Validate email
 */
function validateEmail($email) {
    return filter_var($email, FILTER_VALIDATE_EMAIL) !== false;
}

/**
 * Validate NIC number
 */
function validateNIC($nic) {
    return preg_match(VALIDATION_RULES['nic'], $nic);
}

/**
 * Validate phone number
 */
function validatePhone($phone) {
    return preg_match(VALIDATION_RULES['phone'], $phone);
}

/**
 * Validate file upload
 */
function validateFileUpload($file, $allowedTypes = [], $maxSize = MAX_FILE_SIZE) {
    if ($file['error'] !== UPLOAD_ERR_OK) {
        return 'File upload failed';
    }
    
    if ($file['size'] > $maxSize) {
        return 'File size exceeds maximum limit';
    }
    
    if (!empty($allowedTypes)) {
        $fileExtension = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
        if (!in_array($fileExtension, $allowedTypes)) {
            return 'File type not allowed';
        }
    }
    
    return true;
}

// =============================================
// FILE HANDLING FUNCTIONS
// =============================================

/**
 * Upload file safely
 */
function uploadFile($file, $uploadDir, $allowedTypes = [], $maxSize = MAX_FILE_SIZE) {
    // Validate file
    $validation = validateFileUpload($file, $allowedTypes, $maxSize);
    if ($validation !== true) {
        return ['success' => false, 'error' => $validation];
    }
    
    // Create directory if it doesn't exist
    if (!createDirectory($uploadDir)) {
        return ['success' => false, 'error' => 'Failed to create upload directory'];
    }
    
    // Generate unique filename
    $fileExtension = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
    $filename = uniqid() . '_' . time() . '.' . $fileExtension;
    $filepath = $uploadDir . $filename;
    
    // Move uploaded file
    if (move_uploaded_file($file['tmp_name'], $filepath)) {
        return [
            'success' => true,
            'filename' => $filename,
            'filepath' => $filepath,
            'original_name' => $file['name'],
            'size' => $file['size'],
            'type' => $file['type']
        ];
    }
    
    return ['success' => false, 'error' => 'Failed to move uploaded file'];
}

/**
 * Delete file safely
 */
function deleteFile($filepath) {
    if (file_exists($filepath)) {
        return unlink($filepath);
    }
    return true;
}

/**
 * Get file size in human readable format
 */
function formatFileSize($bytes) {
    if ($bytes == 0) return '0 Bytes';
    
    $k = 1024;
    $sizes = ['Bytes', 'KB', 'MB', 'GB', 'TB'];
    $i = floor(log($bytes) / log($k));
    
    return round($bytes / pow($k, $i), 2) . ' ' . $sizes[$i];
}

// =============================================
// DATA FORMATTING FUNCTIONS
// =============================================

/**
 * Format date for display
 */
function formatDisplayDate($date, $format = DISPLAY_DATE_FORMAT) {
    if (empty($date) || $date === '0000-00-00') {
        return '-';
    }
    
    return date($format, strtotime($date));
}

/**
 * Format datetime for display
 */
function formatDisplayDateTime($datetime, $format = DISPLAY_DATETIME_FORMAT) {
    if (empty($datetime) || $datetime === '0000-00-00 00:00:00') {
        return '-';
    }
    
    return date($format, strtotime($datetime));
}

/**
 * Format currency amount
 */
function formatCurrencyAmount($amount) {
    if (empty($amount) || $amount == 0) {
        return '-';
    }
    
    return 'Rs. ' . number_format($amount, 2);
}

/**
 * Get relative time (e.g., "2 hours ago")
 */
function getRelativeTime($datetime) {
    $time = time() - strtotime($datetime);
    
    if ($time < 60) return 'just now';
    if ($time < 3600) return floor($time/60) . ' minutes ago';
    if ($time < 86400) return floor($time/3600) . ' hours ago';
    if ($time < 2592000) return floor($time/86400) . ' days ago';
    if ($time < 31536000) return floor($time/2592000) . ' months ago';
    
    return floor($time/31536000) . ' years ago';
}

// =============================================
// STATUS AND APPROVAL FUNCTIONS
// =============================================

/**
 * Get next approval level for request
 */
function getNextApprovalLevel($currentStatus) {
    $flow = [
        REQUEST_STATUS_PENDING => ROLE_PROJECT_MANAGER,
        REQUEST_STATUS_APPROVED_BY_PM => ROLE_GENERAL_MANAGER,
        REQUEST_STATUS_APPROVED_BY_GM => ROLE_HR_MANAGER
    ];
    
    return $flow[$currentStatus] ?? null;
}

/**
 * Get next status after approval
 */
function getNextRequestStatus($currentStatus) {
    $flow = [
        REQUEST_STATUS_PENDING => REQUEST_STATUS_APPROVED_BY_PM,
        REQUEST_STATUS_APPROVED_BY_PM => REQUEST_STATUS_APPROVED_BY_GM,
        REQUEST_STATUS_APPROVED_BY_GM => REQUEST_STATUS_APPROVED_FINAL
    ];
    
    return $flow[$currentStatus] ?? $currentStatus;
}

/**
 * Check if user can approve request
 */
function canApproveRequest($requestStatus, $userRole) {
    $nextApprovalLevel = getNextApprovalLevel($requestStatus);
    return $nextApprovalLevel === $userRole || $userRole === ROLE_SUPER_ADMIN;
}

/**
 * Get status badge HTML
 */
function getStatusBadge($status, $type = 'request') {
    $color = getStatusColor($status);
    $name = getStatusName($status, $type);
    
    return '<span class="status-badge" style="background-color: ' . $color . '; color: white;">' . 
           htmlspecialchars($name) . '</span>';
}

// =============================================
// PAGINATION FUNCTIONS
// =============================================

/**
 * Generate pagination data
 */
function getPaginationData($totalRecords, $currentPage = 1, $recordsPerPage = RECORDS_PER_PAGE) {
    $totalPages = ceil($totalRecords / $recordsPerPage);
    $currentPage = max(1, min($currentPage, $totalPages));
    $offset = ($currentPage - 1) * $recordsPerPage;
    
    return [
        'total_records' => $totalRecords,
        'total_pages' => $totalPages,
        'current_page' => $currentPage,
        'records_per_page' => $recordsPerPage,
        'offset' => $offset,
        'has_previous' => $currentPage > 1,
        'has_next' => $currentPage < $totalPages,
        'previous_page' => $currentPage - 1,
        'next_page' => $currentPage + 1
    ];
}

/**
 * Generate pagination HTML
 */
function generatePaginationHTML($pagination, $baseUrl = '') {
    if ($pagination['total_pages'] <= 1) {
        return '';
    }
    
    $html = '<nav aria-label="Page navigation"><ul class="pagination justify-content-center">';
    
    // Previous button
    if ($pagination['has_previous']) {
        $html .= '<li class="page-item">';
        $html .= '<a class="page-link" href="' . $baseUrl . '?page=' . $pagination['previous_page'] . '">';
        $html .= '<i class="fas fa-chevron-left"></i> Previous</a></li>';
    } else {
        $html .= '<li class="page-item disabled">';
        $html .= '<span class="page-link"><i class="fas fa-chevron-left"></i> Previous</span></li>';
    }
    
    // Page numbers
    $start = max(1, $pagination['current_page'] - 2);
    $end = min($pagination['total_pages'], $pagination['current_page'] + 2);
    
    if ($start > 1) {
        $html .= '<li class="page-item"><a class="page-link" href="' . $baseUrl . '?page=1">1</a></li>';
        if ($start > 2) {
            $html .= '<li class="page-item disabled"><span class="page-link">...</span></li>';
        }
    }
    
    for ($i = $start; $i <= $end; $i++) {
        if ($i == $pagination['current_page']) {
            $html .= '<li class="page-item active"><span class="page-link">' . $i . '</span></li>';
        } else {
            $html .= '<li class="page-item"><a class="page-link" href="' . $baseUrl . '?page=' . $i . '">' . $i . '</a></li>';
        }
    }
    
    if ($end < $pagination['total_pages']) {
        if ($end < $pagination['total_pages'] - 1) {
            $html .= '<li class="page-item disabled"><span class="page-link">...</span></li>';
        }
        $html .= '<li class="page-item"><a class="page-link" href="' . $baseUrl . '?page=' . $pagination['total_pages'] . '">' . $pagination['total_pages'] . '</a></li>';
    }
    
    // Next button
    if ($pagination['has_next']) {
        $html .= '<li class="page-item">';
        $html .= '<a class="page-link" href="' . $baseUrl . '?page=' . $pagination['next_page'] . '">';
        $html .= 'Next <i class="fas fa-chevron-right"></i></a></li>';
    } else {
        $html .= '<li class="page-item disabled">';
        $html .= '<span class="page-link">Next <i class="fas fa-chevron-right"></i></span></li>';
    }
    
    $html .= '</ul></nav>';
    
    return $html;
}

// =============================================
// SEARCH AND FILTER FUNCTIONS
// =============================================

/**
 * Build search query
 */
function buildSearchQuery($searchTerm, $searchFields) {
    if (empty($searchTerm) || strlen($searchTerm) < SEARCH_MIN_LENGTH) {
        return ['where' => '', 'params' => []];
    }
    
    $conditions = [];
    $params = [];
    
    foreach ($searchFields as $field) {
        $conditions[] = "$field LIKE ?";
        $params[] = "%$searchTerm%";
    }
    
    return [
        'where' => '(' . implode(' OR ', $conditions) . ')',
        'params' => $params
    ];
}

/**
 * Apply filters to query
 */
function applyFilters($filters, $allowedFilters) {
    $conditions = [];
    $params = [];
    
    foreach ($filters as $key => $value) {
        if (isset($allowedFilters[$key]) && !empty($value)) {
            $field = $allowedFilters[$key];
            
            if (is_array($value)) {
                $placeholders = str_repeat('?,', count($value) - 1) . '?';
                $conditions[] = "$field IN ($placeholders)";
                $params = array_merge($params, $value);
            } else {
                $conditions[] = "$field = ?";
                $params[] = $value;
            }
        }
    }
    
    return [
        'where' => implode(' AND ', $conditions),
        'params' => $params
    ];
}

// =============================================
// NOTIFICATION FUNCTIONS
// =============================================

/**
 * Send email notification
 */
function sendNotification($to, $subject, $message, $type = 'email') {
    if (!ENABLE_EMAIL_NOTIFICATIONS) {
        return false;
    }
    
    try {
        // Simple mail function (replace with PHPMailer for production)
        $headers = "From: " . FROM_NAME . " <" . FROM_EMAIL . ">\r\n";
        $headers .= "Content-Type: text/html; charset=UTF-8\r\n";
        
        $htmlMessage = getEmailTemplate($subject, $message);
        
        return mail($to, $subject, $htmlMessage, $headers);
    } catch (Exception $e) {
        error_log("Email notification error: " . $e->getMessage());
        return false;
    }
}

/**
 * Get email template
 */
function getEmailTemplate($subject, $message) {
    return "
    <html>
    <head>
        <title>$subject</title>
        <style>
            body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
            .container { max-width: 600px; margin: 0 auto; padding: 20px; }
            .header { background: " . PRIMARY_COLOR . "; color: white; padding: 20px; text-align: center; }
            .content { padding: 20px; background: #f9f9f9; }
            .footer { text-align: center; padding: 10px; font-size: 12px; color: #666; }
        </style>
    </head>
    <body>
        <div class='container'>
            <div class='header'>
                <h2>" . COMPANY_NAME . "</h2>
                <p>" . APP_NAME . "</p>
            </div>
            <div class='content'>
                $message
            </div>
            <div class='footer'>
                <p>&copy; " . date('Y') . " " . COMPANY_NAME . ". All rights reserved.</p>
            </div>
        </div>
    </body>
    </html>
    ";
}

/**
 * Notify request approval
 */
function notifyRequestApproval($requestId, $status, $comments = '') {
    try {
        $sql = "SELECT er.*, u.full_name, u.email 
                FROM employee_requests er 
                JOIN users u ON er.requested_by = u.id 
                WHERE er.id = ?";
        $request = fetchOne($sql, [$requestId]);
        
        if (!$request || empty($request['email'])) {
            return false;
        }
        
        $statusName = getStatusName($status);
        $subject = "Employee Request " . $statusName;
        
        $message = "
        <h3>Employee Request Update</h3>
        <p>Dear {$request['full_name']},</p>
        <p>Your employee request <strong>{$request['request_number']}</strong> has been <strong>$statusName</strong>.</p>
        <p><strong>Project:</strong> {$request['site_project']}</p>
        <p><strong>Request Date:</strong> " . formatDisplayDate($request['request_date']) . "</p>
        ";
        
        if (!empty($comments)) {
            $message .= "<p><strong>Comments:</strong> $comments</p>";
        }
        
        $message .= "
        <p>Please log in to the system to view the complete details.</p>
        <p>Best regards,<br>" . COMPANY_NAME . "</p>
        ";
        
        return sendNotification($request['email'], $subject, $message);
    } catch (Exception $e) {
        error_log("Notification error: " . $e->getMessage());
        return false;
    }
}

// =============================================
// DASHBOARD FUNCTIONS
// =============================================

/**
 * Get dashboard statistics for user role
 */
function getDashboardStats($userId, $userRole) {
    try {
        $stats = [];
        
        switch ($userRole) {
            case ROLE_SITE_ADMIN:
                $stats = getSiteAdminStats($userId);
                break;
            case ROLE_PROJECT_MANAGER:
                $stats = getProjectManagerStats($userId);
                break;
            case ROLE_GENERAL_MANAGER:
                $stats = getGeneralManagerStats();
                break;
            case ROLE_HR_MANAGER:
                $stats = getHRManagerStats();
                break;
            case ROLE_ACCOUNTANT:
                $stats = getAccountantStats();
                break;
            case ROLE_SUPER_ADMIN:
                $stats = getSuperAdminStats();
                break;
        }
        
        return $stats;
    } catch (Exception $e) {
        error_log("Dashboard stats error: " . $e->getMessage());
        return [];
    }
}

/**
 * Get site admin statistics
 */
function getSiteAdminStats($userId) {
    $sql = "SELECT 
                COUNT(CASE WHEN status = 'pending' THEN 1 END) as pending_requests,
                COUNT(CASE WHEN status IN ('approved_by_pm', 'approved_by_gm', 'approved_final') THEN 1 END) as approved_requests,
                COUNT(CASE WHEN status = 'rejected' THEN 1 END) as rejected_requests,
                COUNT(*) as total_requests
            FROM employee_requests 
            WHERE requested_by = ?";
    
    $requestStats = fetchOne($sql, [$userId]);
    
    $sql = "SELECT COUNT(*) as total_applications
            FROM job_applications ja
            JOIN employee_requests er ON ja.request_id = er.id
            WHERE er.requested_by = ?";
    
    $appStats = fetchOne($sql, [$userId]);
    
    return array_merge($requestStats, $appStats);
}

/**
 * Get project manager statistics
 */
function getProjectManagerStats($userId) {
    $user = fetchOne("SELECT site_project FROM users WHERE id = ?", [$userId]);
    
    $sql = "SELECT 
                COUNT(CASE WHEN er.status = 'pending' THEN 1 END) as pending_requests,
                COUNT(CASE WHEN ja.status IN ('submitted', 'under_review') THEN 1 END) as pending_applications,
                COUNT(CASE WHEN er.status != 'rejected' THEN 1 END) as total_requests,
                COUNT(ja.id) as total_applications
            FROM employee_requests er
            LEFT JOIN job_applications ja ON er.id = ja.request_id
            WHERE er.site_project = ?";
    
    return fetchOne($sql, [$user['site_project']]);
}

/**
 * Get general manager statistics
 */
function getGeneralManagerStats() {
    $sql = "SELECT 
                COUNT(CASE WHEN er.status = 'approved_by_pm' THEN 1 END) as pending_requests,
                COUNT(CASE WHEN er.status != 'rejected' THEN 1 END) as total_requests,
                COUNT(ja.id) as total_applications,
                COUNT(CASE WHEN ja.status = 'hired' THEN 1 END) as hired_employees
            FROM employee_requests er
            LEFT JOIN job_applications ja ON er.id = ja.request_id";
    
    return fetchOne($sql);
}

/**
 * Get HR manager statistics
 */
function getHRManagerStats() {
    $sql = "SELECT 
                COUNT(CASE WHEN er.status = 'approved_by_gm' THEN 1 END) as pending_request_approvals,
                COUNT(CASE WHEN ja.status = 'approved_by_pm' THEN 1 END) as pending_application_approvals,
                COUNT(CASE WHEN ja.status = 'hired' THEN 1 END) as hired_employees,
                COUNT(ja.id) as total_applications
            FROM employee_requests er
            LEFT JOIN job_applications ja ON er.id = ja.request_id";
    
    return fetchOne($sql);
}

/**
 * Get accountant statistics
 */
function getAccountantStats() {
    $sql = "SELECT 
                COUNT(CASE WHEN ja.status = 'approved_by_hr' THEN 1 END) as pending_approvals,
                COUNT(CASE WHEN ja.status = 'hired' THEN 1 END) as hired_employees,
                SUM(CASE WHEN ja.status = 'hired' THEN ja.expected_salary END) as total_salary_cost,
                COUNT(ja.id) as total_applications
            FROM job_applications ja";
    
    return fetchOne($sql);
}

/**
 * Get super admin statistics
 */
function getSuperAdminStats() {
    $sql = "SELECT 
                (SELECT COUNT(*) FROM users WHERE status = 'active') as active_users,
                (SELECT COUNT(*) FROM employee_requests WHERE status != 'rejected') as active_requests,
                (SELECT COUNT(*) FROM job_applications WHERE status != 'rejected') as total_applications,
                (SELECT COUNT(*) FROM sites_projects WHERE status = 'active') as active_projects";
    
    return fetchOne($sql);
}

// =============================================
// REPORT FUNCTIONS
// =============================================

/**
 * Get request summary report data
 */
function getRequestSummaryReport($filters = []) {
    $whereConditions = ['1=1'];
    $params = [];
    
    if (!empty($filters['date_from'])) {
        $whereConditions[] = 'er.request_date >= ?';
        $params[] = $filters['date_from'];
    }
    
    if (!empty($filters['date_to'])) {
        $whereConditions[] = 'er.request_date <= ?';
        $params[] = $filters['date_to'];
    }
    
    if (!empty($filters['site_project'])) {
        $whereConditions[] = 'er.site_project = ?';
        $params[] = $filters['site_project'];
    }
    
    if (!empty($filters['status'])) {
        $whereConditions[] = 'er.status = ?';
        $params[] = $filters['status'];
    }
    
    $sql = "SELECT er.*, u.full_name as requested_by_name,
                   COUNT(ec.id) as categories_count,
                   COUNT(ja.id) as applications_count
            FROM employee_requests er
            LEFT JOIN users u ON er.requested_by = u.id
            LEFT JOIN employee_categories ec ON er.id = ec.request_id
            LEFT JOIN job_applications ja ON er.id = ja.request_id
            WHERE " . implode(' AND ', $whereConditions) . "
            GROUP BY er.id
            ORDER BY er.created_at DESC";
    
    return fetchAll($sql, $params);
}

/**
 * Get application summary report data
 */
function getApplicationSummaryReport($filters = []) {
    $whereConditions = ['1=1'];
    $params = [];
    
    if (!empty($filters['date_from'])) {
        $whereConditions[] = 'ja.submitted_at >= ?';
        $params[] = $filters['date_from'];
    }
    
    if (!empty($filters['date_to'])) {
        $whereConditions[] = 'ja.submitted_at <= ?';
        $params[] = $filters['date_to'];
    }
    
    if (!empty($filters['position'])) {
        $whereConditions[] = 'ja.position_applied LIKE ?';
        $params[] = '%' . $filters['position'] . '%';
    }
    
    if (!empty($filters['status'])) {
        $whereConditions[] = 'ja.status = ?';
        $params[] = $filters['status'];
    }
    
    $sql = "SELECT ja.*, er.site_project, er.request_number,
                   u.full_name as submitted_by_name
            FROM job_applications ja
            LEFT JOIN employee_requests er ON ja.request_id = er.id
            LEFT JOIN users u ON ja.submitted_by = u.id
            WHERE " . implode(' AND ', $whereConditions) . "
            ORDER BY ja.submitted_at DESC";
    
    return fetchAll($sql, $params);
}

// =============================================
// UTILITY FUNCTIONS
// =============================================

/**
 * Clean and validate input data
 */
function cleanInput($data) {
    if (is_array($data)) {
        return array_map('cleanInput', $data);
    }
    
    return htmlspecialchars(strip_tags(trim($data)), ENT_QUOTES, 'UTF-8');
}

/**
 * Generate random string
 */
function generateRandomString($length = 10) {
    $characters = '0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ';
    $randomString = '';
    
    for ($i = 0; $i < $length; $i++) {
        $randomString .= $characters[rand(0, strlen($characters) - 1)];
    }
    
    return $randomString;
}

/**
 * Check if string is JSON
 */
function isJson($string) {
    json_decode($string);
    return (json_last_error() == JSON_ERROR_NONE);
}

/**
 * Convert array to CSV line
 */
function arrayToCSV($array) {
    $output = fopen('php://output', 'w');
    foreach ($array as $row) {
        fputcsv($output, $row);
    }
    fclose($output);
}

/**
 * Get client IP address
 */
function getClientIP() {
    $ipKeys = ['HTTP_CLIENT_IP', 'HTTP_X_FORWARDED_FOR', 'REMOTE_ADDR'];
    
    foreach ($ipKeys as $key) {
        if (array_key_exists($key, $_SERVER) === true) {
            foreach (explode(',', $_SERVER[$key]) as $ip) {
                $ip = trim($ip);
                if (filter_var($ip, FILTER_VALIDATE_IP, 
                    FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE) !== false) {
                    return $ip;
                }
            }
        }
    }
    
    return $_SERVER['REMOTE_ADDR'] ?? '127.0.0.1';
}

/**
 * Log system error
 */
function logSystemError($message, $context = []) {
    $logMessage = date('Y-m-d H:i:s') . " - " . $message;
    
    if (!empty($context)) {
        $logMessage .= " - Context: " . json_encode($context);
    }
    
    error_log($logMessage);
}

/**
 * Check if request is AJAX
 */
function isAjaxRequest() {
    return isset($_SERVER['HTTP_X_REQUESTED_WITH']) && 
           strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) === 'xmlhttprequest';
}

/**
 * Send JSON response
 */
function sendJsonResponse($data, $statusCode = 200) {
    http_response_code($statusCode);
    header('Content-Type: application/json');
    echo json_encode($data);
    exit;
}

/**
 * Redirect with message
 */
function redirectWithMessage($url, $message, $type = 'info') {
    setFlashMessage($message, $type);
    header('Location: ' . $url);
    exit;
}

/**
 * Get browser information
 */
function getBrowserInfo() {
    $userAgent = $_SERVER['HTTP_USER_AGENT'] ?? 'Unknown';
    
    if (strpos($userAgent, 'Chrome') !== false) {
        return 'Chrome';
    } elseif (strpos($userAgent, 'Firefox') !== false) {
        return 'Firefox';
    } elseif (strpos($userAgent, 'Safari') !== false) {
        return 'Safari';
    } elseif (strpos($userAgent, 'Edge') !== false) {
        return 'Edge';
    } elseif (strpos($userAgent, 'Opera') !== false) {
        return 'Opera';
    } else {
        return 'Unknown';
    }
}

/**
 * Calculate age from date of birth
 */
function calculateAge($dateOfBirth) {
    $birthDate = new DateTime($dateOfBirth);
    $today = new DateTime('today');
    return $birthDate->diff($today)->y;
}

/**
 * Generate breadcrumb array
 */
function generateBreadcrumbs($items) {
    $breadcrumbs = [];
    
    foreach ($items as $item) {
        $breadcrumbs[] = [
            'title' => $item['title'] ?? '',
            'url' => $item['url'] ?? '#'
        ];
    }
    
    return $breadcrumbs;
}

/**
 * Get file icon based on extension
 */
function getFileIcon($filename) {
    $extension = strtolower(pathinfo($filename, PATHINFO_EXTENSION));
    
    $icons = [
        'pdf' => 'fa-file-pdf',
        'doc' => 'fa-file-word',
        'docx' => 'fa-file-word',
        'xls' => 'fa-file-excel',
        'xlsx' => 'fa-file-excel',
        'jpg' => 'fa-file-image',
        'jpeg' => 'fa-file-image',
        'png' => 'fa-file-image',
        'gif' => 'fa-file-image',
        'txt' => 'fa-file-alt',
        'zip' => 'fa-file-archive',
        'rar' => 'fa-file-archive'
    ];
    
    return 'fas ' . ($icons[$extension] ?? 'fa-file');
}

/**
 * Check if mobile device
 */
function isMobileDevice() {
    return preg_match('/Mobile|Android|iPhone|iPad|iPod|BlackBerry|IEMobile|Opera Mini/i', 
                     $_SERVER['HTTP_USER_AGENT'] ?? '');
}

/**
 * Get system timezone
 */
function getSystemTimezone() {
    return date_default_timezone_get();
}

/**
 * Convert timezone
 */
function convertTimezone($datetime, $fromTz = 'UTC', $toTz = 'Asia/Colombo') {
    $date = new DateTime($datetime, new DateTimeZone($fromTz));
    $date->setTimezone(new DateTimeZone($toTz));
    return $date->format('Y-m-d H:i:s');
}

// =============================================
// INITIALIZATION
// =============================================

// Set default timezone
date_default_timezone_set('Asia/Colombo');

// Set session configuration BEFORE starting session
if (session_status() === PHP_SESSION_NONE) {
    ini_set('session.cookie_httponly', 1);
    ini_set('session.use_only_cookies', 1);
    ini_set('session.cookie_secure', isset($_SERVER['HTTPS']));
    
    // Initialize session
    session_start();
}

// Check session timeout
if (isLoggedIn() && isset($_SESSION['login_time'])) {
    if (time() - $_SESSION['login_time'] > SESSION_TIMEOUT) {
        logoutUser();
        setFlashMessage('Session expired. Please log in again.', 'warning');
        if (!in_array(basename($_SERVER['PHP_SELF']), ['login.php', 'index.php'])) {
            header('Location: ' . BASE_URL . 'auth/login.php');
            exit;
        }
    }
}


/////////////////////////////////

/**
 * Additional Functions for Application Management
 * Add these functions to your existing functions.php file
 */

/**
 * Check if a position has available slots
 */
function hasAvailablePositions($categoryId) {
    try {
        $sql = "SELECT ec.quantity,
                       COUNT(CASE WHEN ja.status = 'hired' THEN 1 END) as hired_count,
                       (ec.quantity - COUNT(CASE WHEN ja.status = 'hired' THEN 1 END)) as available_positions
                FROM employee_categories ec
                LEFT JOIN job_applications ja ON ec.id = ja.category_id
                WHERE ec.id = ?
                GROUP BY ec.id";
        
        $result = fetchOne($sql, [$categoryId]);
        
        return $result && $result['available_positions'] > 0;
    } catch (Exception $e) {
        error_log("Error checking position availability: " . $e->getMessage());
        return false;
    }
}

/**
 * Get position availability details
 */
function getPositionAvailability($categoryId) {
    try {
        $sql = "SELECT ec.id, ec.position_title, ec.quantity,
                       COUNT(CASE WHEN ja.status = 'hired' THEN 1 END) as hired_count,
                       COUNT(CASE WHEN ja.status IN ('submitted', 'under_review', 'approved_by_pm', 'approved_by_hr', 'approved_by_accountant') THEN 1 END) as pending_count,
                       (ec.quantity - COUNT(CASE WHEN ja.status = 'hired' THEN 1 END)) as available_positions
                FROM employee_categories ec
                LEFT JOIN job_applications ja ON ec.id = ja.category_id
                WHERE ec.id = ?
                GROUP BY ec.id";
        
        return fetchOne($sql, [$categoryId]);
    } catch (Exception $e) {
        error_log("Error getting position availability: " . $e->getMessage());
        return null;
    }
}

/**
 * Check if user can apply for a position (not duplicate NIC)
 */
function canApplyForPosition($nic, $categoryId) {
    try {
        $sql = "SELECT COUNT(*) as count 
                FROM job_applications 
                WHERE nic_number = ? AND category_id = ? AND status != 'rejected'";
        
        $result = fetchOne($sql, [$nic, $categoryId]);
        
        return $result['count'] == 0;
    } catch (Exception $e) {
        error_log("Error checking application eligibility: " . $e->getMessage());
        return false;
    }
}

/**
 * Get applications summary for a request
 */
function getRequestApplicationsSummary($requestId) {
    try {
        $sql = "SELECT ec.id as category_id, ec.position_title, ec.quantity,
                       COUNT(ja.id) as total_applications,
                       COUNT(CASE WHEN ja.status = 'hired' THEN 1 END) as hired_count,
                       COUNT(CASE WHEN ja.status IN ('submitted', 'under_review', 'approved_by_pm', 'approved_by_hr', 'approved_by_accountant') THEN 1 END) as pending_count,
                       (ec.quantity - COUNT(CASE WHEN ja.status = 'hired' THEN 1 END)) as available_positions
                FROM employee_categories ec
                LEFT JOIN job_applications ja ON ec.id = ja.category_id
                WHERE ec.request_id = ?
                GROUP BY ec.id, ec.position_title, ec.quantity
                ORDER BY ec.position_title";
        
        return fetchAll($sql, [$requestId]);
    } catch (Exception $e) {
        error_log("Error getting request applications summary: " . $e->getMessage());
        return [];
    }
}

/**
 * Update application status and check if position is now full
 */
function updateApplicationStatus($applicationId, $newStatus, $userId, $comments = '') {
    try {
        beginTransaction();
        
        // Get current application details
        $application = fetchOne("SELECT * FROM job_applications WHERE id = ?", [$applicationId]);
        if (!$application) {
            throw new Exception("Application not found");
        }
        
        $oldStatus = $application['status'];
        
        // Update application status
        updateRecord('job_applications', 
            ['status' => $newStatus], 
            'id = ?', 
            [$applicationId]
        );
        
        // Log the status change
        logActivity($userId, 'update_status', 'job_applications', $applicationId, 
                   ['old_status' => $oldStatus], 
                   ['new_status' => $newStatus, 'comments' => $comments]);
        
        // If status changed to 'hired', check if position is now full
        if ($newStatus === 'hired') {
            $availability = getPositionAvailability($application['category_id']);
            
            if ($availability && $availability['available_positions'] <= 0) {
                // Position is now full - log this event
                logActivity($userId, 'position_filled', 'employee_categories', $application['category_id']);
                
                // Optional: Send notification that position is filled
                // notifyPositionFilled($application['category_id']);
            }
        }
        
        commit();
        return true;
        
    } catch (Exception $e) {
        rollback();
        error_log("Error updating application status: " . $e->getMessage());
        return false;
    }
}

/**
 * Get hiring statistics for dashboard
 */
function getHiringStatistics($userId = null, $userRole = null) {
    try {
        $whereClause = '';
        $params = [];
        
        // Filter by user's project if they're site admin
        if ($userRole === ROLE_SITE_ADMIN && $userId) {
            $user = fetchOne("SELECT site_project FROM users WHERE id = ?", [$userId]);
            if ($user && $user['site_project']) {
                $whereClause = "WHERE er.site_project = ?";
                $params[] = $user['site_project'];
            }
        }
        
        $sql = "SELECT 
                    COUNT(DISTINCT er.id) as total_requests,
                    COUNT(DISTINCT ec.id) as total_positions,
                    SUM(ec.quantity) as total_positions_requested,
                    COUNT(DISTINCT ja.id) as total_applications,
                    COUNT(CASE WHEN ja.status = 'hired' THEN 1 END) as total_hired,
                    COUNT(CASE WHEN ja.status IN ('submitted', 'under_review', 'approved_by_pm', 'approved_by_hr', 'approved_by_accountant') THEN 1 END) as pending_applications,
                    COUNT(CASE WHEN ja.status = 'rejected' THEN 1 END) as rejected_applications
                FROM employee_requests er
                LEFT JOIN employee_categories ec ON er.id = ec.request_id
                LEFT JOIN job_applications ja ON ec.id = ja.category_id
                {$whereClause}";
        
        return fetchOne($sql, $params);
    } catch (Exception $e) {
        error_log("Error getting hiring statistics: " . $e->getMessage());
        return [];
    }
}

/**
 * Get positions that are close to being filled (80% or more)
 */
function getPositionsNearlyFilled($threshold = 0.8) {
    try {
        $sql = "SELECT ec.id, ec.position_title, ec.quantity, er.request_number, er.site_project,
                       COUNT(CASE WHEN ja.status = 'hired' THEN 1 END) as hired_count,
                       (COUNT(CASE WHEN ja.status = 'hired' THEN 1 END) / ec.quantity) as fill_percentage,
                       (ec.quantity - COUNT(CASE WHEN ja.status = 'hired' THEN 1 END)) as remaining_positions
                FROM employee_categories ec
                JOIN employee_requests er ON ec.request_id = er.id
                LEFT JOIN job_applications ja ON ec.id = ja.category_id
                WHERE er.status = 'approved_final'
                GROUP BY ec.id
                HAVING fill_percentage >= ? AND remaining_positions > 0
                ORDER BY fill_percentage DESC";
        
        return fetchAll($sql, [$threshold]);
    } catch (Exception $e) {
        error_log("Error getting positions nearly filled: " . $e->getMessage());
        return [];
    }
}

/**
 * Check and prevent application submission if position is full
 */
function validateApplicationSubmission($categoryId, $nic) {
    $errors = [];
    
    // Check if position has available slots
    if (!hasAvailablePositions($categoryId)) {
        $errors[] = "This position is no longer available for applications.";
    }
    
    // Check for duplicate NIC
    if (!canApplyForPosition($nic, $categoryId)) {
        $errors[] = "An application with this NIC already exists for this position.";
    }
    
    return $errors;
}

/**
 * Generate position availability report
 */
function generateAvailabilityReport($siteProject = null) {
    try {
        $whereClause = "WHERE er.status IN ('approved_by_pm', 'approved_by_gm', 'approved_final')";
        $params = [];
        
        if ($siteProject) {
            $whereClause .= " AND er.site_project = ?";
            $params[] = $siteProject;
        }
        
        $sql = "SELECT er.request_number, er.site_project, er.request_date,
                       ec.position_title, ec.quantity,
                       COUNT(ja.id) as total_applications,
                       COUNT(CASE WHEN ja.status = 'hired' THEN 1 END) as hired_count,
                       COUNT(CASE WHEN ja.status IN ('submitted', 'under_review', 'approved_by_pm', 'approved_by_hr', 'approved_by_accountant') THEN 1 END) as pending_count,
                       (ec.quantity - COUNT(CASE WHEN ja.status = 'hired' THEN 1 END)) as available_positions,
                       CASE 
                           WHEN (ec.quantity - COUNT(CASE WHEN ja.status = 'hired' THEN 1 END)) = 0 THEN 'Filled'
                           WHEN (COUNT(CASE WHEN ja.status = 'hired' THEN 1 END) / ec.quantity) >= 0.8 THEN 'Nearly Full'
                           ELSE 'Available'
                       END as status
                FROM employee_requests er
                JOIN employee_categories ec ON er.id = ec.request_id
                LEFT JOIN job_applications ja ON ec.id = ja.category_id
                {$whereClause}
                GROUP BY er.id, ec.id
                ORDER BY er.request_date DESC, ec.position_title";
        
        return fetchAll($sql, $params);
    } catch (Exception $e) {
        error_log("Error generating availability report: " . $e->getMessage());
        return [];
    }
}


/////////////////////////////////


//////////////MULTIPLE PROJECTS////////////////////

/**
 * User Projects Helper Functions
 * Add these functions to your existing functions.php file
 */

/**
 * Get projects assigned to a user
 */
function getUserProjects($userId) {
    try {
        $sql = "SELECT sp.id, sp.name, sp.code, sp.location, sp.status,
                       up.assigned_at, up.assigned_by,
                       assigned_by_user.full_name as assigned_by_name
                FROM user_projects up
                JOIN sites_projects sp ON up.project_id = sp.id
                LEFT JOIN users assigned_by_user ON up.assigned_by = assigned_by_user.id
                WHERE up.user_id = ?
                ORDER BY sp.name";
        
        return fetchAll($sql, [$userId]);
    } catch (Exception $e) {
        error_log("Error getting user projects: " . $e->getMessage());
        return [];
    }
}

/**
 * Get project names assigned to a user (for display)
 */
function getUserProjectNames($userId) {
    $projects = getUserProjects($userId);
    return array_column($projects, 'name');
}

/**
 * Assign projects to a user
 */
function assignProjectsToUser($userId, $projectIds, $assignedBy) {
    try {
        beginTransaction();
        
        // Remove existing assignments
        deleteRecord('user_projects', 'user_id = ?', [$userId]);
        
        // Add new assignments
        foreach ($projectIds as $projectId) {
            if (!empty($projectId)) {
                insertRecord('user_projects', [
                    'user_id' => $userId,
                    'project_id' => $projectId,
                    'assigned_by' => $assignedBy
                ]);
            }
        }
        
        commit();
        return true;
    } catch (Exception $e) {
        rollback();
        error_log("Error assigning projects to user: " . $e->getMessage());
        return false;
    }
}

/**
 * Check if user has access to a specific project
 */
function userHasProjectAccess($userId, $projectId) {
    try {
        $sql = "SELECT COUNT(*) as count FROM user_projects WHERE user_id = ? AND project_id = ?";
        $result = fetchOne($sql, [$userId, $projectId]);
        return $result['count'] > 0;
    } catch (Exception $e) {
        error_log("Error checking project access: " . $e->getMessage());
        return false;
    }
}

/**
 * Check if user has access to a project by name
 */
function userHasProjectAccessByName($userId, $projectName) {
    try {
        $sql = "SELECT COUNT(*) as count 
                FROM user_projects up
                JOIN sites_projects sp ON up.project_id = sp.id
                WHERE up.user_id = ? AND sp.name = ?";
        $result = fetchOne($sql, [$userId, $projectName]);
        return $result['count'] > 0;
    } catch (Exception $e) {
        error_log("Error checking project access by name: " . $e->getMessage());
        return false;
    }
}

/**
 * Get users assigned to a specific project
 */
function getProjectUsers($projectId) {
    try {
        $sql = "SELECT u.id, u.username, u.full_name, u.role, u.email, u.phone,
                       up.assigned_at, up.assigned_by,
                       assigned_by_user.full_name as assigned_by_name
                FROM user_projects up
                JOIN users u ON up.user_id = u.id
                LEFT JOIN users assigned_by_user ON up.assigned_by = assigned_by_user.id
                WHERE up.project_id = ? AND u.status = 'active'
                ORDER BY u.full_name";
        
        return fetchAll($sql, [$projectId]);
    } catch (Exception $e) {
        error_log("Error getting project users: " . $e->getMessage());
        return [];
    }
}

/**
 * Updated function to get user-specific project filters for queries
 */
function getUserProjectFilter($userId, $userRole) {
    // Super admin and some roles can see all projects
    if (in_array($userRole, [ROLE_SUPER_ADMIN, ROLE_GENERAL_MANAGER, ROLE_HR_MANAGER, ROLE_ACCOUNTANT])) {
        return ['where' => '', 'params' => []];
    }
    
    // Site admins and project managers only see their assigned projects
    if (in_array($userRole, [ROLE_SITE_ADMIN, ROLE_PROJECT_MANAGER])) {
        $projects = getUserProjectNames($userId);
        if (empty($projects)) {
            return ['where' => 'AND 1=0', 'params' => []]; // No projects assigned
        }
        
        $placeholders = str_repeat('?,', count($projects) - 1) . '?';
        return [
            'where' => "AND site_project IN ($placeholders)",
            'params' => $projects
        ];
    }
    
    return ['where' => '', 'params' => []];
}

/**
 * Updated dashboard statistics for site admins with multiple projects
 */
function getSiteAdminStatsMultiProject($userId) {
    $projects = getUserProjectNames($userId);
    
    if (empty($projects)) {
        return [
            'pending_requests' => 0,
            'approved_requests' => 0,
            'rejected_requests' => 0,
            'total_requests' => 0,
            'total_applications' => 0
        ];
    }
    
    $placeholders = str_repeat('?,', count($projects) - 1) . '?';
    
    $sql = "SELECT 
                COUNT(CASE WHEN er.status = 'pending' THEN 1 END) as pending_requests,
                COUNT(CASE WHEN er.status IN ('approved_by_pm', 'approved_by_gm', 'approved_final') THEN 1 END) as approved_requests,
                COUNT(CASE WHEN er.status = 'rejected' THEN 1 END) as rejected_requests,
                COUNT(er.id) as total_requests
            FROM employee_requests er 
            WHERE er.site_project IN ($placeholders)";
    
    $requestStats = fetchOne($sql, $projects);
    
    $sql = "SELECT COUNT(ja.id) as total_applications
            FROM job_applications ja
            JOIN employee_requests er ON ja.request_id = er.id
            WHERE er.site_project IN ($placeholders)";
    
    $appStats = fetchOne($sql, $projects);
    
    return array_merge($requestStats, $appStats);
}

//////////////MULTIPLE PROJECTS////////////////////




?>