<?php
/**
 * User Profile Management
 * Employee Recruitment Management System
 * RR Construction (Pvt) Ltd
 */

// Page configuration
$page_title = "My Profile";
$active_menu = "profile";
$breadcrumbs = [
    ['title' => 'My Profile', 'url' => '#']
];

// Include header
require_once 'includes/header.php';

// Require login
if (!isLoggedIn()) {
    header('Location: ' . BASE_URL . 'auth/login.php');
    exit;
}

// Get current user details
$current_user = getCurrentUser();
if (!$current_user) {
    setFlashMessage('User session expired. Please login again.', 'danger');
    header('Location: ' . BASE_URL . 'auth/login.php');
    exit;
}

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        // Verify CSRF token
        if (!isset($_POST['csrf_token']) || !verifyCSRFToken($_POST['csrf_token'])) {
            throw new Exception('Invalid security token. Please try again.');
        }
        
        // Get and validate form data
        $full_name = trim($_POST['full_name'] ?? '');
        $email = trim($_POST['email'] ?? '');
        $phone = trim($_POST['phone'] ?? '');
        $site_project = trim($_POST['site_project'] ?? '');
        
        // Validation
        $errors = [];
        
        if (empty($full_name)) {
            $errors['full_name'] = 'Full name is required';
        }
        
        if (!empty($email) && !validateEmail($email)) {
            $errors['email'] = 'Please enter a valid email address';
        }
        
        if (!empty($phone) && !validatePhone($phone)) {
            $errors['phone'] = 'Please enter a valid phone number';
        }
        
        // Check if email is already taken by another user
        if (!empty($email)) {
            $emailCheck = fetchOne(
                "SELECT id FROM users WHERE email = ? AND id != ?", 
                [$email, $current_user['id']]
            );
            if ($emailCheck) {
                $errors['email'] = 'This email is already registered to another user';
            }
        }
        
        if (empty($errors)) {
            // Prepare update data
            $updateData = [
                'full_name' => $full_name,
                'email' => $email ?: null,
                'phone' => $phone ?: null,
                'updated_at' => date('Y-m-d H:i:s')
            ];
            
            // Add site_project only for super admins
            if ($current_user['role'] === ROLE_SUPER_ADMIN) {
                $updateData['site_project'] = $site_project ?: null;
            }
            
            // Update user record
            $updated = updateRecord(
                'users', 
                $updateData, 
                'id = ?', 
                [$current_user['id']]
            );
            
            if ($updated) {
                // Log the activity
                logActivity(
                    $current_user['id'], 
                    'profile_update', 
                    'users', 
                    $current_user['id'],
                    ['old_name' => $current_user['full_name']],
                    ['new_name' => $full_name]
                );
                
                // Update session data
                $_SESSION['full_name'] = $full_name;
                
                setFlashMessage('Profile updated successfully!', 'success');
                header('Location: ' . BASE_URL . 'profile.php');
                exit;
            } else {
                throw new Exception('Failed to update profile. Please try again.');
            }
        }
        
    } catch (Exception $e) {
        setFlashMessage('Error: ' . $e->getMessage(), 'danger');
    }
}

// Get updated user data
$current_user = getCurrentUser();

// Get user's activity logs (last 10 activities)
$activity_logs = fetchAll(
    "SELECT action, table_name, record_id, created_at 
     FROM system_logs 
     WHERE user_id = ? 
     ORDER BY created_at DESC 
     LIMIT 10",
    [$current_user['id']]
);

// Get user's statistics based on role
$user_stats = [];
try {
    switch ($current_user['role']) {
        case ROLE_SITE_ADMIN:
            $user_stats = fetchOne(
                "SELECT 
                    COUNT(CASE WHEN er.status = 'pending' THEN 1 END) as pending_requests,
                    COUNT(CASE WHEN er.status IN ('approved_by_pm', 'approved_by_gm', 'approved_final') THEN 1 END) as approved_requests,
                    COUNT(er.id) as total_requests,
                    COUNT(ja.id) as total_applications
                 FROM employee_requests er
                 LEFT JOIN job_applications ja ON er.id = ja.request_id
                 WHERE er.requested_by = ?",
                [$current_user['id']]
            );
            break;
            
        case ROLE_PROJECT_MANAGER:
            $user_stats = fetchOne(
                "SELECT 
                    COUNT(CASE WHEN er.status = 'pending' THEN 1 END) as pending_approvals,
                    COUNT(CASE WHEN ja.status IN ('submitted', 'under_review') THEN 1 END) as pending_applications,
                    COUNT(er.id) as total_requests_reviewed
                 FROM employee_requests er
                 LEFT JOIN job_applications ja ON er.id = ja.request_id
                 WHERE er.site_project = ?",
                [$current_user['site_project']]
            );
            break;
            
        case ROLE_HR_MANAGER:
            $user_stats = fetchOne(
                "SELECT 
                    COUNT(CASE WHEN er.status = 'approved_by_gm' THEN 1 END) as pending_request_approvals,
                    COUNT(CASE WHEN ja.status = 'approved_by_pm' THEN 1 END) as pending_application_approvals,
                    COUNT(CASE WHEN ja.status = 'hired' THEN 1 END) as total_hired
                 FROM employee_requests er
                 LEFT JOIN job_applications ja ON er.id = ja.request_id"
            );
            break;
            
        case ROLE_ACCOUNTANT:
            $user_stats = fetchOne(
                "SELECT 
                    COUNT(CASE WHEN ja.status = 'approved_by_hr' THEN 1 END) as pending_approvals,
                    COUNT(CASE WHEN ja.status = 'hired' THEN 1 END) as total_hired,
                    SUM(CASE WHEN ja.status = 'hired' THEN ja.expected_salary END) as total_salary_approved
                 FROM job_applications ja"
            );
            break;
    }
} catch (Exception $e) {
    error_log("Error fetching user stats: " . $e->getMessage());
}

// Get available projects for super admin
$available_projects = [];
if ($current_user['role'] === ROLE_SUPER_ADMIN) {
    $available_projects = fetchAll(
        "SELECT name FROM sites_projects WHERE status = 'active' ORDER BY name"
    );
}
?>

<div class="row">
    <div class="col-md-4">
        <!-- Profile Card -->
        <div class="card mb-4">
            <div class="card-header">
                <h5 class="mb-0">
                    <i class="fas fa-user-circle me-2"></i>Profile Information
                </h5>
            </div>
            <div class="card-body text-center">
                <div class="mb-3">
                    <div class="bg-primary rounded-circle d-inline-flex align-items-center justify-content-center" 
                         style="width: 80px; height: 80px; font-size: 2rem; color: white;">
                        <i class="fas fa-user"></i>
                    </div>
                </div>
                <h5><?php echo htmlspecialchars($current_user['full_name']); ?></h5>
                <p class="text-muted mb-2"><?php echo getRoleName($current_user['role']); ?></p>
                <?php if (!empty($current_user['site_project'])): ?>
                <p class="text-muted small">
                    <i class="fas fa-map-marker-alt me-1"></i>
                    <?php echo htmlspecialchars($current_user['site_project']); ?>
                </p>
                <?php endif; ?>
                <div class="mt-3">
                    <span class="badge bg-success">
                        <i class="fas fa-check-circle me-1"></i>Active
                    </span>
                </div>
            </div>
        </div>

        <!-- Account Info -->
        <div class="card mb-4">
            <div class="card-header">
                <h6 class="mb-0">
                    <i class="fas fa-info-circle me-2"></i>Account Details
                </h6>
            </div>
            <div class="card-body">
                <div class="row mb-2">
                    <div class="col-sm-5"><strong>Username:</strong></div>
                    <div class="col-sm-7"><?php echo htmlspecialchars($current_user['username']); ?></div>
                </div>
                <div class="row mb-2">
                    <div class="col-sm-5"><strong>Role:</strong></div>
                    <div class="col-sm-7"><?php echo getRoleName($current_user['role']); ?></div>
                </div>
                <div class="row mb-2">
                    <div class="col-sm-5"><strong>Status:</strong></div>
                    <div class="col-sm-7">
                        <span class="badge bg-success"><?php echo ucfirst($current_user['status']); ?></span>
                    </div>
                </div>
                <div class="row mb-2">
                    <div class="col-sm-5"><strong>Created:</strong></div>
                    <div class="col-sm-7"><?php echo formatDisplayDate($current_user['created_at']); ?></div>
                </div>
                <div class="row">
                    <div class="col-sm-5"><strong>Last Updated:</strong></div>
                    <div class="col-sm-7"><?php echo formatDisplayDate($current_user['updated_at']); ?></div>
                </div>
            </div>
        </div>

        <!-- Quick Stats -->
        <?php if (!empty($user_stats)): ?>
        <div class="card">
            <div class="card-header">
                <h6 class="mb-0">
                    <i class="fas fa-chart-bar me-2"></i>Quick Statistics
                </h6>
            </div>
            <div class="card-body">
                <?php if ($current_user['role'] === ROLE_SITE_ADMIN): ?>
                <div class="row text-center">
                    <div class="col-6">
                        <div class="p-2">
                            <h4 class="text-primary"><?php echo $user_stats['total_requests'] ?? 0; ?></h4>
                            <small class="text-muted">Total Requests</small>
                        </div>
                    </div>
                    <div class="col-6">
                        <div class="p-2">
                            <h4 class="text-info"><?php echo $user_stats['total_applications'] ?? 0; ?></h4>
                            <small class="text-muted">Applications</small>
                        </div>
                    </div>
                    <div class="col-6">
                        <div class="p-2">
                            <h4 class="text-warning"><?php echo $user_stats['pending_requests'] ?? 0; ?></h4>
                            <small class="text-muted">Pending</small>
                        </div>
                    </div>
                    <div class="col-6">
                        <div class="p-2">
                            <h4 class="text-success"><?php echo $user_stats['approved_requests'] ?? 0; ?></h4>
                            <small class="text-muted">Approved</small>
                        </div>
                    </div>
                </div>
                <?php elseif ($current_user['role'] === ROLE_ACCOUNTANT): ?>
                <div class="row text-center">
                    <div class="col-12 mb-2">
                        <h4 class="text-warning"><?php echo $user_stats['pending_approvals'] ?? 0; ?></h4>
                        <small class="text-muted">Pending Approvals</small>
                    </div>
                    <div class="col-6">
                        <div class="p-2">
                            <h4 class="text-success"><?php echo $user_stats['total_hired'] ?? 0; ?></h4>
                            <small class="text-muted">Total Hired</small>
                        </div>
                    </div>
                    <div class="col-6">
                        <div class="p-2">
                            <h4 class="text-info"><?php echo formatCurrencyAmount($user_stats['total_salary_approved'] ?? 0); ?></h4>
                            <small class="text-muted">Salary Cost</small>
                        </div>
                    </div>
                </div>
                <?php endif; ?>
            </div>
        </div>
        <?php endif; ?>
    </div>

    <div class="col-md-8">
        <!-- Edit Profile Form -->
        <div class="card mb-4">
            <div class="card-header">
                <h5 class="mb-0">
                    <i class="fas fa-edit me-2"></i>Edit Profile
                </h5>
            </div>
            <div class="card-body">
                <form method="POST" id="profileForm">
                    <input type="hidden" name="csrf_token" value="<?php echo generateCSRFToken(); ?>">
                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="full_name" class="form-label">
                                    Full Name <span class="text-danger">*</span>
                                </label>
                                <input type="text" 
                                       class="form-control <?php echo isset($errors['full_name']) ? 'is-invalid' : ''; ?>" 
                                       id="full_name" 
                                       name="full_name" 
                                       value="<?php echo htmlspecialchars($current_user['full_name'] ?? ''); ?>" 
                                       required>
                                <?php if (isset($errors['full_name'])): ?>
                                <div class="invalid-feedback"><?php echo $errors['full_name']; ?></div>
                                <?php endif; ?>
                            </div>
                        </div>
                        
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="email" class="form-label">Email Address</label>
                                <input type="email" 
                                       class="form-control <?php echo isset($errors['email']) ? 'is-invalid' : ''; ?>" 
                                       id="email" 
                                       name="email" 
                                       value="<?php echo htmlspecialchars($current_user['email'] ?? ''); ?>">
                                <?php if (isset($errors['email'])): ?>
                                <div class="invalid-feedback"><?php echo $errors['email']; ?></div>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="phone" class="form-label">Phone Number</label>
                                <input type="tel" 
                                       class="form-control <?php echo isset($errors['phone']) ? 'is-invalid' : ''; ?>" 
                                       id="phone" 
                                       name="phone" 
                                       value="<?php echo htmlspecialchars($current_user['phone'] ?? ''); ?>"
                                       data-type="phone">
                                <?php if (isset($errors['phone'])): ?>
                                <div class="invalid-feedback"><?php echo $errors['phone']; ?></div>
                                <?php endif; ?>
                            </div>
                        </div>
                        
                        <?php if ($current_user['role'] === ROLE_SUPER_ADMIN): ?>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="site_project" class="form-label">Site/Project</label>
                                <input type="text" 
                                       class="form-control" 
                                       id="site_project" 
                                       name="site_project" 
                                       value="<?php echo htmlspecialchars($current_user['site_project'] ?? ''); ?>"
                                       list="projects_list">
                                <datalist id="projects_list">
                                    <?php foreach ($available_projects as $project): ?>
                                    <option value="<?php echo htmlspecialchars($project['name']); ?>">
                                    <?php endforeach; ?>
                                </datalist>
                                <small class="text-muted">Assign user to a specific project</small>
                            </div>
                        </div>
                        <?php else: ?>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label">Site/Project</label>
                                <input type="text" 
                                       class="form-control" 
                                       value="<?php echo htmlspecialchars($current_user['site_project'] ?? 'Not Assigned'); ?>" 
                                       readonly>
                                <small class="text-muted">Project assignment can only be changed by Super Admin</small>
                            </div>
                        </div>
                        <?php endif; ?>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label">Username</label>
                                <input type="text" class="form-control" value="<?php echo htmlspecialchars($current_user['username']); ?>" readonly>
                                <small class="text-muted">Username cannot be changed</small>
                            </div>
                        </div>
                        
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label">Role</label>
                                <input type="text" class="form-control" value="<?php echo getRoleName($current_user['role']); ?>" readonly>
                                <small class="text-muted">Role can only be changed by administrators</small>
                            </div>
                        </div>
                    </div>
                    
                    <div class="d-flex justify-content-between">
                        <a href="<?php echo BASE_URL; ?>change-password.php" class="btn btn-outline-primary">
                            <i class="fas fa-key me-2"></i>Change Password
                        </a>
                        <button type="submit" class="btn btn-primary">
                            <i class="fas fa-save me-2"></i>Update Profile
                        </button>
                    </div>
                </form>
            </div>
        </div>

        <!-- Recent Activity -->
        <div class="card">
            <div class="card-header">
                <h5 class="mb-0">
                    <i class="fas fa-history me-2"></i>Recent Activity
                </h5>
            </div>
            <div class="card-body">
                <?php if (!empty($activity_logs)): ?>
                <div class="table-responsive">
                    <table class="table table-sm">
                        <thead>
                            <tr>
                                <th>Action</th>
                                <th>Target</th>
                                <th>Date</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($activity_logs as $log): ?>
                            <tr>
                                <td>
                                    <?php
                                    $actionIcons = [
                                        'login' => 'fas fa-sign-in-alt text-success',
                                        'logout' => 'fas fa-sign-out-alt text-muted',
                                        'create' => 'fas fa-plus text-primary',
                                        'update' => 'fas fa-edit text-warning',
                                        'delete' => 'fas fa-trash text-danger',
                                        'approve' => 'fas fa-check text-success',
                                        'reject' => 'fas fa-times text-danger'
                                    ];
                                    $icon = $actionIcons[$log['action']] ?? 'fas fa-circle text-info';
                                    ?>
                                    <i class="<?php echo $icon; ?> me-2"></i>
                                    <?php echo ucfirst(str_replace('_', ' ', $log['action'])); ?>
                                </td>
                                <td>
                                    <span class="badge bg-light text-dark">
                                        <?php echo ucfirst(str_replace('_', ' ', $log['table_name'])); ?>
                                        #<?php echo $log['record_id']; ?>
                                    </span>
                                </td>
                                <td>
                                    <small class="text-muted">
                                        <?php echo getRelativeTime($log['created_at']); ?>
                                    </small>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
                <?php else: ?>
                <div class="text-center text-muted py-4">
                    <i class="fas fa-history fa-2x mb-3"></i>
                    <p>No recent activity recorded</p>
                </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<script>
// Page specific JavaScript
function pageInit() {
    // Form validation
    $('#profileForm').on('submit', function(e) {
        if (!validateForm('profileForm')) {
            e.preventDefault();
            showAlert('Please correct the errors and try again.', 'warning');
        }
    });
    
    // Phone number formatting
    $('#phone').on('input', function() {
        let value = $(this).val().replace(/[^\d+\-\s()]/g, '');
        $(this).val(value);
    });
    
    // Real-time validation
    $('#email').on('blur', function() {
        const email = $(this).val().trim();
        if (email && !isValidEmail(email)) {
            $(this).addClass('is-invalid');
            if (!$(this).next('.invalid-feedback').length) {
                $(this).after('<div class="invalid-feedback">Please enter a valid email address</div>');
            }
        } else {
            $(this).removeClass('is-invalid');
            $(this).next('.invalid-feedback').remove();
        }
    });
    
    $('#phone').on('blur', function() {
        const phone = $(this).val().trim();
        if (phone && !isValidPhone(phone)) {
            $(this).addClass('is-invalid');
            if (!$(this).next('.invalid-feedback').length) {
                $(this).after('<div class="invalid-feedback">Please enter a valid phone number</div>');
            }
        } else {
            $(this).removeClass('is-invalid');
            $(this).next('.invalid-feedback').remove();
        }
    });
}
</script>

<?php require_once 'includes/footer.php'; ?>