<?php
/**
 * Excel Export
 * Employee Recruitment Management System
 * RR Construction (Pvt) Ltd
 */

$page_title = 'Export to Excel';
$active_menu = 'reports';

require_once '../includes/header.php';
requirePermission(ROLE_PROJECT_MANAGER);

$currentUser = getCurrentUser();
$errors = [];
$available_exports = [];

// Define available exports based on user role
if ($currentUser['role'] === ROLE_PROJECT_MANAGER) {
    $available_exports = [
        'project_requests' => 'Project Employee Requests',
        'project_applications' => 'Project Applications',
        'project_summary' => 'Project Summary Data'
    ];
} elseif (in_array($currentUser['role'], [ROLE_GENERAL_MANAGER, ROLE_HR_MANAGER, ROLE_ACCOUNTANT, ROLE_SUPER_ADMIN])) {
    $available_exports = [
        'all_requests' => 'All Employee Requests',
        'all_applications' => 'All Applications',
        'hiring_data' => 'Hiring Data Export',
        'project_statistics' => 'Project Statistics',
        'user_activity' => 'User Activity Log',
        'salary_analysis' => 'Salary Analysis Data'
    ];
}

// Handle Excel export request
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['export_excel'])) {
    if (!verifyCSRFToken($_POST['csrf_token'] ?? '')) {
        $errors[] = 'Invalid security token.';
    } else {
        $export_type = $_POST['export_type'] ?? '';
        $date_from = $_POST['date_from'] ?? '';
        $date_to = $_POST['date_to'] ?? '';
        $project_filter = $_POST['project_filter'] ?? '';
        $status_filter = $_POST['status_filter'] ?? '';
        
        if (empty($export_type) || !array_key_exists($export_type, $available_exports)) {
            $errors[] = 'Please select a valid export type.';
        }
        
        if (empty($errors)) {
            try {
                // Generate the Excel file
                $excel_data = generateExcelData($export_type, $date_from, $date_to, $project_filter, $status_filter, $currentUser);
                
                if ($excel_data) {
                    // Log activity
                    logActivity($currentUser['id'], 'export_excel', 'reports', 0, null, [
                        'export_type' => $export_type,
                        'date_from' => $date_from,
                        'date_to' => $date_to
                    ]);
                    
                    // Send Excel file to browser
                    $filename = $export_type . '_' . date('Y-m-d_H-i-s') . '.csv';
                    
                    header('Content-Type: text/csv');
                    header('Content-Disposition: attachment; filename="' . $filename . '"');
                    header('Cache-Control: private, must-revalidate');
                    
                    echo $excel_data;
                    exit;
                } else {
                    $errors[] = 'No data available for export.';
                }
            } catch (Exception $e) {
                error_log("Excel export error: " . $e->getMessage());
                $errors[] = 'Error generating Excel export.';
            }
        }
    }
}

// Get available projects for filter
$projects = [];
try {
    if ($currentUser['role'] === ROLE_PROJECT_MANAGER) {
        $sql = "SELECT DISTINCT site_project as name FROM employee_requests WHERE site_project = ?";
        $projects = fetchAll($sql, [$currentUser['site_project']]);
    } else {
        $sql = "SELECT DISTINCT name FROM sites_projects WHERE status = 'active' ORDER BY name";
        $projects = fetchAll($sql);
    }
} catch (Exception $e) {
    // Handle error silently
}

/**
 * Generate Excel/CSV data
 */
function generateExcelData($export_type, $date_from, $date_to, $project_filter, $status_filter, $user) {
    switch ($export_type) {
        case 'project_requests':
        case 'all_requests':
            return exportRequestsData($date_from, $date_to, $project_filter, $status_filter, $user);
            
        case 'project_applications':
        case 'all_applications':
            return exportApplicationsData($date_from, $date_to, $project_filter, $status_filter, $user);
            
        case 'hiring_data':
            return exportHiringData($date_from, $date_to, $project_filter);
            
        case 'project_statistics':
            return exportProjectStatistics($date_from, $date_to);
            
        case 'user_activity':
            return exportUserActivity($date_from, $date_to);
            
        case 'salary_analysis':
            return exportSalaryAnalysis($date_from, $date_to, $project_filter);
            
        case 'project_summary':
            return exportProjectSummary($user);
            
        default:
            return false;
    }
}

/**
 * Export Employee Requests Data
 */
function exportRequestsData($date_from, $date_to, $project_filter, $status_filter, $user) {
    $where_conditions = ['1=1'];
    $params = [];
    
    // Role-based filtering
    if ($user['role'] === ROLE_PROJECT_MANAGER) {
        $where_conditions[] = 'er.site_project = ?';
        $params[] = $user['site_project'];
    }
    
    // Date filtering
    if ($date_from) {
        $where_conditions[] = 'er.request_date >= ?';
        $params[] = $date_from;
    }
    if ($date_to) {
        $where_conditions[] = 'er.request_date <= ?';
        $params[] = $date_to;
    }
    
    // Project filtering
    if ($project_filter) {
        $where_conditions[] = 'er.site_project = ?';
        $params[] = $project_filter;
    }
    
    // Status filtering
    if ($status_filter) {
        $where_conditions[] = 'er.status = ?';
        $params[] = $status_filter;
    }
    
    $where_clause = implode(' AND ', $where_conditions);
    
    try {
        $sql = "SELECT er.request_number, er.site_project, er.request_date, er.requirement_reason,
                       er.total_employees_requested, er.status, er.urgent_requirement,
                       er.expected_start_date, er.created_at, u.full_name as requested_by,
                       COUNT(ec.id) as positions_count, COUNT(ja.id) as applications_received
                FROM employee_requests er
                LEFT JOIN users u ON er.requested_by = u.id
                LEFT JOIN employee_categories ec ON er.id = ec.request_id
                LEFT JOIN job_applications ja ON er.id = ja.request_id
                WHERE $where_clause
                GROUP BY er.id
                ORDER BY er.created_at DESC";
        
        $requests = fetchAll($sql, $params);
        
        if (empty($requests)) {
            return false;
        }
        
        // Generate CSV
        $output = fopen('php://output', 'w');
        ob_start();
        
        // CSV Headers
        $headers = [
            'Request Number', 'Project', 'Request Date', 'Reason', 'Total Employees',
            'Status', 'Urgent', 'Expected Start', 'Positions', 'Applications',
            'Requested By', 'Created Date'
        ];
        fputcsv($output, $headers);
        
        // Data rows
        foreach ($requests as $request) {
            $row = [
                $request['request_number'],
                $request['site_project'],
                $request['request_date'],
                ucwords(str_replace('_', ' ', $request['requirement_reason'])),
                $request['total_employees_requested'],
                ucwords(str_replace('_', ' ', $request['status'])),
                $request['urgent_requirement'] ? 'Yes' : 'No',
                $request['expected_start_date'] ?: 'Not specified',
                $request['positions_count'],
                $request['applications_received'],
                $request['requested_by'],
                date('Y-m-d H:i:s', strtotime($request['created_at']))
            ];
            fputcsv($output, $row);
        }
        
        fclose($output);
        return ob_get_clean();
        
    } catch (Exception $e) {
        error_log("Export requests error: " . $e->getMessage());
        return false;
    }
}

/**
 * Export Applications Data
 */
function exportApplicationsData($date_from, $date_to, $project_filter, $status_filter, $user) {
    $where_conditions = ['1=1'];
    $params = [];
    
    // Role-based filtering
    if ($user['role'] === ROLE_PROJECT_MANAGER) {
        $where_conditions[] = 'er.site_project = ?';
        $params[] = $user['site_project'];
    }
    
    // Date filtering
    if ($date_from) {
        $where_conditions[] = 'ja.submitted_at >= ?';
        $params[] = $date_from;
    }
    if ($date_to) {
        $where_conditions[] = 'ja.submitted_at <= ?';
        $params[] = $date_to;
    }
    
    // Project filtering
    if ($project_filter) {
        $where_conditions[] = 'er.site_project = ?';
        $params[] = $project_filter;
    }
    
    // Status filtering
    if ($status_filter) {
        $where_conditions[] = 'ja.status = ?';
        $params[] = $status_filter;
    }
    
    $where_clause = implode(' AND ', $where_conditions);
    
    try {
        $sql = "SELECT ja.application_number, ja.full_name_english, ja.nic_number, ja.age, ja.gender,
                       ja.phone_mobile, ja.email, ja.position_applied, ja.expected_salary,
                       ja.status, ja.submitted_at, er.request_number, er.site_project,
                       ja.high_school, ja.ol_results, ja.al_results, ja.special_skills,
                       ja.current_employer, ja.current_position, ja.current_salary,
                       u.full_name as submitted_by
                FROM job_applications ja
                LEFT JOIN employee_requests er ON ja.request_id = er.id
                LEFT JOIN users u ON ja.submitted_by = u.id
                WHERE $where_clause
                ORDER BY ja.submitted_at DESC";
        
        $applications = fetchAll($sql, $params);
        
        if (empty($applications)) {
            return false;
        }
        
        // Generate CSV
        $output = fopen('php://output', 'w');
        ob_start();
        
        // CSV Headers
        $headers = [
            'Application Number', 'Full Name', 'NIC', 'Age', 'Gender', 'Phone', 'Email',
            'Position Applied', 'Expected Salary', 'Status', 'Request Number', 'Project',
            'High School', 'O/L Results', 'A/L Results', 'Special Skills',
            'Current Employer', 'Current Position', 'Current Salary',
            'Submitted By', 'Submitted Date'
        ];
        fputcsv($output, $headers);
        
        // Data rows
        foreach ($applications as $app) {
            $row = [
                $app['application_number'],
                $app['full_name_english'],
                $app['nic_number'],
                $app['age'],
                ucfirst($app['gender']),
                $app['phone_mobile'],
                $app['email'] ?: 'Not provided',
                $app['position_applied'],
                $app['expected_salary'] ?: '0',
                ucwords(str_replace('_', ' ', $app['status'])),
                $app['request_number'],
                $app['site_project'],
                $app['high_school'] ?: 'Not specified',
                $app['ol_results'] ?: 'Not specified',
                $app['al_results'] ?: 'Not specified',
                $app['special_skills'] ?: 'None specified',
                $app['current_employer'] ?: 'Not employed',
                $app['current_position'] ?: 'N/A',
                $app['current_salary'] ?: '0',
                $app['submitted_by'],
                date('Y-m-d H:i:s', strtotime($app['submitted_at']))
            ];
            fputcsv($output, $row);
        }
        
        fclose($output);
        return ob_get_clean();
        
    } catch (Exception $e) {
        error_log("Export applications error: " . $e->getMessage());
        return false;
    }
}

/**
 * Export Hiring Data
 */
function exportHiringData($date_from, $date_to, $project_filter) {
    $where_conditions = ["ja.status = 'hired'"];
    $params = [];
    
    if ($date_from) {
        $where_conditions[] = 'ja.updated_at >= ?';
        $params[] = $date_from;
    }
    if ($date_to) {
        $where_conditions[] = 'ja.updated_at <= ?';
        $params[] = $date_to;
    }
    if ($project_filter) {
        $where_conditions[] = 'er.site_project = ?';
        $params[] = $project_filter;
    }
    
    $where_clause = implode(' AND ', $where_conditions);
    
    try {
        $sql = "SELECT ja.full_name_english, ja.position_applied, ja.expected_salary,
                       er.site_project, ja.updated_at as hired_date,
                       ja.age, ja.gender, ja.phone_mobile, ja.email
                FROM job_applications ja
                LEFT JOIN employee_requests er ON ja.request_id = er.id
                WHERE $where_clause
                ORDER BY ja.updated_at DESC";
        
        $hired = fetchAll($sql, $params);
        
        if (empty($hired)) {
            return false;
        }
        
        $output = fopen('php://output', 'w');
        ob_start();
        
        // Headers
        $headers = ['Name', 'Position', 'Salary', 'Project', 'Hired Date', 'Age', 'Gender', 'Phone', 'Email'];
        fputcsv($output, $headers);
        
        // Data
        foreach ($hired as $person) {
            $row = [
                $person['full_name_english'],
                $person['position_applied'],
                $person['expected_salary'],
                $person['site_project'],
                date('Y-m-d', strtotime($person['hired_date'])),
                $person['age'],
                ucfirst($person['gender']),
                $person['phone_mobile'],
                $person['email'] ?: 'Not provided'
            ];
            fputcsv($output, $row);
        }
        
        fclose($output);
        return ob_get_clean();
        
    } catch (Exception $e) {
        return false;
    }
}

/**
 * Export Project Statistics
 */
function exportProjectStatistics($date_from, $date_to) {
    try {
        $date_filter = '';
        $params = [];
        
        if ($date_from) {
            $date_filter .= ' AND er.created_at >= ?';
            $params[] = $date_from;
        }
        if ($date_to) {
            $date_filter .= ' AND er.created_at <= ?';
            $params[] = $date_to;
        }
        
        $sql = "SELECT sp.name as project_name, sp.code, sp.location,
                       COUNT(er.id) as total_requests,
                       SUM(er.total_employees_requested) as total_employees_requested,
                       COUNT(CASE WHEN ja.status = 'hired' THEN 1 END) as hired_count,
                       AVG(ja.expected_salary) as avg_salary
                FROM sites_projects sp
                LEFT JOIN employee_requests er ON sp.name = er.site_project $date_filter
                LEFT JOIN job_applications ja ON er.id = ja.request_id
                WHERE sp.status = 'active'
                GROUP BY sp.id
                ORDER BY total_requests DESC";
        
        $stats = fetchAll($sql, $params);
        
        if (empty($stats)) {
            return false;
        }
        
        $output = fopen('php://output', 'w');
        ob_start();
        
        $headers = ['Project Name', 'Code', 'Location', 'Total Requests', 'Employees Requested', 'Hired', 'Avg Salary'];
        fputcsv($output, $headers);
        
        foreach ($stats as $stat) {
            $row = [
                $stat['project_name'],
                $stat['code'],
                $stat['location'],
                $stat['total_requests'] ?: '0',
                $stat['total_employees_requested'] ?: '0',
                $stat['hired_count'] ?: '0',
                $stat['avg_salary'] ? number_format($stat['avg_salary'], 2) : '0'
            ];
            fputcsv($output, $row);
        }
        
        fclose($output);
        return ob_get_clean();
        
    } catch (Exception $e) {
        return false;
    }
}

/**
 * Export User Activity
 */
function exportUserActivity($date_from, $date_to) {
    $where_conditions = ['1=1'];
    $params = [];
    
    if ($date_from) {
        $where_conditions[] = 'sl.created_at >= ?';
        $params[] = $date_from;
    }
    if ($date_to) {
        $where_conditions[] = 'sl.created_at <= ?';
        $params[] = $date_to;
    }
    
    $where_clause = implode(' AND ', $where_conditions);
    
    try {
        $sql = "SELECT u.full_name, u.role, sl.action, sl.table_name, sl.created_at, sl.ip_address
                FROM system_logs sl
                LEFT JOIN users u ON sl.user_id = u.id
                WHERE $where_clause
                ORDER BY sl.created_at DESC
                LIMIT 1000";
        
        $activities = fetchAll($sql, $params);
        
        if (empty($activities)) {
            return false;
        }
        
        $output = fopen('php://output', 'w');
        ob_start();
        
        $headers = ['User', 'Role', 'Action', 'Table', 'Date Time', 'IP Address'];
        fputcsv($output, $headers);
        
        foreach ($activities as $activity) {
            $row = [
                $activity['full_name'] ?: 'Unknown',
                $activity['role'] ?: 'Unknown',
                $activity['action'],
                $activity['table_name'],
                date('Y-m-d H:i:s', strtotime($activity['created_at'])),
                $activity['ip_address']
            ];
            fputcsv($output, $row);
        }
        
        fclose($output);
        return ob_get_clean();
        
    } catch (Exception $e) {
        return false;
    }
}

/**
 * Export Salary Analysis
 */
function exportSalaryAnalysis($date_from, $date_to, $project_filter) {
    $where_conditions = ['ja.expected_salary > 0'];
    $params = [];
    
    if ($date_from) {
        $where_conditions[] = 'ja.submitted_at >= ?';
        $params[] = $date_from;
    }
    if ($date_to) {
        $where_conditions[] = 'ja.submitted_at <= ?';
        $params[] = $date_to;
    }
    if ($project_filter) {
        $where_conditions[] = 'er.site_project = ?';
        $params[] = $project_filter;
    }
    
    $where_clause = implode(' AND ', $where_conditions);
    
    try {
        $sql = "SELECT ja.position_applied, er.site_project, ja.expected_salary, ja.status,
                       ja.age, ja.gender, ja.special_skills
                FROM job_applications ja
                LEFT JOIN employee_requests er ON ja.request_id = er.id
                WHERE $where_clause
                ORDER BY ja.expected_salary DESC";
        
        $salaries = fetchAll($sql, $params);
        
        if (empty($salaries)) {
            return false;
        }
        
        $output = fopen('php://output', 'w');
        ob_start();
        
        $headers = ['Position', 'Project', 'Expected Salary', 'Status', 'Age', 'Gender', 'Skills'];
        fputcsv($output, $headers);
        
        foreach ($salaries as $salary) {
            $row = [
                $salary['position_applied'],
                $salary['site_project'],
                $salary['expected_salary'],
                ucwords(str_replace('_', ' ', $salary['status'])),
                $salary['age'],
                ucfirst($salary['gender']),
                $salary['special_skills'] ?: 'None specified'
            ];
            fputcsv($output, $row);
        }
        
        fclose($output);
        return ob_get_clean();
        
    } catch (Exception $e) {
        return false;
    }
}

/**
 * Export Project Summary
 */
function exportProjectSummary($user) {
    try {
        $sql = "SELECT 'Requests' as type, COUNT(*) as count, er.status
                FROM employee_requests er
                WHERE er.site_project = ?
                GROUP BY er.status
                UNION ALL
                SELECT 'Applications' as type, COUNT(*) as count, ja.status
                FROM job_applications ja
                LEFT JOIN employee_requests er ON ja.request_id = er.id
                WHERE er.site_project = ?
                GROUP BY ja.status";
        
        $summary = fetchAll($sql, [$user['site_project'], $user['site_project']]);
        
        if (empty($summary)) {
            return false;
        }
        
        $output = fopen('php://output', 'w');
        ob_start();
        
        $headers = ['Type', 'Status', 'Count'];
        fputcsv($output, $headers);
        
        foreach ($summary as $item) {
            $row = [
                $item['type'],
                ucwords(str_replace('_', ' ', $item['status'])),
                $item['count']
            ];
            fputcsv($output, $row);
        }
        
        fclose($output);
        return ob_get_clean();
        
    } catch (Exception $e) {
        return false;
    }
}
?>

<div class="row mb-4">
    <div class="col-12">
        <h4 class="mb-0">Export to Excel</h4>
        <small class="text-muted">Export recruitment data to Excel/CSV format</small>
    </div>
</div>

<?php if (!empty($errors)): ?>
<div class="alert alert-danger">
    <ul class="mb-0">
        <?php foreach ($errors as $error): ?>
            <li><?php echo htmlspecialchars($error); ?></li>
        <?php endforeach; ?>
    </ul>
</div>
<?php endif; ?>

<!-- Export Configuration Form -->
<div class="card">
    <div class="card-header">Export Configuration</div>
    <div class="card-body">
        <form method="POST" id="exportForm">
            <input type="hidden" name="csrf_token" value="<?php echo generateCSRFToken(); ?>">
            
            <div class="row mb-3">
                <div class="col-md-6">
                    <label class="form-label">Export Type *</label>
                    <select name="export_type" class="form-control" required>
                        <option value="">Select export type</option>
                        <?php foreach ($available_exports as $key => $name): ?>
                            <option value="<?php echo $key; ?>"><?php echo htmlspecialchars($name); ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="col-md-6">
                    <label class="form-label">Project Filter</label>
                    <select name="project_filter" class="form-control">
                        <option value="">All Projects</option>
                        <?php foreach ($projects as $project): ?>
                            <option value="<?php echo htmlspecialchars($project['name']); ?>">
                                <?php echo htmlspecialchars($project['name']); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
            </div>
            
            <div class="row mb-3">
                <div class="col-md-6">
                    <label class="form-label">Date From</label>
                    <input type="date" name="date_from" class="form-control">
                </div>
                <div class="col-md-6">
                    <label class="form-label">Date To</label>
                    <input type="date" name="date_to" class="form-control">
                </div>
            </div>
            
            <div class="row mb-3">
                <div class="col-md-6">
                    <label class="form-label">Status Filter</label>
                    <select name="status_filter" class="form-control">
                        <option value="">All Statuses</option>
                        <option value="pending">Pending</option>
                        <option value="approved_by_pm">Approved by PM</option>
                        <option value="approved_by_gm">Approved by GM</option>
                        <option value="approved_final">Final Approval</option>
                        <option value="rejected">Rejected</option>
                        <option value="hired">Hired</option>
                    </select>
                </div>
            </div>
            
            <div class="text-end">
                <button type="submit" name="export_excel" class="btn btn-success">
                    <i class="fas fa-file-excel"></i> Export to Excel/CSV
                </button>
            </div>
        </form>
    </div>
</div>

<!-- Export Information -->
<div class="card mt-3">
    <div class="card-header">Export Information</div>
    <div class="card-body">
        <div class="row">
            <div class="col-md-6">
                <h6 class="text-primary mb-3">File Format</h6>
                <ul class="list-unstyled">
                    <li><i class="fas fa-check text-success me-2"></i>CSV format (Excel compatible)</li>
                    <li><i class="fas fa-check text-success me-2"></i>UTF-8 encoding</li>
                    <li><i class="fas fa-check text-success me-2"></i>Proper headers included</li>
                    <li><i class="fas fa-check text-success me-2"></i>Date formats standardized</li>
                </ul>
            </div>
            <div class="col-md-6">
                <h6 class="text-primary mb-3">Data Included</h6>
                <ul class="list-unstyled">
                    <li><i class="fas fa-info-circle text-info me-2"></i>All visible columns</li>
                    <li><i class="fas fa-info-circle text-info me-2"></i>Filtered data only</li>
                    <li><i class="fas fa-info-circle text-info me-2"></i>Formatted for analysis</li>
                    <li><i class="fas fa-info-circle text-info me-2"></i>Ready for Excel/Sheets</li>
                </ul>
            </div>
        </div>
        
        <div class="alert alert-info mt-3 mb-0">
            <i class="fas fa-lightbulb me-2"></i>
            <strong>Tip:</strong> Use date filters to limit export size and focus on specific time periods. 
            Large datasets may take a moment to process.
        </div>
    </div>
</div>

<!-- Available Exports -->
<div class="card mt-3">
    <div class="card-header">Available Exports</div>
    <div class="card-body">
        <div class="row">
            <?php foreach ($available_exports as $key => $name): ?>
            <div class="col-md-6 mb-3">
                <div class="border p-3 rounded">
                    <h6 class="text-success mb-2">
                        <i class="fas fa-file-excel me-2"></i>
                        <?php echo htmlspecialchars($name); ?>
                    </h6>
                    <small class="text-muted">
                        <?php
                        switch ($key) {
                            case 'project_requests':
                                echo 'Employee requests for your project with detailed information and status.';
                                break;
                            case 'all_requests':
                                echo 'Complete database of all employee requests across all projects.';
                                break;
                            case 'project_applications':
                                echo 'Job applications for your project including applicant details and qualifications.';
                                break;
                            case 'all_applications':
                                echo 'Comprehensive list of all job applications with personal and professional data.';
                                break;
                            case 'hiring_data':
                                echo 'Successfully hired employees with contact information and job details.';
                                break;
                            case 'project_statistics':
                                echo 'Statistical summary of recruitment activities across all projects.';
                                break;
                            case 'user_activity':
                                echo 'System activity log showing user actions and timestamps.';
                                break;
                            case 'salary_analysis':
                                echo 'Salary expectations and analysis data for compensation planning.';
                                break;
                            case 'project_summary':
                                echo 'Summary statistics for your specific project activities.';
                                break;
                        }
                        ?>
                    </small>
                </div>
            </div>
            <?php endforeach; ?>
        </div>
    </div>
</div>

<script>
document.getElementById('exportForm').addEventListener('submit', function() {
    const button = this.querySelector('button[type="submit"]');
    button.disabled = true;
    button.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Exporting...';
    
    // Re-enable button after 10 seconds (in case of error)
    setTimeout(() => {
        button.disabled = false;
        button.innerHTML = '<i class="fas fa-file-excel"></i> Export to Excel/CSV';
    }, 10000);
});
</script>

<?php require_once '../includes/footer.php'; ?>