<?php
/**
 * PDF Report Generator
 * Employee Recruitment Management System
 * RR Construction (Pvt) Ltd
 */

$page_title = 'Generate PDF Reports';
$active_menu = 'reports';

require_once '../includes/header.php';
requirePermission(ROLE_PROJECT_MANAGER);

$currentUser = getCurrentUser();
$errors = [];
$available_reports = [];

// Define available reports based on user role
if ($currentUser['role'] === ROLE_PROJECT_MANAGER) {
    $available_reports = [
        'project_requests' => 'Project Employee Requests',
        'project_applications' => 'Project Applications Summary'
    ];
} elseif (in_array($currentUser['role'], [ROLE_GENERAL_MANAGER, ROLE_HR_MANAGER, ROLE_ACCOUNTANT, ROLE_SUPER_ADMIN])) {
    $available_reports = [
        'all_requests' => 'All Employee Requests',
        'all_applications' => 'All Applications Summary',
        'hiring_summary' => 'Hiring Summary Report',
        'project_comparison' => 'Project Comparison Report',
        'monthly_summary' => 'Monthly Activity Summary'
    ];
}

// Handle PDF generation request
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['generate_pdf'])) {
    if (!verifyCSRFToken($_POST['csrf_token'] ?? '')) {
        $errors[] = 'Invalid security token.';
    } else {
        $report_type = $_POST['report_type'] ?? '';
        $date_from = $_POST['date_from'] ?? '';
        $date_to = $_POST['date_to'] ?? '';
        $project_filter = $_POST['project_filter'] ?? '';
        
        if (empty($report_type) || !array_key_exists($report_type, $available_reports)) {
            $errors[] = 'Please select a valid report type.';
        }
        
        if (empty($errors)) {
            try {
                // Generate the PDF
                $pdf_content = generatePDFReport($report_type, $date_from, $date_to, $project_filter, $currentUser);
                
                if ($pdf_content) {
                    // Log activity
                    logActivity($currentUser['id'], 'generate_pdf', 'reports', 0, null, [
                        'report_type' => $report_type,
                        'date_from' => $date_from,
                        'date_to' => $date_to
                    ]);
                    
                    // Send PDF to browser
                    $filename = $report_type . '_' . date('Y-m-d_H-i-s') . '.pdf';
                    
                    header('Content-Type: application/pdf');
                    header('Content-Disposition: attachment; filename="' . $filename . '"');
                    header('Content-Length: ' . strlen($pdf_content));
                    header('Cache-Control: private, must-revalidate');
                    
                    echo $pdf_content;
                    exit;
                } else {
                    $errors[] = 'Failed to generate PDF report.';
                }
            } catch (Exception $e) {
                error_log("PDF generation error: " . $e->getMessage());
                $errors[] = 'Error generating PDF report.';
            }
        }
    }
}

// Get available projects for filter
$projects = [];
try {
    if ($currentUser['role'] === ROLE_PROJECT_MANAGER) {
        $sql = "SELECT DISTINCT site_project as name FROM employee_requests WHERE site_project = ?";
        $projects = fetchAll($sql, [$currentUser['site_project']]);
    } else {
        $sql = "SELECT DISTINCT name FROM sites_projects WHERE status = 'active' ORDER BY name";
        $projects = fetchAll($sql);
    }
} catch (Exception $e) {
    // Handle error silently
}

/**
 * Generate PDF Report Content
 */
function generatePDFReport($report_type, $date_from, $date_to, $project_filter, $user) {
    // Simple HTML to PDF conversion (you can replace with TCPDF or similar library)
    
    $html = generateReportHTML($report_type, $date_from, $date_to, $project_filter, $user);
    
    // For this example, we'll return HTML content
    // In production, use a proper PDF library like TCPDF or mPDF
    return $html;
}

/**
 * Generate HTML content for reports
 */
function generateReportHTML($report_type, $date_from, $date_to, $project_filter, $user) {
    ob_start();
    
    // Report header
    echo "<html><head><title>Report</title>";
    echo "<style>
        body { font-family: Arial, sans-serif; font-size: 12px; margin: 20px; }
        .header { text-align: center; margin-bottom: 30px; border-bottom: 2px solid #dc2626; padding-bottom: 10px; }
        .company-name { font-size: 18px; font-weight: bold; color: #dc2626; }
        .report-title { font-size: 14px; margin: 10px 0; }
        .report-info { font-size: 10px; color: #666; }
        table { width: 100%; border-collapse: collapse; margin: 20px 0; }
        th, td { border: 1px solid #ddd; padding: 8px; text-align: left; }
        th { background-color: #f5f5f5; font-weight: bold; }
        .text-center { text-align: center; }
        .text-right { text-align: right; }
        .summary-box { background: #f9f9f9; padding: 15px; margin: 10px 0; border: 1px solid #ddd; }
    </style></head><body>";
    
    // Header
    echo "<div class='header'>";
    echo "<div class='company-name'>" . COMPANY_NAME . "</div>";
    echo "<div class='report-title'>" . htmlspecialchars($available_reports[$report_type]) . "</div>";
    echo "<div class='report-info'>Generated on " . date('d/m/Y H:i:s') . " by " . htmlspecialchars($user['full_name']) . "</div>";
    if ($date_from || $date_to) {
        echo "<div class='report-info'>Period: ";
        echo $date_from ? date('d/m/Y', strtotime($date_from)) : 'Start';
        echo " to ";
        echo $date_to ? date('d/m/Y', strtotime($date_to)) : 'Present';
        echo "</div>";
    }
    echo "</div>";
    
    // Generate report content based on type
    switch ($report_type) {
        case 'project_requests':
        case 'all_requests':
            generateRequestsReport($date_from, $date_to, $project_filter, $user);
            break;
            
        case 'project_applications':
        case 'all_applications':
            generateApplicationsReport($date_from, $date_to, $project_filter, $user);
            break;
            
        case 'hiring_summary':
            generateHiringSummaryReport($date_from, $date_to, $project_filter);
            break;
            
        case 'project_comparison':
            generateProjectComparisonReport($date_from, $date_to);
            break;
            
        case 'monthly_summary':
            generateMonthlySummaryReport($date_from, $date_to);
            break;
    }
    
    echo "</body></html>";
    
    return ob_get_clean();
}

/**
 * Generate Employee Requests Report
 */
function generateRequestsReport($date_from, $date_to, $project_filter, $user) {
    $where_conditions = ['1=1'];
    $params = [];
    
    // Role-based filtering
    if ($user['role'] === ROLE_PROJECT_MANAGER) {
        $where_conditions[] = 'er.site_project = ?';
        $params[] = $user['site_project'];
    }
    
    // Date filtering
    if ($date_from) {
        $where_conditions[] = 'er.request_date >= ?';
        $params[] = $date_from;
    }
    if ($date_to) {
        $where_conditions[] = 'er.request_date <= ?';
        $params[] = $date_to;
    }
    
    // Project filtering
    if ($project_filter) {
        $where_conditions[] = 'er.site_project = ?';
        $params[] = $project_filter;
    }
    
    $where_clause = implode(' AND ', $where_conditions);
    
    try {
        $sql = "SELECT er.*, u.full_name as requested_by_name,
                       COUNT(ec.id) as categories_count,
                       COUNT(ja.id) as applications_count
                FROM employee_requests er
                LEFT JOIN users u ON er.requested_by = u.id
                LEFT JOIN employee_categories ec ON er.id = ec.request_id
                LEFT JOIN job_applications ja ON er.id = ja.request_id
                WHERE $where_clause
                GROUP BY er.id
                ORDER BY er.created_at DESC";
        
        $requests = fetchAll($sql, $params);
        
        // Summary
        $total_requests = count($requests);
        $total_employees = array_sum(array_column($requests, 'total_employees_requested'));
        $pending_count = count(array_filter($requests, function($r) { return $r['status'] === 'pending'; }));
        $approved_count = count(array_filter($requests, function($r) { return in_array($r['status'], ['approved_by_pm', 'approved_by_gm', 'approved_final']); }));
        
        echo "<div class='summary-box'>";
        echo "<h3>Summary</h3>";
        echo "<p><strong>Total Requests:</strong> $total_requests</p>";
        echo "<p><strong>Total Employees Requested:</strong> $total_employees</p>";
        echo "<p><strong>Pending Approval:</strong> $pending_count</p>";
        echo "<p><strong>Approved:</strong> $approved_count</p>";
        echo "</div>";
        
        // Detailed table
        echo "<h3>Detailed Report</h3>";
        echo "<table>";
        echo "<tr>";
        echo "<th>Request #</th>";
        echo "<th>Project</th>";
        echo "<th>Requested By</th>";
        echo "<th>Date</th>";
        echo "<th>Employees</th>";
        echo "<th>Status</th>";
        echo "<th>Applications</th>";
        echo "</tr>";
        
        foreach ($requests as $request) {
            echo "<tr>";
            echo "<td>" . htmlspecialchars($request['request_number']) . "</td>";
            echo "<td>" . htmlspecialchars($request['site_project']) . "</td>";
            echo "<td>" . htmlspecialchars($request['requested_by_name']) . "</td>";
            echo "<td>" . date('d/m/Y', strtotime($request['request_date'])) . "</td>";
            echo "<td class='text-center'>" . $request['total_employees_requested'] . "</td>";
            echo "<td>" . ucwords(str_replace('_', ' ', $request['status'])) . "</td>";
            echo "<td class='text-center'>" . $request['applications_count'] . "</td>";
            echo "</tr>";
        }
        
        echo "</table>";
        
    } catch (Exception $e) {
        echo "<p>Error generating requests report: " . htmlspecialchars($e->getMessage()) . "</p>";
    }
}

/**
 * Generate Applications Report
 */
function generateApplicationsReport($date_from, $date_to, $project_filter, $user) {
    $where_conditions = ['1=1'];
    $params = [];
    
    // Role-based filtering
    if ($user['role'] === ROLE_PROJECT_MANAGER) {
        $where_conditions[] = 'er.site_project = ?';
        $params[] = $user['site_project'];
    }
    
    // Date filtering
    if ($date_from) {
        $where_conditions[] = 'ja.submitted_at >= ?';
        $params[] = $date_from;
    }
    if ($date_to) {
        $where_conditions[] = 'ja.submitted_at <= ?';
        $params[] = $date_to;
    }
    
    // Project filtering
    if ($project_filter) {
        $where_conditions[] = 'er.site_project = ?';
        $params[] = $project_filter;
    }
    
    $where_clause = implode(' AND ', $where_conditions);
    
    try {
        $sql = "SELECT ja.*, er.request_number, er.site_project, u.full_name as submitted_by_name
                FROM job_applications ja
                LEFT JOIN employee_requests er ON ja.request_id = er.id
                LEFT JOIN users u ON ja.submitted_by = u.id
                WHERE $where_clause
                ORDER BY ja.submitted_at DESC";
        
        $applications = fetchAll($sql, $params);
        
        // Summary
        $total_applications = count($applications);
        $hired_count = count(array_filter($applications, function($a) { return $a['status'] === 'hired'; }));
        $pending_count = count(array_filter($applications, function($a) { return in_array($a['status'], ['submitted', 'under_review', 'approved_by_pm', 'approved_by_hr']); }));
        $avg_salary = 0;
        
        $salaries = array_filter(array_column($applications, 'expected_salary'));
        if (!empty($salaries)) {
            $avg_salary = array_sum($salaries) / count($salaries);
        }
        
        echo "<div class='summary-box'>";
        echo "<h3>Summary</h3>";
        echo "<p><strong>Total Applications:</strong> $total_applications</p>";
        echo "<p><strong>Hired:</strong> $hired_count</p>";
        echo "<p><strong>Pending:</strong> $pending_count</p>";
        echo "<p><strong>Average Expected Salary:</strong> Rs. " . number_format($avg_salary, 2) . "</p>";
        echo "</div>";
        
        // Detailed table
        echo "<h3>Detailed Report</h3>";
        echo "<table>";
        echo "<tr>";
        echo "<th>Application #</th>";
        echo "<th>Name</th>";
        echo "<th>Position</th>";
        echo "<th>Project</th>";
        echo "<th>Expected Salary</th>";
        echo "<th>Status</th>";
        echo "<th>Submitted</th>";
        echo "</tr>";
        
        foreach ($applications as $app) {
            echo "<tr>";
            echo "<td>" . htmlspecialchars($app['application_number']) . "</td>";
            echo "<td>" . htmlspecialchars($app['full_name_english']) . "</td>";
            echo "<td>" . htmlspecialchars($app['position_applied']) . "</td>";
            echo "<td>" . htmlspecialchars($app['site_project']) . "</td>";
            echo "<td class='text-right'>Rs. " . number_format($app['expected_salary'], 2) . "</td>";
            echo "<td>" . ucwords(str_replace('_', ' ', $app['status'])) . "</td>";
            echo "<td>" . date('d/m/Y', strtotime($app['submitted_at'])) . "</td>";
            echo "</tr>";
        }
        
        echo "</table>";
        
    } catch (Exception $e) {
        echo "<p>Error generating applications report: " . htmlspecialchars($e->getMessage()) . "</p>";
    }
}

/**
 * Generate Hiring Summary Report
 */
function generateHiringSummaryReport($date_from, $date_to, $project_filter) {
    echo "<p>Hiring Summary Report - Advanced reporting feature coming soon.</p>";
}

/**
 * Generate Project Comparison Report
 */
function generateProjectComparisonReport($date_from, $date_to) {
    echo "<p>Project Comparison Report - Advanced reporting feature coming soon.</p>";
}

/**
 * Generate Monthly Summary Report
 */
function generateMonthlySummaryReport($date_from, $date_to) {
    echo "<p>Monthly Summary Report - Advanced reporting feature coming soon.</p>";
}
?>

<div class="row mb-4">
    <div class="col-12">
        <h4 class="mb-0">Generate PDF Reports</h4>
        <small class="text-muted">Create detailed PDF reports for recruitment data</small>
    </div>
</div>

<?php if (!empty($errors)): ?>
<div class="alert alert-danger">
    <ul class="mb-0">
        <?php foreach ($errors as $error): ?>
            <li><?php echo htmlspecialchars($error); ?></li>
        <?php endforeach; ?>
    </ul>
</div>
<?php endif; ?>

<!-- Report Generation Form -->
<div class="card">
    <div class="card-header">Report Configuration</div>
    <div class="card-body">
        <form method="POST" id="reportForm">
            <input type="hidden" name="csrf_token" value="<?php echo generateCSRFToken(); ?>">
            
            <div class="row mb-3">
                <div class="col-md-6">
                    <label class="form-label">Report Type *</label>
                    <select name="report_type" class="form-control" required>
                        <option value="">Select report type</option>
                        <?php foreach ($available_reports as $key => $name): ?>
                            <option value="<?php echo $key; ?>"><?php echo htmlspecialchars($name); ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="col-md-6">
                    <label class="form-label">Project Filter</label>
                    <select name="project_filter" class="form-control">
                        <option value="">All Projects</option>
                        <?php foreach ($projects as $project): ?>
                            <option value="<?php echo htmlspecialchars($project['name']); ?>">
                                <?php echo htmlspecialchars($project['name']); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
            </div>
            
            <div class="row mb-3">
                <div class="col-md-6">
                    <label class="form-label">Date From</label>
                    <input type="date" name="date_from" class="form-control">
                </div>
                <div class="col-md-6">
                    <label class="form-label">Date To</label>
                    <input type="date" name="date_to" class="form-control">
                </div>
            </div>
            
            <div class="text-end">
                <button type="submit" name="generate_pdf" class="btn btn-primary">
                    <i class="fas fa-file-pdf"></i> Generate PDF Report
                </button>
            </div>
        </form>
    </div>
</div>

<!-- Available Reports Info -->
<div class="card mt-3">
    <div class="card-header">Available Reports</div>
    <div class="card-body">
        <div class="row">
            <?php foreach ($available_reports as $key => $name): ?>
            <div class="col-md-6 mb-3">
                <div class="border p-3 rounded">
                    <h6 class="text-primary mb-2"><?php echo htmlspecialchars($name); ?></h6>
                    <small class="text-muted">
                        <?php
                        switch ($key) {
                            case 'project_requests':
                                echo 'Employee requests for your project with status and progress details.';
                                break;
                            case 'all_requests':
                                echo 'Comprehensive view of all employee requests across all projects.';
                                break;
                            case 'project_applications':
                                echo 'Job applications for your project including applicant details.';
                                break;
                            case 'all_applications':
                                echo 'Complete list of all job applications with hiring status.';
                                break;
                            case 'hiring_summary':
                                echo 'Summary of hiring activities with statistics and trends.';
                                break;
                            case 'project_comparison':
                                echo 'Comparative analysis of recruitment across different projects.';
                                break;
                            case 'monthly_summary':
                                echo 'Monthly breakdown of recruitment activities and performance.';
                                break;
                        }
                        ?>
                    </small>
                </div>
            </div>
            <?php endforeach; ?>
        </div>
    </div>
</div>

<script>
document.getElementById('reportForm').addEventListener('submit', function() {
    const button = this.querySelector('button[type="submit"]');
    button.disabled = true;
    button.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Generating...';
    
    // Re-enable button after 5 seconds (in case of error)
    setTimeout(() => {
        button.disabled = false;
        button.innerHTML = '<i class="fas fa-file-pdf"></i> Generate PDF Report';
    }, 5000);
});
</script>

<?php require_once '../includes/footer.php'; ?>