<?php
/**
 * Approve Employee Requests
 * Employee Recruitment Management System
 * RR Construction (Pvt) Ltd
 */

$page_title = 'Approve Employee Requests';
$active_menu = 'requests';

// Include header (this will handle all authentication and setup)
require_once '../includes/header.php';

// Check permissions
requirePermission(ROLE_PROJECT_MANAGER);

$currentUser = getCurrentUser();
$success = false;
$errors = [];

// Process approval/rejection
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!verifyCSRFToken($_POST['csrf_token'] ?? '')) {
        $errors[] = 'Invalid security token.';
    } else {
        $request_id = (int)($_POST['request_id'] ?? 0);
        $action = $_POST['action'] ?? '';
        $comments = cleanInput($_POST['comments'] ?? '');
        
        if ($request_id && in_array($action, ['approve', 'reject'])) {
            try {
                // Get request details
                $request = fetchOne("SELECT * FROM employee_requests WHERE id = ?", [$request_id]);
                
                if ($request && canUserApproveRequest($request, $currentUser['role'])) {
                    beginTransaction();
                    
                    $new_status = '';
                    
                    // Determine new status based on user role and current status
                    if ($action === 'approve') {
                        switch ($currentUser['role']) {
                            case ROLE_PROJECT_MANAGER:
                                if ($request['status'] === 'pending') {
                                    $new_status = 'approved_by_pm';
                                }
                                break;
                            case ROLE_GENERAL_MANAGER:
                                if ($request['status'] === 'approved_by_pm') {
                                    $new_status = 'approved_by_gm';
                                }
                                break;
                            case ROLE_HR_MANAGER:
                                if ($request['status'] === 'approved_by_gm') {
                                    $new_status = 'approved_final';
                                }
                                break;
                        }
                    } else {
                        // Rejection
                        $new_status = 'rejected';
                    }
                    
                    if ($new_status) {
                        // Update request status
                        $update_data = [
                            'status' => $new_status
                        ];
                        
                        if ($action === 'reject') {
                            $update_data['rejection_reason'] = $comments;
                        }
                        
                        // Use proper parameter format for updateRecord function
                        updateRecord('employee_requests', $update_data, 'id = :id', ['id' => $request_id]);
                        
                        // Insert approval log
                        $approval_data = [
                            'request_id' => $request_id,
                            'approved_by' => $currentUser['id'],
                            'approval_level' => $currentUser['role'],
                            'action' => $action === 'approve' ? 'approved' : 'rejected',
                            'comments' => $comments,
                            'approval_date' => date('Y-m-d')
                        ];
                        insertRecord('approval_logs', $approval_data);
                        
                        // Log activity
                        logActivity($currentUser['id'], $action, 'employee_requests', $request_id);
                        
                        // Send notification
                        notifyRequestApproval($request_id, $new_status, $comments);
                        
                        commit();
                        $success = true;
                        setFlashMessage("Request " . ($action === 'approve' ? 'approved' : 'rejected') . " successfully!", 'success');
                        
                        // Redirect to avoid resubmission
                        header('Location: ' . $_SERVER['PHP_SELF']);
                        exit();
                        
                    } else {
                        $errors[] = 'Invalid approval action for current request status.';
                    }
                    
                } else {
                    $errors[] = 'Request not found or you do not have permission to approve this request.';
                }
            } catch (Exception $e) {
                rollback();
                error_log("Approval error: " . $e->getMessage());
                $errors[] = 'Failed to process approval. Please try again.';
            }
        } else {
            $errors[] = 'Invalid request data.';
        }
    }
}

/**
 * Check if user can approve a specific request
 */
function canUserApproveRequest($request, $userRole) {
    switch ($userRole) {
        case ROLE_PROJECT_MANAGER:
            return $request['status'] === 'pending';
        case ROLE_GENERAL_MANAGER:
            return $request['status'] === 'approved_by_pm';
        case ROLE_HR_MANAGER:
            return $request['status'] === 'approved_by_gm';
        default:
            return false;
    }
}

/**
 * Get pending requests for user role
 */
function getPendingRequestsForUser($userRole) {
    $status_condition = '';
    switch ($userRole) {
        case ROLE_PROJECT_MANAGER:
            $status_condition = "er.status = 'pending'";
            break;
        case ROLE_GENERAL_MANAGER:
            $status_condition = "er.status = 'approved_by_pm'";
            break;
        case ROLE_HR_MANAGER:
            $status_condition = "er.status = 'approved_by_gm'";
            break;
        default:
            return [];
    }
    
    $sql = "SELECT er.*, u.full_name as requested_by_name, u.site_project
            FROM employee_requests er 
            JOIN users u ON er.requested_by = u.id 
            WHERE $status_condition AND u.status = 'active'
            ORDER BY er.created_at ASC";
    
    return fetchAll($sql);
}

// Get pending requests for current user role
$pending_requests = getPendingRequestsForUser($currentUser['role']);

// Set breadcrumbs
$breadcrumbs = [
    ['title' => 'Employee Requests', 'url' => BASE_URL . 'requests/view_requests.php'],
    ['title' => 'Pending Approvals', 'url' => '']
];

?>

<!-- Page Content -->
<div class="row">
    <div class="col-12">
        <div class="card">
            <div class="card-header d-flex justify-content-between align-items-center">
                <h5 class="mb-0">
                    <i class="fas fa-check-circle me-2"></i>
                    Pending Requests for Approval
                </h5>
                <div>
                    <span class="badge bg-warning text-dark me-2">
                        <?php echo count($pending_requests); ?> Pending
                    </span>
                    <span class="text-muted small">
                        Role: <?php echo getRoleName($currentUser['role']); ?>
                    </span>
                </div>
            </div>
            <div class="card-body">
                
                <?php if (!empty($errors)): ?>
                    <div class="alert alert-danger">
                        <i class="fas fa-exclamation-circle me-2"></i>
                        <ul class="mb-0">
                            <?php foreach ($errors as $error): ?>
                                <li><?php echo htmlspecialchars($error); ?></li>
                            <?php endforeach; ?>
                        </ul>
                    </div>
                <?php endif; ?>

                <?php if (empty($pending_requests)): ?>
                    <div class="text-center py-5">
                        <i class="fas fa-check-circle text-success" style="font-size: 4rem; opacity: 0.3;"></i>
                        <h4 class="mt-3 text-muted">No Pending Requests</h4>
                        <p class="text-muted">
                            There are no employee requests pending for your approval at this time.
                        </p>
                        <a href="<?php echo BASE_URL; ?>requests/view_requests.php" class="btn btn-outline-primary">
                            <i class="fas fa-list me-2"></i>View All Requests
                        </a>
                    </div>
                <?php else: ?>
                    <div class="table-responsive">
                        <table class="table table-hover">
                            <thead>
                                <tr>
                                    <th>Request #</th>
                                    <th>Site/Project</th>
                                    <th>Requested By</th>
                                    <th>Date</th>
                                    <th>Reason</th>
                                    <th>Employees</th>
                                    <th>Priority</th>
                                    <th>Status</th>
                                    <th width="200">Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($pending_requests as $request): ?>
                                    <tr>
                                        <td>
                                            <strong><?php echo htmlspecialchars($request['request_number']); ?></strong>
                                        </td>
                                        <td>
                                            <div class="fw-bold"><?php echo htmlspecialchars($request['site_project']); ?></div>
                                        </td>
                                        <td><?php echo htmlspecialchars($request['requested_by_name']); ?></td>
                                        <td><?php echo formatDisplayDate($request['request_date']); ?></td>
                                        <td>
                                            <span class="badge <?php echo $request['requirement_reason'] === 'replacement' ? 'bg-info' : 'bg-primary'; ?>">
                                                <?php echo ucfirst(str_replace('_', ' ', $request['requirement_reason'])); ?>
                                            </span>
                                        </td>
                                        <td>
                                            <span class="badge bg-secondary"><?php echo $request['total_employees_requested']; ?></span>
                                        </td>
                                        <td>
                                            <?php if ($request['urgent_requirement']): ?>
                                                <span class="badge bg-danger">
                                                    <i class="fas fa-exclamation-triangle me-1"></i>URGENT
                                                </span>
                                            <?php else: ?>
                                                <span class="text-muted">Normal</span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <?php echo getStatusBadge($request['status'], 'request'); ?>
                                        </td>
                                        <td>
                                            <div class="btn-group" role="group">
                                                <button type="button" 
                                                        class="btn btn-success btn-sm" 
                                                        onclick="openApprovalModal(<?php echo $request['id']; ?>, 'approve')"
                                                        title="Approve Request">
                                                    <i class="fas fa-check"></i> Approve
                                                </button>
                                                <button type="button" 
                                                        class="btn btn-outline-danger btn-sm" 
                                                        onclick="openApprovalModal(<?php echo $request['id']; ?>, 'reject')"
                                                        title="Reject Request">
                                                    <i class="fas fa-times"></i> Reject
                                                </button>
                                                <a href="view_request.php?id=<?php echo $request['id']; ?>" 
                                                   class="btn btn-outline-secondary btn-sm"
                                                   title="View Details">
                                                    <i class="fas fa-eye"></i>
                                                </a>
                                                <button type="button" 
                                                        class="btn btn-outline-info btn-sm" 
                                                        onclick="viewRequestDetails(<?php echo $request['id']; ?>)"
                                                        title="Quick View">
                                                    <i class="fas fa-search"></i>
                                                </button>
                                            </div>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php endif; ?>
                
                <div class="mt-3">
                    <a href="<?php echo BASE_URL; ?>requests/view_requests.php" class="btn btn-outline-secondary">
                        <i class="fas fa-arrow-left me-2"></i>Back to All Requests
                    </a>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Approval Modal -->
<div class="modal fade" id="approvalModal" tabindex="-1" aria-labelledby="approvalModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="approvalModalLabel">
                    <i class="fas fa-check-circle me-2"></i>
                    Approve Request
                </h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <form method="POST" id="approvalForm">
                <div class="modal-body">
                    <input type="hidden" name="csrf_token" value="<?php echo generateCSRFToken(); ?>">
                    <input type="hidden" id="requestId" name="request_id" value="">
                    <input type="hidden" id="action" name="action" value="">
                    
                    <div class="mb-3">
                        <label for="comments" class="form-label">Comments:</label>
                        <textarea class="form-control" 
                                  id="comments" 
                                  name="comments" 
                                  rows="4"
                                  placeholder="Enter your comments (optional for approval, required for rejection)"></textarea>
                        <div class="form-text" id="commentsHelp">
                            Optional for approvals, required for rejections.
                        </div>
                    </div>
                    
                    <div class="alert alert-info" id="approvalInfo">
                        <i class="fas fa-info-circle me-2"></i>
                        <span id="approvalInfoText">This action will approve the request and move it to the next approval level.</span>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
                        <i class="fas fa-times me-2"></i>Cancel
                    </button>
                    <button type="submit" class="btn btn-success" id="submitBtn">
                        <i class="fas fa-check me-2"></i>Approve
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Request Details Modal (Quick View) -->
<div class="modal fade" id="requestDetailsModal" tabindex="-1" aria-labelledby="requestDetailsModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="requestDetailsModalLabel">
                    <i class="fas fa-file-alt me-2"></i>
                    Request Details
                </h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body" id="requestDetailsContent">
                <div class="text-center py-4">
                    <div class="spinner-border text-primary" role="status">
                        <span class="visually-hidden">Loading...</span>
                    </div>
                    <p class="mt-2 text-muted">Loading request details...</p>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
                    <i class="fas fa-times me-2"></i>Close
                </button>
                <a href="#" id="viewFullRequestBtn" class="btn btn-outline-primary" target="_blank">
                    <i class="fas fa-external-link-alt me-2"></i>View Full Page
                </a>
            </div>
        </div>
    </div>
</div>

<script>
function openApprovalModal(requestId, action) {
    document.getElementById('requestId').value = requestId;
    document.getElementById('action').value = action;
    
    const modal = document.getElementById('approvalModal');
    const title = modal.querySelector('#approvalModalLabel');
    const submitBtn = modal.querySelector('#submitBtn');
    const infoText = modal.querySelector('#approvalInfoText');
    const commentsField = modal.querySelector('#comments');
    const commentsHelp = modal.querySelector('#commentsHelp');
    
    if (action === 'approve') {
        title.innerHTML = '<i class="fas fa-check-circle me-2"></i>Approve Request #' + requestId;
        submitBtn.innerHTML = '<i class="fas fa-check me-2"></i>Approve';
        submitBtn.className = 'btn btn-success';
        infoText.textContent = 'This action will approve the request and move it to the next approval level.';
        commentsField.placeholder = 'Enter your approval comments (optional)';
        commentsHelp.textContent = 'Optional comments for approval.';
        commentsField.removeAttribute('required');
    } else {
        title.innerHTML = '<i class="fas fa-times-circle me-2"></i>Reject Request #' + requestId;
        submitBtn.innerHTML = '<i class="fas fa-times me-2"></i>Reject';
        submitBtn.className = 'btn btn-danger';
        infoText.textContent = 'This action will reject the request. Please provide a reason for rejection.';
        commentsField.placeholder = 'Enter reason for rejection (required)';
        commentsHelp.textContent = 'Please provide a detailed reason for rejection.';
        commentsField.setAttribute('required', 'required');
    }
    
    commentsField.value = '';
    const modalInstance = new bootstrap.Modal(modal);
    modalInstance.show();
}

function viewRequestDetails(requestId) {
    // Reset modal content
    document.getElementById('requestDetailsContent').innerHTML = `
        <div class="text-center py-4">
            <div class="spinner-border text-primary" role="status">
                <span class="visually-hidden">Loading...</span>
            </div>
            <p class="mt-2 text-muted">Loading request details...</p>
        </div>
    `;
    
    // Update the full view link
    document.getElementById('viewFullRequestBtn').href = 'view_request.php?id=' + requestId;
    
    // Show modal
    const modal = new bootstrap.Modal(document.getElementById('requestDetailsModal'));
    modal.show();
    
    // Fetch request details using your existing AJAX endpoint
    fetch('get_request_details.php?id=' + requestId)
        .then(response => {
            if (!response.ok) {
                throw new Error('Network response was not ok');
            }
            return response.text();
        })
        .then(html => {
            document.getElementById('requestDetailsContent').innerHTML = html;
        })
        .catch(error => {
            console.error('Error loading request details:', error);
            document.getElementById('requestDetailsContent').innerHTML = `
                <div class="alert alert-danger">
                    <i class="fas fa-exclamation-triangle me-2"></i>
                    Error loading request details. Please try again or use the "View Full Page" link.
                </div>
            `;
        });
}

// Form validation
document.getElementById('approvalForm').addEventListener('submit', function(e) {
    const action = document.getElementById('action').value;
    const comments = document.getElementById('comments').value.trim();
    
    if (action === 'reject' && comments === '') {
        e.preventDefault();
        showAlert('Comments are required when rejecting a request.', 'warning');
        return false;
    }
    
    // Show loading
    showLoading('Processing ' + (action === 'approve' ? 'approval' : 'rejection') + '...');
});

// Initialize page
function pageInit() {
    // Auto-refresh every 30 seconds if there are pending requests
    <?php if (!empty($pending_requests)): ?>
    setInterval(function() {
        // Only refresh if no modal is open
        if (!document.querySelector('.modal.show')) {
            location.reload();
        }
    }, 30000);
    <?php endif; ?>
}
</script>

<?php
// Include footer
require_once '../includes/footer.php';
?>