<?php
/**
 * Create Employee Request (R-1 Form)
 * Employee Recruitment Management System
 * RR Construction (Pvt) Ltd
 */

$page_title = 'Create Employee Request';
$active_menu = 'requests';

require_once '../includes/header.php';
requirePermission(ROLE_SITE_ADMIN);

$currentUser = getCurrentUser();
$errors = [];
$success = false;

// Process form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Verify CSRF token
    if (!verifyCSRFToken($_POST['csrf_token'] ?? '')) {
        $errors[] = 'Invalid security token. Please try again.';
    } else {
        // Validate input
        $required = ['site_project', 'request_date', 'requirement_reason', 'total_employees'];
        $validation_errors = validateRequired($_POST, $required);
        
        if (empty($validation_errors)) {
            try {
                beginTransaction();
                
                // Generate request number
                $request_number = generateRequestNumber();
                
                // Insert main request
                $request_data = [
                    'request_number' => $request_number,
                    'site_project' => cleanInput($_POST['site_project']),
                    'requested_by' => $currentUser['id'],
                    'request_date' => $_POST['request_date'],
                    'requirement_reason' => $_POST['requirement_reason'],
                    'total_employees_requested' => (int)$_POST['total_employees'],
                    'urgent_requirement' => isset($_POST['urgent_requirement']) ? 1 : 0,
                    'expected_start_date' => $_POST['expected_start_date'] ?? null
                ];
                
                // Add replacement details if applicable
                if ($_POST['requirement_reason'] === 'replacement') {
                    $request_data['previous_employee_name'] = cleanInput($_POST['prev_emp_name'] ?? '');
                    $request_data['previous_employee_service_no'] = cleanInput($_POST['prev_emp_service'] ?? '');
                    $request_data['previous_employee_position'] = cleanInput($_POST['prev_emp_position'] ?? '');
                }
                
                $request_id = insertRecord('employee_requests', $request_data);
                
                // Insert employee categories
                if (isset($_POST['categories']) && is_array($_POST['categories'])) {
                    foreach ($_POST['categories'] as $category) {
                        if (!empty($category['position']) && !empty($category['quantity'])) {
                            $category_data = [
                                'request_id' => $request_id,
                                'category' => cleanInput($category['category']),
                                'position_title' => cleanInput($category['position']),
                                'quantity' => (int)$category['quantity'],
                                'salary_range_min' => !empty($category['salary_min']) ? (float)$category['salary_min'] : null,
                                'salary_range_max' => !empty($category['salary_max']) ? (float)$category['salary_max'] : null,
                                'experience_required' => cleanInput($category['experience'] ?? ''),
                                'special_requirements' => cleanInput($category['requirements'] ?? '')
                            ];
                            insertRecord('employee_categories', $category_data);
                        }
                    }
                }
                
                // Log activity
                logActivity($currentUser['id'], 'create', 'employee_requests', $request_id);
                
                commit();
                $success = true;
                setFlashMessage("Employee request $request_number created successfully!", 'success');
                
            } catch (Exception $e) {
                rollback();
                error_log("Error creating request: " . $e->getMessage());
                $errors[] = 'Failed to create request. Please try again.';
            }
        } else {
            $errors = array_merge($errors, array_values($validation_errors));
        }
    }
}
?>

<div class="row mb-4">
    <div class="col-12">
        <h4 class="mb-0">Create Employee Request (R-1 Form)</h4>
        <small class="text-muted">Submit new employee requirements</small>
    </div>
</div>

<?php if (!empty($errors)): ?>
<div class="alert alert-danger">
    <ul class="mb-0">
        <?php foreach ($errors as $error): ?>
            <li><?php echo htmlspecialchars($error); ?></li>
        <?php endforeach; ?>
    </ul>
</div>
<?php endif; ?>

<?php if ($success): ?>
<div class="alert alert-success">
    Request created successfully! <a href="view_requests.php">View all requests</a>
</div>
<?php endif; ?>

<div class="card">
    <div class="card-header">Employee Request Form</div>
    <div class="card-body">
        <form method="POST" id="requestForm">
            <input type="hidden" name="csrf_token" value="<?php echo generateCSRFToken(); ?>">
            
            <!-- Basic Information -->
            <div class="row mb-3">
                <div class="col-md-6">
                    <label class="form-label">Site/Project *</label>
                    <input type="text" name="site_project" class="form-control" 
                           value="<?php echo htmlspecialchars($currentUser['site_project']); ?>" readonly required>
                </div>
                <div class="col-md-6">
                    <label class="form-label">Request Date *</label>
                    <input type="date" name="request_date" class="form-control" 
                           value="<?php echo date('Y-m-d'); ?>" required>
                </div>
            </div>
            
            <div class="row mb-3">
                <div class="col-md-6">
                    <label class="form-label">Requirement Reason *</label>
                    <select name="requirement_reason" class="form-control" required onchange="toggleReplacementFields()">
                        <option value="">Select reason</option>
                        <option value="new_requirement">New Employee Requirement</option>
                        <option value="replacement">Employee Replacement</option>
                    </select>
                </div>
                <div class="col-md-6">
                    <label class="form-label">Expected Start Date</label>
                    <input type="date" name="expected_start_date" class="form-control">
                </div>
            </div>
            
            <!-- Replacement Details (hidden by default) -->
            <div id="replacementFields" style="display: none;">
                <h6 class="text-muted mb-3">Previous Employee Details</h6>
                <div class="row mb-3">
                    <div class="col-md-4">
                        <label class="form-label">Previous Employee Name</label>
                        <input type="text" name="prev_emp_name" class="form-control">
                    </div>
                    <div class="col-md-4">
                        <label class="form-label">Service Number</label>
                        <input type="text" name="prev_emp_service" class="form-control">
                    </div>
                    <div class="col-md-4">
                        <label class="form-label">Position</label>
                        <input type="text" name="prev_emp_position" class="form-control">
                    </div>
                </div>
            </div>
            
            <!-- Employee Categories -->
            <h6 class="text-muted mb-3">Employee Requirements</h6>
            <div id="categoriesContainer">
                <div class="category-row border p-3 mb-3">
                    <div class="row">
                        <div class="col-md-3">
                            <label class="form-label">Category *</label>
                            <select name="categories[0][category]" class="form-control" required>
                                <option value="">Select category</option>
                                <?php foreach (EMPLOYEE_CATEGORIES as $key => $value): ?>
                                    <option value="<?php echo $key; ?>"><?php echo $value; ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="col-md-3">
                            <label class="form-label">Position Title *</label>
                            <input type="text" name="categories[0][position]" class="form-control" required>
                        </div>
                        <div class="col-md-2">
                            <label class="form-label">Quantity *</label>
                            <input type="number" name="categories[0][quantity]" class="form-control" min="1" required>
                        </div>
                        <div class="col-md-2">
                            <label class="form-label">Min Salary</label>
                            <input type="number" name="categories[0][salary_min]" class="form-control" step="0.01">
                        </div>
                        <div class="col-md-2">
                            <label class="form-label">Max Salary</label>
                            <input type="number" name="categories[0][salary_max]" class="form-control" step="0.01">
                        </div>
                    </div>
                    <div class="row mt-2">
                        <div class="col-md-6">
                            <label class="form-label">Experience Required</label>
                            <input type="text" name="categories[0][experience]" class="form-control" 
                                   placeholder="e.g., 2+ years">
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">Special Requirements</label>
                            <input type="text" name="categories[0][requirements]" class="form-control" 
                                   placeholder="e.g., Valid license, Certification">
                        </div>
                    </div>
                </div>
            </div>
            
            <button type="button" class="btn btn-outline-secondary btn-sm mb-3" onclick="addCategory()">
                <i class="fas fa-plus"></i> Add Another Position
            </button>
            
            <div class="row mb-3">
                <div class="col-md-6">
                    <label class="form-label">Total Employees Requested *</label>
                    <input type="number" name="total_employees" class="form-control" min="1" required readonly>
                    <small class="text-muted">Auto-calculated from quantities above</small>
                </div>
                <div class="col-md-6">
                    <div class="form-check mt-4">
                        <input type="checkbox" name="urgent_requirement" class="form-check-input" id="urgent">
                        <label class="form-check-label" for="urgent">Urgent Requirement</label>
                    </div>
                </div>
            </div>
            
            <div class="text-end">
                <a href="view_requests.php" class="btn btn-outline-secondary">Cancel</a>
                <button type="submit" class="btn btn-primary">
                    <i class="fas fa-save"></i> Submit Request
                </button>
            </div>
        </form>
    </div>
</div>

<script>
let categoryIndex = 1;

function toggleReplacementFields() {
    const reason = document.querySelector('select[name="requirement_reason"]').value;
    const fields = document.getElementById('replacementFields');
    fields.style.display = reason === 'replacement' ? 'block' : 'none';
}

function addCategory() {
    const container = document.getElementById('categoriesContainer');
    const categoryHtml = `
        <div class="category-row border p-3 mb-3">
            <div class="d-flex justify-content-between align-items-center mb-2">
                <span class="text-muted">Position ${categoryIndex + 1}</span>
                <button type="button" class="btn btn-outline-danger btn-sm" onclick="removeCategory(this)">
                    <i class="fas fa-times"></i> Remove
                </button>
            </div>
            <div class="row">
                <div class="col-md-3">
                    <label class="form-label">Category *</label>
                    <select name="categories[${categoryIndex}][category]" class="form-control" required>
                        <option value="">Select category</option>
                        <?php foreach (EMPLOYEE_CATEGORIES as $key => $value): ?>
                            <option value="<?php echo $key; ?>"><?php echo $value; ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="col-md-3">
                    <label class="form-label">Position Title *</label>
                    <input type="text" name="categories[${categoryIndex}][position]" class="form-control" required>
                </div>
                <div class="col-md-2">
                    <label class="form-label">Quantity *</label>
                    <input type="number" name="categories[${categoryIndex}][quantity]" class="form-control" min="1" required>
                </div>
                <div class="col-md-2">
                    <label class="form-label">Min Salary</label>
                    <input type="number" name="categories[${categoryIndex}][salary_min]" class="form-control" step="0.01">
                </div>
                <div class="col-md-2">
                    <label class="form-label">Max Salary</label>
                    <input type="number" name="categories[${categoryIndex}][salary_max]" class="form-control" step="0.01">
                </div>
            </div>
            <div class="row mt-2">
                <div class="col-md-6">
                    <label class="form-label">Experience Required</label>
                    <input type="text" name="categories[${categoryIndex}][experience]" class="form-control">
                </div>
                <div class="col-md-6">
                    <label class="form-label">Special Requirements</label>
                    <input type="text" name="categories[${categoryIndex}][requirements]" class="form-control">
                </div>
            </div>
        </div>
    `;
    container.insertAdjacentHTML('beforeend', categoryHtml);
    categoryIndex++;
    updateTotalEmployees();
}

function removeCategory(button) {
    button.closest('.category-row').remove();
    updateTotalEmployees();
}

function updateTotalEmployees() {
    const quantities = document.querySelectorAll('input[name*="[quantity]"]');
    let total = 0;
    quantities.forEach(input => {
        if (input.value) total += parseInt(input.value);
    });
    document.querySelector('input[name="total_employees"]').value = total;
}

// Auto-calculate total when quantities change
document.addEventListener('input', function(e) {
    if (e.target.name && e.target.name.includes('[quantity]')) {
        updateTotalEmployees();
    }
});

// Initialize
updateTotalEmployees();
</script>

<?php require_once '../includes/footer.php'; ?>