<?php
/**
 * Get Request Details (AJAX)
 * Employee Recruitment Management System
 * RR Construction (Pvt) Ltd
 */

// Define access constant
define('HRM_ACCESS', true);

// Start session
session_start();

// Include required files
require_once '../config/constants.php';
require_once '../config/database.php';
require_once '../includes/functions.php';

// Set content type
header('Content-Type: text/html; charset=UTF-8');

// Check authentication
if (!isLoggedIn()) {
    echo '<div class="alert alert-danger">Access denied. Please log in.</div>';
    exit;
}

$currentUser = getCurrentUser();
if (!$currentUser) {
    echo '<div class="alert alert-danger">User session invalid.</div>';
    exit;
}

// Get request ID
$request_id = (int)($_GET['id'] ?? 0);

if (!$request_id) {
    echo '<div class="alert alert-danger">Invalid request ID.</div>';
    exit;
}

try {
    // Get request details
    $sql = "SELECT er.*, u.full_name as requested_by_name, u.email as requester_email
            FROM employee_requests er
            LEFT JOIN users u ON er.requested_by = u.id
            WHERE er.id = ?";
    
    $request = fetchOne($sql, [$request_id]);
    
    if (!$request) {
        echo '<div class="alert alert-danger">Request not found.</div>';
        exit;
    }
    
    // Check permissions
    $has_access = false;
    
    if ($currentUser['role'] === ROLE_SUPER_ADMIN) {
        $has_access = true;
    } elseif ($currentUser['role'] === ROLE_SITE_ADMIN) {
        $has_access = ($request['requested_by'] == $currentUser['id']);
    } elseif ($currentUser['role'] === ROLE_PROJECT_MANAGER) {
        $has_access = ($request['site_project'] === $currentUser['site_project']);
    } elseif (in_array($currentUser['role'], [ROLE_GENERAL_MANAGER, ROLE_HR_MANAGER, ROLE_ACCOUNTANT])) {
        $has_access = true;
    }
    
    if (!$has_access) {
        echo '<div class="alert alert-danger">Access denied. You do not have permission to view this request.</div>';
        exit;
    }
    
    // Get employee categories for this request
    $categories_sql = "SELECT * FROM employee_categories WHERE request_id = ? ORDER BY id";
    $categories = fetchAll($categories_sql, [$request_id]);
    
    // Get applications for this request
    $applications_sql = "SELECT COUNT(*) as count, status, COUNT(CASE WHEN status = 'hired' THEN 1 END) as hired_count
                        FROM job_applications 
                        WHERE request_id = ? 
                        GROUP BY status";
    $applications = fetchAll($applications_sql, [$request_id]);
    
    // Get approval history
    $approval_sql = "SELECT al.*, u.full_name as approved_by_name
                     FROM approval_logs al
                     LEFT JOIN users u ON al.approved_by = u.id
                     WHERE al.request_id = ?
                     ORDER BY al.approved_at ASC";
    $approvals = fetchAll($approval_sql, [$request_id]);
    
    ?>
    
    <!-- Request Header -->
    <div class="row mb-3">
        <div class="col-md-6">
            <h5 class="text-primary mb-3">
                <i class="fas fa-file-alt me-2"></i>
                <?php echo htmlspecialchars($request['request_number']); ?>
                <?php if ($request['urgent_requirement']): ?>
                    <span class="badge bg-danger ms-2">Urgent</span>
                <?php endif; ?>
            </h5>
        </div>
        <div class="col-md-6 text-end">
            <span class="badge bg-<?php echo getStatusColor($request['status']); ?> p-2">
                <?php echo getStatusName($request['status']); ?>
            </span>
        </div>
    </div>
    
    <!-- Basic Information -->
    <div class="row mb-4">
        <div class="col-md-6">
            <h6 class="text-secondary mb-2">Request Information</h6>
            <table class="table table-sm table-borderless">
                <tr>
                    <td width="140"><strong>Project:</strong></td>
                    <td><?php echo htmlspecialchars($request['site_project']); ?></td>
                </tr>
                <tr>
                    <td><strong>Request Date:</strong></td>
                    <td><?php echo formatDisplayDate($request['request_date']); ?></td>
                </tr>
                <tr>
                    <td><strong>Requirement:</strong></td>
                    <td><?php echo REQUIREMENT_REASONS[$request['requirement_reason']] ?? $request['requirement_reason']; ?></td>
                </tr>
                <tr>
                    <td><strong>Total Employees:</strong></td>
                    <td><?php echo $request['total_employees_requested']; ?></td>
                </tr>
                <tr>
                    <td><strong>Expected Start:</strong></td>
                    <td><?php echo $request['expected_start_date'] ? formatDisplayDate($request['expected_start_date']) : 'Not specified'; ?></td>
                </tr>
            </table>
        </div>
        <div class="col-md-6">
            <h6 class="text-secondary mb-2">Requester Information</h6>
            <table class="table table-sm table-borderless">
                <tr>
                    <td width="140"><strong>Requested by:</strong></td>
                    <td><?php echo htmlspecialchars($request['requested_by_name']); ?></td>
                </tr>
                <tr>
                    <td><strong>Email:</strong></td>
                    <td><?php echo htmlspecialchars($request['requester_email'] ?: 'Not provided'); ?></td>
                </tr>
                <tr>
                    <td><strong>Submitted:</strong></td>
                    <td><?php echo formatDisplayDateTime($request['created_at']); ?></td>
                </tr>
                <tr>
                    <td><strong>Last Updated:</strong></td>
                    <td><?php echo formatDisplayDateTime($request['updated_at']); ?></td>
                </tr>
                <?php if ($request['budget_allocated']): ?>
                <tr>
                    <td><strong>Budget:</strong></td>
                    <td><?php echo formatCurrencyAmount($request['budget_allocated']); ?></td>
                </tr>
                <?php endif; ?>
            </table>
        </div>
    </div>
    
    <!-- Replacement Details (if applicable) -->
    <?php if ($request['requirement_reason'] === 'replacement'): ?>
    <div class="mb-4">
        <h6 class="text-secondary mb-2">Previous Employee Details</h6>
        <div class="card bg-light">
            <div class="card-body">
                <div class="row">
                    <div class="col-md-4">
                        <strong>Name:</strong> <?php echo htmlspecialchars($request['previous_employee_name'] ?: 'Not specified'); ?>
                    </div>
                    <div class="col-md-4">
                        <strong>Service No:</strong> <?php echo htmlspecialchars($request['previous_employee_service_no'] ?: 'Not specified'); ?>
                    </div>
                    <div class="col-md-4">
                        <strong>Position:</strong> <?php echo htmlspecialchars($request['previous_employee_position'] ?: 'Not specified'); ?>
                    </div>
                </div>
            </div>
        </div>
    </div>
    <?php endif; ?>
    
    <!-- Employee Categories -->
    <?php if (!empty($categories)): ?>
    <div class="mb-4">
        <h6 class="text-secondary mb-2">Required Positions (<?php echo count($categories); ?>)</h6>
        <div class="table-responsive">
            <table class="table table-sm table-bordered">
                <thead class="bg-light">
                    <tr>
                        <th>Category</th>
                        <th>Position Title</th>
                        <th>Quantity</th>
                        <th>Salary Range</th>
                        <th>Experience</th>
                        <th>Special Requirements</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($categories as $category): ?>
                    <tr>
                        <td><?php echo htmlspecialchars($category['category']); ?></td>
                        <td><strong><?php echo htmlspecialchars($category['position_title']); ?></strong></td>
                        <td class="text-center">
                            <span class="badge bg-primary"><?php echo $category['quantity']; ?></span>
                        </td>
                        <td>
                            <?php 
                            if ($category['salary_range_min'] && $category['salary_range_max']) {
                                echo formatCurrencyAmount($category['salary_range_min']) . ' - ' . formatCurrencyAmount($category['salary_range_max']);
                            } else {
                                echo 'Not specified';
                            }
                            ?>
                        </td>
                        <td><?php echo htmlspecialchars($category['experience_required'] ?: 'Not specified'); ?></td>
                        <td><?php echo htmlspecialchars($category['special_requirements'] ?: 'None'); ?></td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    </div>
    <?php endif; ?>
    
    <!-- Applications Summary -->
    <?php if (!empty($applications)): ?>
    <div class="mb-4">
        <h6 class="text-secondary mb-2">Applications Summary</h6>
        <div class="row">
            <?php 
            $total_applications = array_sum(array_column($applications, 'count'));
            $total_hired = array_sum(array_column($applications, 'hired_count'));
            ?>
            <div class="col-md-6">
                <div class="card bg-info text-white">
                    <div class="card-body text-center">
                        <h4><?php echo $total_applications; ?></h4>
                        <small>Total Applications</small>
                    </div>
                </div>
            </div>
            <div class="col-md-6">
                <div class="card bg-success text-white">
                    <div class="card-body text-center">
                        <h4><?php echo $total_hired; ?></h4>
                        <small>Hired</small>
                    </div>
                </div>
            </div>
        </div>
        
        <?php if ($total_applications > 0): ?>
        <div class="mt-3">
            <a href="../employees/view_applications.php?request_id=<?php echo $request['id']; ?>" 
               class="btn btn-outline-primary btn-sm" target="_blank">
                <i class="fas fa-users me-2"></i>View All Applications
            </a>
        </div>
        <?php endif; ?>
    </div>
    <?php endif; ?>
    
    <!-- Approval History -->
    <?php if (!empty($approvals)): ?>
    <div class="mb-4">
        <h6 class="text-secondary mb-2">Approval History</h6>
        <div class="table-responsive">
            <table class="table table-sm">
                <thead class="bg-light">
                    <tr>
                        <th>Level</th>
                        <th>Approved By</th>
                        <th>Action</th>
                        <th>Comments</th>
                        <th>Date</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($approvals as $approval): ?>
                    <tr>
                        <td><?php echo getRoleName($approval['approval_level']); ?></td>
                        <td><?php echo htmlspecialchars($approval['approved_by_name']); ?></td>
                        <td>
                            <span class="badge <?php echo $approval['action'] === 'approved' ? 'bg-success' : 'bg-danger'; ?>">
                                <?php echo ucfirst($approval['action']); ?>
                            </span>
                        </td>
                        <td><?php echo htmlspecialchars($approval['comments'] ?: 'No comments'); ?></td>
                        <td><?php echo formatDisplayDateTime($approval['approved_at']); ?></td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    </div>
    <?php endif; ?>
    
    <!-- Rejection Reason (if applicable) -->
    <?php if ($request['status'] === 'rejected' && $request['rejection_reason']): ?>
    <div class="alert alert-danger">
        <h6><i class="fas fa-exclamation-triangle me-2"></i>Rejection Reason</h6>
        <p class="mb-0"><?php echo htmlspecialchars($request['rejection_reason']); ?></p>
    </div>
    <?php endif; ?>
    
    <?php
    
} catch (Exception $e) {
    error_log("Error fetching request details: " . $e->getMessage());
    echo '<div class="alert alert-danger">Error loading request details. Please try again.</div>';
}
?>