<?php
/**
 * Print Employee Request
 * Employee Recruitment Management System
 * RR Construction (Pvt) Ltd
 */

// Define access constant
define('HRM_ACCESS', true);

// Start session if not already started
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Include required files
require_once '../config/constants.php';
require_once '../config/database.php';
require_once '../includes/functions.php';

// Check if user is logged in
if (!isLoggedIn()) {
    header('Location: ' . BASE_URL . 'auth/login.php');
    exit;
}

$currentUser = getCurrentUser();
$request_id = (int)($_GET['id'] ?? 0);

if (!$request_id) {
    die('Invalid request ID');
}

// Get request details
try {
    $sql = "SELECT er.*, u.full_name as requested_by_name, u.role as requested_by_role
            FROM employee_requests er
            JOIN users u ON er.requested_by = u.id
            WHERE er.id = ?";
    
    // Add role-based access control
    $params = [$request_id];
    
    switch ($currentUser['role']) {
        case ROLE_SUPER_ADMIN:
            // Super admin can view all requests
            break;
            
        case ROLE_SITE_ADMIN:
            $sql .= " AND er.requested_by = ?";
            $params[] = $currentUser['id'];
            break;
            
        case ROLE_PROJECT_MANAGER:
            $sql .= " AND er.site_project = ?";
            $params[] = $currentUser['site_project'];
            break;
            
        case ROLE_GENERAL_MANAGER:
        case ROLE_HR_MANAGER:
        case ROLE_ACCOUNTANT:
            // These roles can view all requests
            break;
            
        default:
            die('Access denied');
            break;
    }
    
    $request = fetchOne($sql, $params);
    
    if (!$request) {
        die('Request not found or access denied');
    }
    
} catch (Exception $e) {
    error_log("Error fetching request: " . $e->getMessage());
    die('Error loading request details');
}

// Get employee categories for this request
try {
    $sql = "SELECT * FROM employee_categories WHERE request_id = ? ORDER BY position_title";
    $categories = fetchAll($sql, [$request_id]);
} catch (Exception $e) {
    error_log("Error fetching categories: " . $e->getMessage());
    $categories = [];
}

// Get applications count for this request
try {
    $sql = "SELECT COUNT(*) as total_applications,
                   COUNT(CASE WHEN ja.status = 'hired' THEN 1 END) as hired_count
            FROM employee_categories ec
            LEFT JOIN job_applications ja ON ec.id = ja.category_id
            WHERE ec.request_id = ?";
    $app_stats = fetchOne($sql, [$request_id]);
} catch (Exception $e) {
    $app_stats = ['total_applications' => 0, 'hired_count' => 0];
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Employee Request - <?php echo htmlspecialchars($request['request_number']); ?></title>
    <style>
        /* A4 Print Styles */
        @page {
            size: A4;
            margin: 15mm;
        }
        
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: Arial, sans-serif;
            font-size: 11px;
            line-height: 1.3;
            color: #000;
            background: white;
        }
        
        .container {
            max-width: 100%;
            padding: 0;
        }
        
        /* Header */
        .header {
            text-align: center;
            margin-bottom: 20px;
            border-bottom: 2px solid #000;
            padding-bottom: 10px;
        }
        
        .company-name {
            font-size: 18px;
            font-weight: bold;
            margin-bottom: 5px;
        }
        
        .system-name {
            font-size: 12px;
            color: #666;
            margin-bottom: 5px;
        }
        
        .document-title {
            font-size: 16px;
            font-weight: bold;
            margin-top: 10px;
        }
        
        /* Content Layout */
        .content {
            margin-bottom: 15px;
        }
        
        .section {
            margin-bottom: 15px;
            page-break-inside: avoid;
        }
        
        .section-title {
            font-size: 12px;
            font-weight: bold;
            background-color: #f0f0f0;
            padding: 5px;
            border: 1px solid #ccc;
            margin-bottom: 8px;
        }
        
        /* Tables */
        table {
            width: 100%;
            border-collapse: collapse;
            margin-bottom: 10px;
        }
        
        .info-table td {
            padding: 4px 8px;
            border: 1px solid #ddd;
            vertical-align: top;
        }
        
        .info-table .label {
            font-weight: bold;
            background-color: #f9f9f9;
            width: 30%;
        }
        
        .categories-table th,
        .categories-table td {
            padding: 6px;
            border: 1px solid #000;
            text-align: left;
        }
        
        .categories-table th {
            background-color: #e0e0e0;
            font-weight: bold;
            text-align: center;
        }
        
        .categories-table td {
            font-size: 10px;
        }
        
        /* Status styling */
        .status {
            display: inline-block;
            padding: 2px 6px;
            border: 1px solid #000;
            font-weight: bold;
            text-transform: uppercase;
        }
        
        .status-pending { background-color: #fff3cd; }
        .status-approved { background-color: #d4edda; }
        .status-rejected { background-color: #f8d7da; }
        
        /* Two column layout */
        .row {
            display: flex;
            margin-bottom: 10px;
        }
        
        .col-6 {
            width: 50%;
            padding-right: 10px;
        }
        
        .col-6:last-child {
            padding-right: 0;
            padding-left: 10px;
        }
        
        /* Footer */
        .footer {
            position: fixed;
            bottom: 10mm;
            left: 15mm;
            right: 15mm;
            text-align: center;
            font-size: 9px;
            color: #666;
            border-top: 1px solid #ccc;
            padding-top: 5px;
        }
        
        /* Signature section */
        .signatures {
            margin-top: 20px;
            display: flex;
            justify-content: space-between;
        }
        
        .signature-box {
            width: 30%;
            text-align: center;
            border-top: 1px solid #000;
            padding-top: 5px;
            margin-top: 30px;
        }
        
        /* Print-specific */
        @media print {
            body {
                -webkit-print-color-adjust: exact;
                print-color-adjust: exact;
            }
            
            .no-print {
                display: none !important;
            }
            
            .page-break {
                page-break-before: always;
            }
        }
        
        /* Screen view button */
        .print-button {
            position: fixed;
            top: 10px;
            right: 10px;
            padding: 10px 20px;
            background: #007bff;
            color: white;
            border: none;
            border-radius: 5px;
            cursor: pointer;
            font-size: 14px;
            z-index: 1000;
        }
        
        .print-button:hover {
            background: #0056b3;
        }
        
        @media print {
            .print-button {
                display: none;
            }
        }
    </style>
</head>
<body>
    <button class="print-button no-print" onclick="window.print()">
        <i class="fas fa-print"></i> Print
    </button>

    <div class="container">
        <!-- Header -->
        <div class="header">
            <div class="company-name"><?php echo COMPANY_NAME; ?></div>
            <div class="system-name"><?php echo APP_NAME; ?></div>
            <div class="document-title">EMPLOYEE RECRUITMENT REQUEST</div>
        </div>

        <!-- Request Information -->
        <div class="section">
            <div class="section-title">REQUEST INFORMATION</div>
            <div class="row">
                <div class="col-6">
                    <table class="info-table">
                        <tr>
                            <td class="label">Request Number:</td>
                            <td><strong><?php echo htmlspecialchars($request['request_number']); ?></strong></td>
                        </tr>
                        <tr>
                            <td class="label">Site/Project:</td>
                            <td><?php echo htmlspecialchars($request['site_project']); ?></td>
                        </tr>
                        <tr>
                            <td class="label">Request Date:</td>
                            <td><?php echo formatDisplayDate($request['request_date']); ?></td>
                        </tr>
                        <tr>
                            <td class="label">Requested By:</td>
                            <td><?php echo htmlspecialchars($request['requested_by_name']); ?></td>
                        </tr>
                    </table>
                </div>
                <div class="col-6">
                    <table class="info-table">
                        <tr>
                            <td class="label">Status:</td>
                            <td>
                                <span class="status status-<?php echo str_replace('_', '-', $request['status']); ?>">
                                    <?php echo getStatusName($request['status']); ?>
                                </span>
                            </td>
                        </tr>
                        <tr>
                            <td class="label">Requirement Reason:</td>
                            <td><?php echo REQUIREMENT_REASONS[$request['requirement_reason']] ?? $request['requirement_reason']; ?></td>
                        </tr>
                        <tr>
                            <td class="label">Expected Start Date:</td>
                            <td><?php echo $request['expected_start_date'] ? formatDisplayDate($request['expected_start_date']) : 'Not specified'; ?></td>
                        </tr>
                        <tr>
                            <td class="label">Urgent Request:</td>
                            <td><?php echo $request['urgent_requirement'] ? 'YES' : 'No'; ?></td>
                        </tr>
                    </table>
                </div>
            </div>
        </div>

        <!-- Previous Employee Information (if replacement) -->
        <?php if ($request['requirement_reason'] === 'replacement' && !empty($request['previous_employee_name'])): ?>
        <div class="section">
            <div class="section-title">PREVIOUS EMPLOYEE INFORMATION</div>
            <table class="info-table">
                <tr>
                    <td class="label" style="width: 25%;">Employee Name:</td>
                    <td style="width: 25%;"><?php echo htmlspecialchars($request['previous_employee_name']); ?></td>
                    <td class="label" style="width: 25%;">Service Number:</td>
                    <td style="width: 25%;"><?php echo htmlspecialchars($request['previous_employee_service_no'] ?? 'N/A'); ?></td>
                </tr>
                <tr>
                    <td class="label">Position:</td>
                    <td colspan="3"><?php echo htmlspecialchars($request['previous_employee_position'] ?? 'N/A'); ?></td>
                </tr>
            </table>
        </div>
        <?php endif; ?>

        <!-- Employee Categories -->
        <div class="section">
            <div class="section-title">POSITIONS REQUESTED</div>
            <?php if (!empty($categories)): ?>
            <table class="categories-table">
                <thead>
                    <tr>
                        <th style="width: 5%;">#</th>
                        <th style="width: 20%;">Category</th>
                        <th style="width: 25%;">Position Title</th>
                        <th style="width: 8%;">Qty</th>
                        <th style="width: 15%;">Salary Range</th>
                        <th style="width: 12%;">Experience</th>
                        <th style="width: 15%;">Special Requirements</th>
                    </tr>
                </thead>
                <tbody>
                    <?php 
                    $total_quantity = 0;
                    foreach ($categories as $index => $category): 
                        $total_quantity += $category['quantity'];
                    ?>
                    <tr>
                        <td style="text-align: center;"><?php echo $index + 1; ?></td>
                        <td><?php echo htmlspecialchars($category['category']); ?></td>
                        <td><?php echo htmlspecialchars($category['position_title']); ?></td>
                        <td style="text-align: center;"><strong><?php echo $category['quantity']; ?></strong></td>
                        <td style="text-align: center;">
                            <?php 
                            if (!empty($category['salary_range_min']) || !empty($category['salary_range_max'])) {
                                $min = $category['salary_range_min'] ? 'Rs. ' . number_format($category['salary_range_min']) : '';
                                $max = $category['salary_range_max'] ? 'Rs. ' . number_format($category['salary_range_max']) : '';
                                echo $min . ($min && $max ? ' - ' : '') . $max;
                            } else {
                                echo 'Not specified';
                            }
                            ?>
                        </td>
                        <td><?php echo htmlspecialchars($category['experience_required'] ?: 'Any'); ?></td>
                        <td><?php echo htmlspecialchars($category['special_requirements'] ?: 'None'); ?></td>
                    </tr>
                    <?php endforeach; ?>
                    <tr style="background-color: #f0f0f0; font-weight: bold;">
                        <td colspan="3" style="text-align: center;">TOTAL EMPLOYEES REQUESTED</td>
                        <td style="text-align: center;"><?php echo $total_quantity; ?></td>
                        <td colspan="3"></td>
                    </tr>
                </tbody>
            </table>
            <?php else: ?>
            <p>No positions specified for this request.</p>
            <?php endif; ?>
        </div>

        <!-- Application Statistics -->
        <div class="section">
            <div class="section-title">APPLICATION STATUS</div>
            <table class="info-table">
                <tr>
                    <td class="label" style="width: 25%;">Total Applications Received:</td>
                    <td style="width: 25%;"><strong><?php echo $app_stats['total_applications']; ?></strong></td>
                    <td class="label" style="width: 25%;">Employees Hired:</td>
                    <td style="width: 25%;"><strong><?php echo $app_stats['hired_count']; ?></strong></td>
                </tr>
                <tr>
                    <td class="label">Remaining Positions:</td>
                    <td><strong><?php echo $total_quantity - $app_stats['hired_count']; ?></strong></td>
                    <td class="label">Completion Status:</td>
                    <td>
                        <?php 
                        $completion_percentage = $total_quantity > 0 ? round(($app_stats['hired_count'] / $total_quantity) * 100) : 0;
                        echo $completion_percentage . '% Complete';
                        ?>
                    </td>
                </tr>
            </table>
        </div>

        <!-- Budget Information -->
        <?php if (!empty($request['budget_allocated'])): ?>
        <div class="section">
            <div class="section-title">BUDGET INFORMATION</div>
            <table class="info-table">
                <tr>
                    <td class="label" style="width: 30%;">Budget Allocated:</td>
                    <td><strong>Rs. <?php echo number_format($request['budget_allocated'], 2); ?></strong></td>
                </tr>
            </table>
        </div>
        <?php endif; ?>

        <!-- Additional Information -->
        <div class="section">
            <div class="section-title">ADDITIONAL INFORMATION</div>
            <table class="info-table">
                <tr>
                    <td class="label" style="width: 25%;">Created Date:</td>
                    <td style="width: 25%;"><?php echo formatDisplayDateTime($request['created_at']); ?></td>
                    <td class="label" style="width: 25%;">Last Updated:</td>
                    <td style="width: 25%;"><?php echo formatDisplayDateTime($request['updated_at']); ?></td>
                </tr>
            </table>
        </div>

        <!-- Signatures -->
        <div class="signatures">
            <div class="signature-box">
                <div>Requested By</div>
                <div style="margin-top: 20px; font-size: 10px;">
                    <?php echo htmlspecialchars($request['requested_by_name']); ?>
                </div>
            </div>
            <div class="signature-box">
                <div>Project Manager</div>
                <div style="margin-top: 20px; font-size: 10px;">
                    Approval
                </div>
            </div>
            <div class="signature-box">
                <div>HR Manager</div>
                <div style="margin-top: 20px; font-size: 10px;">
                    Approval
                </div>
            </div>
        </div>
    </div>

    <!-- Footer -->
    <div class="footer">
        <div>
            Generated on <?php echo date('d/m/Y H:i:s'); ?> | 
            <?php echo COMPANY_NAME; ?> - <?php echo APP_NAME; ?> | 
            Page 1 of 1
        </div>
    </div>

    <script>
        // Auto-print when page loads (optional)
        // window.onload = function() { window.print(); };
        
        // Print function
        function printDocument() {
            window.print();
        }
        
        // Close window after printing (if opened in new tab)
        window.onafterprint = function() {
            // Uncomment the line below if you want to auto-close after printing
            // window.close();
        };
    </script>
</body>
</html>