<?php
/**
 * View Employee Requests
 * Employee Recruitment Management System
 * RR Construction (Pvt) Ltd
 */

$page_title = 'Employee Requests';
$active_menu = 'requests';

require_once '../includes/header.php';
requirePermission(ROLE_SITE_ADMIN);

$currentUser = getCurrentUser();

// Get filter parameters
$status_filter = $_GET['status'] ?? '';
$search = $_GET['search'] ?? '';
$page = max(1, (int)($_GET['page'] ?? 1));
$records_per_page = 20;
$offset = ($page - 1) * $records_per_page;

// Build query conditions
$where_conditions = ['1=1'];
$params = [];

// Role-based filtering
if ($currentUser['role'] === ROLE_SITE_ADMIN) {
    $where_conditions[] = 'er.requested_by = ?';
    $params[] = $currentUser['id'];
} elseif ($currentUser['role'] === ROLE_PROJECT_MANAGER) {
    $where_conditions[] = 'er.site_project = ?';
    $params[] = $currentUser['site_project'];
}

// Status filter
if (!empty($status_filter)) {
    $where_conditions[] = 'er.status = ?';
    $params[] = $status_filter;
}

// Search filter
if (!empty($search)) {
    $where_conditions[] = '(er.request_number LIKE ? OR er.site_project LIKE ?)';
    $params[] = "%$search%";
    $params[] = "%$search%";
}

$where_clause = implode(' AND ', $where_conditions);

// Get total count
$count_sql = "SELECT COUNT(*) as total FROM employee_requests er WHERE $where_clause";
$total_records = fetchOne($count_sql, $params)['total'];

// Get requests with detailed application breakdown
$sql = "SELECT er.*, u.full_name as requested_by_name,
               COUNT(DISTINCT ec.id) as categories_count,
               COUNT(DISTINCT ja.id) as applications_count,
               COUNT(CASE WHEN ja.status = 'submitted' THEN 1 END) as submitted_apps,
               COUNT(CASE WHEN ja.status IN ('approved_by_pm', 'approved_by_hr', 'approved_by_accountant') THEN 1 END) as pending_apps,
               COUNT(CASE WHEN ja.status = 'hired' THEN 1 END) as hired_apps,
               COUNT(CASE WHEN ja.status = 'rejected' THEN 1 END) as rejected_apps,
               SUM(ec.quantity) as total_positions_needed
        FROM employee_requests er
        LEFT JOIN users u ON er.requested_by = u.id
        LEFT JOIN employee_categories ec ON er.id = ec.request_id
        LEFT JOIN job_applications ja ON ec.id = ja.category_id
        WHERE $where_clause
        GROUP BY er.id
        ORDER BY er.created_at DESC
        LIMIT $records_per_page OFFSET $offset";

$requests = fetchAll($sql, $params);

// Pagination data
$total_pages = ceil($total_records / $records_per_page);
?>

<div class="row mb-4">
    <div class="col-md-6">
        <h4 class="mb-0">Employee Requests</h4>
        <small class="text-muted">Manage employee recruitment requests</small>
    </div>
    <div class="col-md-6 text-end">
        <?php if ($currentUser['role'] === ROLE_SITE_ADMIN): ?>
        <a href="create_request.php" class="btn btn-primary btn-sm">
            <i class="fas fa-plus"></i> New Request
        </a>
        <?php endif; ?>
    </div>
</div>

<!-- Filters -->
<div class="card mb-3">
    <div class="card-body">
        <form method="GET" class="row g-3">
            <div class="col-md-3">
                <label class="form-label">Status</label>
                <select name="status" class="form-control">
                    <option value="">All Status</option>
                    <?php foreach (REQUEST_STATUSES as $key => $value): ?>
                        <option value="<?php echo $key; ?>" <?php echo $status_filter === $key ? 'selected' : ''; ?>>
                            <?php echo $value; ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
            <div class="col-md-6">
                <label class="form-label">Search</label>
                <input type="text" name="search" class="form-control" 
                       placeholder="Request number or project name" 
                       value="<?php echo htmlspecialchars($search); ?>">
            </div>
            <div class="col-md-3">
                <label class="form-label">&nbsp;</label>
                <div>
                    <button type="submit" class="btn btn-outline-primary btn-sm">
                        <i class="fas fa-search"></i> Filter
                    </button>
                    <a href="view_requests.php" class="btn btn-outline-secondary btn-sm">
                        <i class="fas fa-times"></i> Clear
                    </a>
                </div>
            </div>
        </form>
    </div>
</div>

<!-- Results -->
<div class="card">
    <div class="card-header d-flex justify-content-between align-items-center">
        <span>Requests (<?php echo $total_records; ?>)</span>
        <?php if ($total_records > 0): ?>
        <small class="text-muted">Page <?php echo $page; ?> of <?php echo $total_pages; ?></small>
        <?php endif; ?>
    </div>
    <div class="card-body p-0">
        <?php if (empty($requests)): ?>
        <div class="text-center py-4">
            <i class="fas fa-inbox fa-2x text-muted mb-3"></i>
            <p class="text-muted">No requests found</p>
            <?php if ($currentUser['role'] === ROLE_SITE_ADMIN): ?>
            <a href="create_request.php" class="btn btn-primary btn-sm">Create First Request</a>
            <?php endif; ?>
        </div>
        <?php else: ?>
        <div class="table-responsive">
            <table class="table table-hover mb-0">
                <thead class="bg-light">
                    <tr>
                        <th>Request #</th>
                        <th>Project</th>
                        <th>Date</th>
                        <th>Reason</th>
                        <th>Positions</th>
                        <th>Applications</th>
                        <th>Status</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($requests as $request): ?>
                    <tr>
                        <td>
                            <strong><?php echo htmlspecialchars($request['request_number']); ?></strong>
                            <?php if ($request['urgent_requirement']): ?>
                                <br><small class="text-danger"><i class="fas fa-exclamation-triangle"></i> Urgent</small>
                            <?php endif; ?>
                        </td>
                        <td>
                            <div><?php echo htmlspecialchars($request['site_project']); ?></div>
                            <?php if ($currentUser['role'] !== ROLE_SITE_ADMIN): ?>
                            <small class="text-muted">by <?php echo htmlspecialchars($request['requested_by_name']); ?></small>
                            <?php endif; ?>
                        </td>
                        <td>
                            <div><?php echo formatDisplayDate($request['request_date']); ?></div>
                            <small class="text-muted"><?php echo getRelativeTime($request['created_at']); ?></small>
                        </td>
                        <td>
                            <small><?php echo REQUIREMENT_REASONS[$request['requirement_reason']] ?? $request['requirement_reason']; ?></small>
                        </td>
                        <td class="text-center">
                            <div>
                                <span class="badge bg-info"><?php echo $request['categories_count']; ?></span>
                                <small class="text-muted d-block">Categories</small>
                            </div>
                            <div class="mt-1">
                                <strong><?php echo $request['total_positions_needed'] ?? $request['total_employees_requested']; ?></strong>
                                <small class="text-muted d-block">Total Needed</small>
                            </div>
                        </td>
                        <td class="text-center">
                            <div class="mb-1">
                                <strong><?php echo $request['applications_count']; ?></strong>
                                <small class="text-muted">Total</small>
                            </div>
                            <?php if ($request['applications_count'] > 0): ?>
                            <div class="small">
                                <?php if ($request['submitted_apps'] > 0): ?>
                                <span class="badge bg-info"><?php echo $request['submitted_apps']; ?> New</span>
                                <?php endif; ?>
                                <?php if ($request['pending_apps'] > 0): ?>
                                <span class="badge bg-warning"><?php echo $request['pending_apps']; ?> Pending</span>
                                <?php endif; ?>
                                <?php if ($request['hired_apps'] > 0): ?>
                                <span class="badge bg-success"><?php echo $request['hired_apps']; ?> Hired</span>
                                <?php endif; ?>
                                <?php if ($request['rejected_apps'] > 0): ?>
                                <span class="badge bg-danger"><?php echo $request['rejected_apps']; ?> Rejected</span>
                                <?php endif; ?>
                            </div>
                            <?php endif; ?>
                        </td>
                        <td>
                            <?php echo getStatusBadge($request['status']); ?>
                            <?php if (!empty($request['expected_start_date'])): ?>
                                <br><small class="text-muted">Start: <?php echo formatDisplayDate($request['expected_start_date']); ?></small>
                            <?php endif; ?>
                        </td>
                        <td>
                            <div class="btn-group btn-group-sm">
                                <button type="button" class="btn btn-outline-primary dropdown-toggle" 
                                        data-bs-toggle="dropdown" aria-expanded="false">
                                    <i class="fas fa-cog"></i>
                                </button>
                                <ul class="dropdown-menu">
                                    <li>
                                        <button class="dropdown-item" onclick="viewRequest(<?php echo $request['id']; ?>)">
                                            <i class="fas fa-eye me-2"></i>View Details
                                        </button>
                                    </li>
                                    <li><hr class="dropdown-divider"></li>
                                    <?php if ($request['applications_count'] > 0): ?>
                                    <li>
                                        <a class="dropdown-item" href="../employees/view_applications.php?request_id=<?php echo $request['id']; ?>">
                                            <i class="fas fa-users me-2"></i>View Applications (<?php echo $request['applications_count']; ?>)
                                        </a>
                                    </li>
                                    <?php else: ?>
                                    <li>
                                        <span class="dropdown-item-text text-muted">
                                            <i class="fas fa-info-circle me-2"></i>No Applications Yet
                                        </span>
                                    </li>
                                    <?php endif; ?>
                                    <li><hr class="dropdown-divider"></li>
                                    <?php if ($currentUser['role'] === ROLE_SITE_ADMIN && in_array($request['status'], ['approved_by_pm', 'approved_by_gm', 'approved_final'])): ?>
                                    <li>
                                        <a class="dropdown-item text-success" href="../employees/application_form.php?request_id=<?php echo $request['id']; ?>">
                                            <i class="fas fa-user-plus me-2"></i>Add Application
                                        </a>
                                    </li>
                                    <?php endif; ?>
                                    <li>
                                        <a class="dropdown-item" href="print_request.php?id=<?php echo $request['id']; ?>" target="_blank">
                                            <i class="fas fa-print me-2"></i>Print Request
                                        </a>
                                    </li>
                                    <?php if ($currentUser['role'] === ROLE_SITE_ADMIN && $request['status'] === 'pending'): ?>
                                    <li><hr class="dropdown-divider"></li>
                                    <li>
                                        <a class="dropdown-item" href="edit_request.php?id=<?php echo $request['id']; ?>">
                                            <i class="fas fa-edit me-2"></i>Edit Request
                                        </a>
                                    </li>
                                    <?php endif; ?>
                                    <?php 
                                    // Show approval options based on role and status
                                    $can_approve = false;
                                    if ($currentUser['role'] === ROLE_PROJECT_MANAGER && $request['status'] === 'pending') {
                                        $can_approve = true;
                                    } elseif ($currentUser['role'] === ROLE_GENERAL_MANAGER && $request['status'] === 'approved_by_pm') {
                                        $can_approve = true;
                                    } elseif ($currentUser['role'] === ROLE_HR_MANAGER && $request['status'] === 'approved_by_gm') {
                                        $can_approve = true;
                                    }
                                    
                                    if ($can_approve): ?>
                                    <li><hr class="dropdown-divider"></li>
                                    <li>
                                        <a class="dropdown-item text-success" href="../requests/approve_request.php?id=<?php echo $request['id']; ?>">
                                            <i class="fas fa-check me-2"></i>Review & Approve
                                        </a>
                                    </li>
                                    <?php endif; ?>
                                </ul>
                            </div>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
        <?php endif; ?>
    </div>
</div>

<!-- Request Statistics Summary -->
<?php if (!empty($requests)): ?>
<div class="row mt-4">
    <div class="col-12">
        <div class="card">
            <div class="card-header">
                <i class="fas fa-chart-bar me-2"></i>Request Summary
            </div>
            <div class="card-body">
                <div class="row text-center">
                    <?php
                    $summary = [
                        'total_requests' => count($requests),
                        'total_positions' => array_sum(array_column($requests, 'total_positions_needed')),
                        'total_applications' => array_sum(array_column($requests, 'applications_count')),
                        'total_hired' => array_sum(array_column($requests, 'hired_apps')),
                        'pending_requests' => count(array_filter($requests, fn($r) => $r['status'] === 'pending')),
                        'approved_requests' => count(array_filter($requests, fn($r) => in_array($r['status'], ['approved_by_pm', 'approved_by_gm', 'approved_final'])))
                    ];
                    ?>
                    <div class="col-md-2">
                        <h4 class="text-primary"><?php echo $summary['total_requests']; ?></h4>
                        <small class="text-muted">Total Requests</small>
                    </div>
                    <div class="col-md-2">
                        <h4 class="text-info"><?php echo $summary['total_positions']; ?></h4>
                        <small class="text-muted">Positions Needed</small>
                    </div>
                    <div class="col-md-2">
                        <h4 class="text-warning"><?php echo $summary['total_applications']; ?></h4>
                        <small class="text-muted">Applications</small>
                    </div>
                    <div class="col-md-2">
                        <h4 class="text-success"><?php echo $summary['total_hired']; ?></h4>
                        <small class="text-muted">Hired</small>
                    </div>
                    <div class="col-md-2">
                        <h4 class="text-danger"><?php echo $summary['pending_requests']; ?></h4>
                        <small class="text-muted">Pending</small>
                    </div>
                    <div class="col-md-2">
                        <h4 class="text-success"><?php echo $summary['approved_requests']; ?></h4>
                        <small class="text-muted">Approved</small>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>
<?php endif; ?>

<!-- Pagination -->
<?php if ($total_pages > 1): ?>
<nav class="mt-3">
    <ul class="pagination justify-content-center">
        <?php if ($page > 1): ?>
        <li class="page-item">
            <a class="page-link" href="?page=<?php echo $page-1; ?>&status=<?php echo $status_filter; ?>&search=<?php echo $search; ?>">
                <i class="fas fa-chevron-left"></i> Previous
            </a>
        </li>
        <?php endif; ?>
        
        <?php for ($i = max(1, $page-2); $i <= min($total_pages, $page+2); $i++): ?>
        <li class="page-item <?php echo $i === $page ? 'active' : ''; ?>">
            <a class="page-link" href="?page=<?php echo $i; ?>&status=<?php echo $status_filter; ?>&search=<?php echo $search; ?>">
                <?php echo $i; ?>
            </a>
        </li>
        <?php endfor; ?>
        
        <?php if ($page < $total_pages): ?>
        <li class="page-item">
            <a class="page-link" href="?page=<?php echo $page+1; ?>&status=<?php echo $status_filter; ?>&search=<?php echo $search; ?>">
                Next <i class="fas fa-chevron-right"></i>
            </a>
        </li>
        <?php endif; ?>
    </ul>
</nav>
<?php endif; ?>

<!-- View Request Modal -->
<div class="modal fade" id="viewRequestModal" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Request Details</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body" id="requestDetails">
                Loading...
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
            </div>
        </div>
    </div>
</div>

<script>
function viewRequest(requestId) {
    document.getElementById('requestDetails').innerHTML = 'Loading...';
    const modal = new bootstrap.Modal(document.getElementById('viewRequestModal'));
    modal.show();
    
    // Fetch request details
    fetch(`get_request_details.php?id=${requestId}`)
        .then(response => response.text())
        .then(html => {
            document.getElementById('requestDetails').innerHTML = html;
        })
        .catch(error => {
            document.getElementById('requestDetails').innerHTML = 'Error loading request details.';
        });
}

// Auto-refresh functionality for real-time updates
function refreshRequestData() {
    const currentUrl = new URL(window.location);
    fetch(currentUrl)
        .then(response => response.text())
        .then(html => {
            // Parse the new HTML and update only the table content
            const parser = new DOMParser();
            const newDoc = parser.parseFromString(html, 'text/html');
            const newTableBody = newDoc.querySelector('tbody');
            const currentTableBody = document.querySelector('tbody');
            
            if (newTableBody && currentTableBody) {
                currentTableBody.innerHTML = newTableBody.innerHTML;
            }
        })
        .catch(error => {
            console.log('Auto-refresh failed:', error);
        });
}

// Refresh every 2 minutes
setInterval(refreshRequestData, 120000);

// Filter change handlers
document.addEventListener('DOMContentLoaded', function() {
    // Auto-submit form when status filter changes
    const statusFilter = document.querySelector('select[name="status"]');
    if (statusFilter) {
        statusFilter.addEventListener('change', function() {
            this.form.submit();
        });
    }
    
    // Search on enter key
    const searchInput = document.querySelector('input[name="search"]');
    if (searchInput) {
        searchInput.addEventListener('keypress', function(e) {
            if (e.key === 'Enter') {
                this.form.submit();
            }
        });
    }
});
</script>

<style>
/* Enhanced styling for better visual separation */
.table th {
    background-color: #f8f9fa;
    border-bottom: 2px solid #dee2e6;
    font-weight: 600;
}

.badge {
    font-size: 0.75em;
    margin: 1px;
}

.btn-group-sm .dropdown-menu {
    font-size: 0.875rem;
}

.dropdown-item {
    padding: 0.5rem 1rem;
}

.dropdown-item i {
    width: 16px;
    text-align: center;
}

/* Responsive adjustments */
@media (max-width: 768px) {
    .table-responsive {
        font-size: 0.875rem;
    }
    
    .btn-group-sm .btn {
        padding: 0.25rem 0.5rem;
    }
}

/* Status badge improvements */
.status-badge {
    display: inline-block;
    padding: 0.375rem 0.75rem;
    font-size: 0.75rem;
    font-weight: 500;
    line-height: 1;
    text-align: center;
    white-space: nowrap;
    vertical-align: baseline;
    border-radius: 0.375rem;
}

/* Application count styling */
.badge.bg-info, .badge.bg-warning, .badge.bg-success, .badge.bg-danger {
    margin: 1px 2px;
    font-size: 0.65em;
}
</style>

<?php require_once '../includes/footer.php'; ?>